// Copyright (C) 1997-1999  Adrian Trapletti
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

//
// Feedforward neural network package
//
 

#include <iostream.h>
#include <math.h>
extern "C" {
#include "nrutil.h"
#include "nrext.h"
#include "nrlinal.h"
}
#include "utils.hh"
#include "str.hh"
#include "linal.hh"
#include "neuro.hh"


extern "C" {
#include "nr.c"  // brute force approach to hide the Numerical Recipes in the executable
}


// class ffnet

// C style for handling the numerical recipes procedures

/* Handler type used to establish the connection between the training method, 
   the objective function, and the derivative of the objective function */

struct ffnet_handler 
{
  long nin, nhid, nout, nw, nr;
  double* ynet; double* hid;
  double** x; double** y; 
  double** cov; double* e;
  int objective_function, regularizer, hid_t, out_t, shortc;
  double reg_control;
  intset* set;
  vec* w_old;
};


static ffnet_handler ffnet_h;  // the handler variable


void forward_propagate (double *x, double *h, double *y, double *w, long nin, long nhid, 
			long nout, int hid_t, int out_t, int shortc)  
  /* forward propagate x through a ffnet represented by the weights w giving 
     the hidden unit outputs h and the net outputs y.
     Input: x[1..nin], w[1..nw], nin, nhid, nout, hid_t, out_t, shortc
     where nw = (nin+1)*nhid+(nhid+1)*nout+nin*nout for GENERAL net, 
     nw = (nin+1)*nhid+(nhid+1)*nout for NONLINEAR net, nw = nout+nin*nout for LINEAR net.
     Output: h[1..nhid], y[1..nout] */
{
  long i, j, k, j_temp, k_temp;
  double sum, big;
  double *w1, *w2;

  w1 = &w[(nin+1)*nhid];  // set w1 s.t. w1[1] is the 1st weight from hidden to output layer
  w2 = &w[(nin+1)*nhid+(nhid+1)*nout];  // w2[1] is the 1st shortcut weight
  if (hid_t == SIG)
  {    
    for (j=0; j<nhid; j++)  // propagate from input to hidden layer
    {
      j_temp = j*(nin+1)+1;
      sum = -w[j_temp];
      for (i=1; i<=nin; i++)
	sum -= x[i]*w[i+j_temp];
      h[j+1] = 1.0/(1.0+exp(sum));
    }
  }
  else if (hid_t == TAN)
  {    
    for (j=0; j<nhid; j++)  // propagate from input to hidden layer
    {
      j_temp = j*(nin+1)+1;
      sum = w[j_temp];
      for (i=1; i<=nin; i++)
	sum += x[i]*w[i+j_temp];
      h[j+1] = tanh(sum);
    }
  }  
  else
    RTerror ("neuro.cc", "forward_propagate (double *x, double *h, double *y, double *w,"
	     "long nin, long nhid, long nout, int hid_t, int out_t, int shortc)", "1");
  for (k=0; k<nout; k++)  // propagate from hidden to output layer
  {
    k_temp = k*(nhid+1)+1;
    sum = w1[k_temp];
    for (j=1; j<=nhid; j++)
      sum += h[j]*w1[j+k_temp];
    y[k+1] = sum;
  }
  if (shortc)  // propagate along the shortcut connections
  {
    for (k=0; k<nout; k++)  
    {
      k_temp = k*nin;
      sum = y[k+1];
      for (j=1; j<=nin; j++)
	sum += x[j]*w2[j+k_temp];
      y[k+1] = sum;
    }
  }
  if (out_t == LIN) {}  // do nothing
  else if (out_t == SOFT)  // calculate normalized exponential
  { 
    big = y[1];  
    for (k=2; k<=nout; k++)  // big becomes the largest y[k], k=1,..,nout.
      if (big < y[k]) big = y[k];
    sum = 0.0;
    for (k=1; k<=nout; k++)  // renormalize with big to avoid overflows 
    { 
      y[k] = exp(y[k]-big);
      sum += y[k];
    }
    for (k=1; k<=nout; k++) y[k] /= sum;
  }
  else
    RTerror ("neuro.cc", "forward_propagate (double *x, double *h, double *y, double *w,"
	     "long nin, long nhid, long nout, int hid_t, int out_t, int shortc)", "2");
}

double ffnet_err_func (double w[])
{
  const double big = 1.79e+308;
  
  int pd;
  long n, k, l;
  double obj = 0.0; double reg = 0.0;
  
  if (ffnet_h.objective_function == SSE)  // evaluate sum of squares
  {
    for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
    {
      forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			 ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
      for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	obj += DSQR(ffnet_h.ynet[k]-ffnet_h.y[n][k]);
    }
  }
  else if (ffnet_h.objective_function == MAD)  // evaluate mean absolute deviation
  {
    for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
    {
      forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			 ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc); 
      for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	obj += fabs(ffnet_h.ynet[k]-ffnet_h.y[n][k]);
    }
  }
  else if (ffnet_h.objective_function == GSSE)  // evaluate generalized sum of squares
  {
    forward_propagate (&ffnet_h.x[1][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
		       ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
    for (k=1; k<=ffnet_h.nout; k++)  // compute error covariance matrix for first observation
      for (l=k; l<=ffnet_h.nout; l++) 
	ffnet_h.cov[k][l] = (ffnet_h.ynet[k]-ffnet_h.y[1][k])*(ffnet_h.ynet[l]-ffnet_h.y[1][l]);
    for (n=2; n<=ffnet_h.nr; n++)  // loop over observations
    {      
      forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			 ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc); 
      for (k=1; k<=ffnet_h.nout; k++)  // compute error covariance matrix
	for (l=k; l<=ffnet_h.nout; l++) 
	  ffnet_h.cov[k][l] += (ffnet_h.ynet[k]-ffnet_h.y[n][k])*(ffnet_h.ynet[l]-ffnet_h.y[n][l]);
    }
    for (k=1; k<=ffnet_h.nout; k++)  // standardize error covariance matrix
      for (l=k; l<=ffnet_h.nout; l++) 
	ffnet_h.cov[k][l] /= (double)ffnet_h.nr; 
    pd = det_chol (ffnet_h.cov, ffnet_h.nout, &obj);  // compute objective function
    if (pd) 
      obj = (double)ffnet_h.nr*log(obj)/2.0;  
    else
      obj = log(big);  // covariance not positive definite due to large elements causing rounding errors
  }         
  else if (ffnet_h.objective_function == ENTROPY)  // evaluate cross-entropy
  {
    for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
    {
      forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			 ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc); 
      for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	obj -= ffnet_h.y[n][k]*log(ffnet_h.ynet[k]);
    }
  }
  else
    RTerror ("neuro.cc", "double ffnet_err_func (double w[])", "1");
  if (ffnet_h.regularizer == NOR)  // no regularizer term
    reg = 0.0;
  else if (ffnet_h.regularizer == WDR)  // evaluate weight decay regularizer term
  {
    for (l=1; l<=ffnet_h.nw; l++)
      reg += DSQR(w[l]);
    reg *= ffnet_h.reg_control;
  }
  else if (ffnet_h.regularizer == LASSO)  // evaluate lasso regularizer term
  {
    for (l=1; l<=ffnet_h.nw; l++)
      reg += fabs(w[l]);
    reg *= ffnet_h.reg_control;
  }
  else
    RTerror ("neuro.cc", "double ffnet_err_func (double w[])", "2");
  return (obj+reg);
}

/* To check the analytical derivatives, computation of the 
   numerical derivatives is provided */

#ifdef NUM_CHECK  
static long kcom;  // communication variables for numdf
static double* wcom;

double num_deriv (double x)  // used by numdf
{
  double temp, f;

  temp = wcom[kcom];
  wcom[kcom] = x;
  f = ffnet_err_func (wcom);
  wcom[kcom] = temp;
  return f;
}
#endif

void ffnet_derr_func (double w[], double dw[])
{
  int pd;
  long n, k, j, l, i, k_temp, j_temp;
  double temp;
  double *dw1, *w1, *dw2;

  dw1 = &dw[(ffnet_h.nin+1)*ffnet_h.nhid];  // w1[1] (dw1[1]) is the (derivative of the)
  w1 = &w[(ffnet_h.nin+1)*ffnet_h.nhid];    // 1st weight from hidden to output layer
  dw2 = &dw[(ffnet_h.nin+1)*ffnet_h.nhid+(ffnet_h.nhid+1)*ffnet_h.nout];  // dw2[1] 1st shortcut weight
  if (ffnet_h.regularizer == NOR)  // no regularizer term
  {
    for (l=1; l<=ffnet_h.nw; l++)
      dw[l] = 0.0;
  }
  else if (ffnet_h.regularizer == WDR)  // evaluate weight decay regularizer term derivatives
  {
    for (l=1; l<=ffnet_h.nw; l++)
      dw[l] = 2.0*ffnet_h.reg_control*w[l];
  }
  else if (ffnet_h.regularizer == LASSO)  // evaluate lasso regularizer term derivatives
  {
    for (l=1; l<=ffnet_h.nw; l++)
      dw[l] = ffnet_h.reg_control*SIGN(1.0,w[l]);
  }
  else
    RTerror ("neuro.cc", "void ffnet_derr_func (double w[], double dw[])", "1");
  /*
    Backpropagation algorithm, c.f. "NNPR, 140-147". Due to efficiency reasons each configuration,
    i.e., activation function, error function, has its own block.
  */
  if ((ffnet_h.objective_function == SSE) && (ffnet_h.out_t == LIN))  
    // sum of squares, linear outputs 
  {
    if (ffnet_h.hid_t == SIG)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc);  
	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = 2.0*(ffnet_h.ynet[k]-ffnet_h.y[n][k]);  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];                           
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts 
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= temp*(1-temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else if (ffnet_h.hid_t == TAN)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc); 
	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = 2.0*(ffnet_h.ynet[k]-ffnet_h.y[n][k]);  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];                          
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= 1-DSQR(temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else
      RTerror ("neuro.cc", "void ffnet_derr_func (double w[], double dw[])", "2");
  }
  else if (((ffnet_h.objective_function == GSSE) && (ffnet_h.out_t == LIN))) 
    /* generalized sum of squares, linear outputs 
       for the computation of the derivative of log(det), see
       'Davidson and MacKinnon (1993): Estimation and Inference in 
       Econometrics, Oxford University Press, NY, p. 317.' */
  {
    forward_propagate (&ffnet_h.x[1][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
		       ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
    for (k=1; k<=ffnet_h.nout; k++)  // compute error covariance matrix for first observation
      for (l=k; l<=ffnet_h.nout; l++) 
	ffnet_h.cov[k][l] = (ffnet_h.ynet[k]-ffnet_h.y[1][k])*(ffnet_h.ynet[l]-ffnet_h.y[1][l]);
    for (n=2; n<=ffnet_h.nr; n++)  // loop over observations
    {      
      forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			 ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
      for (k=1; k<=ffnet_h.nout; k++)  // compute error covariance matrix
	for (l=k; l<=ffnet_h.nout; l++) 
	  ffnet_h.cov[k][l] += (ffnet_h.ynet[k]-ffnet_h.y[n][k])*(ffnet_h.ynet[l]-ffnet_h.y[n][l]);
    }
    for (k=1; k<=ffnet_h.nout; k++)  // standardize error covariance matrix
      for (l=k; l<=ffnet_h.nout; l++) 
	ffnet_h.cov[k][l] /= (double)ffnet_h.nr;
    pd = inv_chol (ffnet_h.cov, ffnet_h.nout);  // compute inverse of covariance matrix
    if (!pd)  // covariance not positive definite due to large elements causing rounding errors
    {
      for (k=1; k<=ffnet_h.nout; k++)  // set error covariance matrix to identity
      {
	for (l=1; l<=ffnet_h.nout; l++) 
	  ffnet_h.cov[k][l] = 0.0;
	ffnet_h.cov[k][k] = 1.0;
      }
    }
    if (ffnet_h.hid_t == SIG)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			   ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
	for (k=1; k<=ffnet_h.nout; k++) 
	  ffnet_h.e[k] = ffnet_h.ynet[k]-ffnet_h.y[n][k];  // compute negative error
	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = 0.0;  // compute delta[k] and save it
	  for (i=1; i<=ffnet_h.nout; i++) 
	    ffnet_h.ynet[k] += ffnet_h.e[i]*ffnet_h.cov[i][k]; 
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts 
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= temp*(1-temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else if (ffnet_h.hid_t == TAN)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, ffnet_h.nhid, 
			   ffnet_h.nout, ffnet_h.hid_t, ffnet_h.out_t, ffnet_h.shortc);  
	for (k=1; k<=ffnet_h.nout; k++) 
	  ffnet_h.e[k] = ffnet_h.ynet[k]-ffnet_h.y[n][k];  // compute negative error
	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = 0.0;  // compute delta[k] and save it
	  for (i=1; i<=ffnet_h.nout; i++) 
	    ffnet_h.ynet[k] += ffnet_h.e[i]*ffnet_h.cov[i][k]; 
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= 1-DSQR(temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
  }
  else if ((ffnet_h.objective_function == ENTROPY) && (ffnet_h.out_t == SOFT))  
    // cross-entropy, softmax outputs, c.f. "NNPR, 237-240"
  {
    if (ffnet_h.hid_t == SIG)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc); 
	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = ffnet_h.ynet[k]-ffnet_h.y[n][k];  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= temp*(1-temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else if (ffnet_h.hid_t == TAN)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc); 
      	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = ffnet_h.ynet[k]-ffnet_h.y[n][k];  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= 1-DSQR(temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else
      RTerror ("neuro.cc", "void ffnet_derr_func (double w[], double dw[])", "3");
  }
  else if (((ffnet_h.objective_function == MAD) && (ffnet_h.out_t == LIN)))  
    // mean absolute deviation, linear outputs 
  {
    if (ffnet_h.hid_t == SIG)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc); 
      	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = SIGN(1.0,ffnet_h.ynet[k]-ffnet_h.y[n][k]);  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= temp*(1-temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else if (ffnet_h.hid_t == TAN)
    {
      for (n=1; n<=ffnet_h.nr; n++)  // loop over observations
      {
	forward_propagate (&ffnet_h.x[n][0], ffnet_h.hid, ffnet_h.ynet, w, ffnet_h.nin, 
			   ffnet_h.nhid, ffnet_h.nout, ffnet_h.hid_t, 
			   ffnet_h.out_t, ffnet_h.shortc); 
  	for (k=1; k<=ffnet_h.nout; k++)  // loop over output units
	{
	  k_temp = (k-1)*(ffnet_h.nhid+1)+1;
	  ffnet_h.ynet[k] = SIGN(1.0,ffnet_h.ynet[k]-ffnet_h.y[n][k]);  // compute delta[k] and save it
	  dw1[k_temp] += ffnet_h.ynet[k];
	  for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	    dw1[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.hid[j];
	  if (ffnet_h.shortc)  // derivatives of shortcuts
	  {
	    k_temp = (k-1)*ffnet_h.nin;
	    for (j=1; j<=ffnet_h.nin; j++)  // loop over input units
	      dw2[j+k_temp] += ffnet_h.ynet[k]*ffnet_h.x[n][j];
	  }
	}
	for (j=1; j<=ffnet_h.nhid; j++)  // loop over hidden units
	{
	  j_temp = (j-1)*(ffnet_h.nin+1)+1;
	  temp = ffnet_h.hid[j];
	  ffnet_h.hid[j] = 0.0;
	  for (k=1; k<=ffnet_h.nout; k++)  // backpropagate with loop over output units 
	    ffnet_h.hid[j] += ffnet_h.ynet[k]*w1[(k-1)*(ffnet_h.nhid+1)+1+j];
	  ffnet_h.hid[j] *= 1-DSQR(temp);  // compute delta[j]
	  dw[j_temp] += ffnet_h.hid[j];
	  for (i=1; i<=ffnet_h.nin; i++)  // loop over input units
	    dw[i+j_temp] += ffnet_h.hid[j]*ffnet_h.x[n][i];
	}
      }
    }
    else 
      RTerror ("neuro.cc", "void ffnet_derr_func (double w[], double dw[])", "4");      
  }  
  else
    RTerror ("neuro.cc", "void ffnet_derr_func (double w[], double dw[])", "5");   
#ifdef NUM_CHECK
  const double tol = 1.0e-5;  /* tolerance for the relative error 
				 between numerical and analytical derivative */

  double df;

  wcom = w;
  for (k=1; k<=ffnet_h.nw; k++)
  {
    kcom = k;
    df = numdf (num_deriv, w[k]);  // compute numerical derivative
    if (fabs(dw[k]-df)/df > tol)
    {
      cout << "...WARNING..." << endl;
      cout << "...Analytical derivative " << k << " is " << dw[k] << "..." << endl;
      cout << "...Numerical derivative " << k << " is " << df << "..." << endl;
      cout << "...Relative error between numerical and analytical derivative is ";
      cout << fabs(dw[k]-df)/df << "..." << endl;
      cout << "...Absolute error between numerical and analytical derivative is ";
      cout << fabs(dw[k]-df) << "..." << endl;
    }
  }
#endif 
  if (!(*ffnet_h.set).empty())  // set derivatives of fixed weights to zero
    for (l=1; l<=ffnet_h.nw; l++)
      if ((*ffnet_h.set).member(l)) dw[l] = 0.0;
}

double ffnet_ann_err_func (double w[])  // for fixed weights with stochastic optimization
{
  long i, j;

  j = 1; 
  for (i=1; i<=(*ffnet_h.w_old).rows(); i++)  // construct the whole weight vector 
    if (!(*ffnet_h.set).member(i))
    {
      (*ffnet_h.w_old)(i) = w[j];
      j++;
    }
  return ffnet_err_func ((*ffnet_h.w_old).elem());  // error function with the whole weight vector 
}

// C/C++ mixed style

double ffnet::train (const mat& x, const mat& y, const intset& fixed_weights, 
		     int objective_function, int regularizer, 
		     double reg_control, const estim_proc& estim)
  // fits this to a given data set of inputs x and targets y
{
  long i, j, k, nr_temp;
  double** x_temp; double** y_temp;
  double errv; 
  int iter;
  ivec I;
  long w_new_n = w.rows()-fixed_weights.size();
  vec w_new(w_new_n); 
  long n = x.rows();
  intset& fixed_weights_nc = const_cast <intset&> (fixed_weights);  // non-const reference

#ifdef CHECK
  if ((x.columns() != nin) || (y.columns() != nout) || (x.rows() != y.rows()))
    RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
	     "const intset& fixed_weights, int objective_function, int regularizer," 
	     "double reg_control, const estim_proc& estim)", "1");
#endif  
  ffnet_h.ynet = dvector (1, nout); ffnet_h.hid = dvector (1, nhid);  // set general handler
  ffnet_h.e = dvector (1, nout); ffnet_h.cov = dmatrix (1, nout, 1, nout); 
  ffnet_h.nin = nin; ffnet_h.nhid = nhid; ffnet_h.nout = nout; ffnet_h.nw = w.rows(); 
  ffnet_h.objective_function = objective_function; ffnet_h.regularizer = regularizer; 
  ffnet_h.hid_t = hid_t; ffnet_h.out_t = out_t; ffnet_h.shortc = shortc; 
  ffnet_h.reg_control = reg_control; 
  ffnet_h.set = &fixed_weights_nc; ffnet_h.w_old = &w;
  if (estim.epoch > 0)  // epoch based estimation
  {    
    ffnet_h.x = dmatrix (1, estim.epoch, 1, nin);  // set epoch handler
    ffnet_h.y = dmatrix (1, estim.epoch, 1, nout); 
    ffnet_h.nr = estim.epoch; 
    if (estim.trace)
      cout << "-- overall objective function values --" << endl;
    for (i=1; i<=estim.itmax; i++)
    {
      if (estim.trace)
      {
	x_temp = ffnet_h.x; y_temp = ffnet_h.y;  // set handler values for whole data set
	nr_temp = ffnet_h.nr;  
	ffnet_h.x = x.elem(); ffnet_h.y = y.elem();
	ffnet_h.nr = x.rows(); 	
	if (i == 1)
	{
	  cout << endl << "-- initial   value " << ffnet_err_func (w.elem());
	  cout << " --" << endl << endl;	
	}
	else
	{
	  cout << endl << "-- iter ";
	  cout.width(4); 
	  cout << i-1 << " value " << ffnet_err_func (w.elem()) << " --" << endl << endl;	
	}
	ffnet_h.x = x_temp; ffnet_h.y = y_temp;  // reset handler values for epoch data set
	ffnet_h.nr = nr_temp; 
      }
      I = rperm (n, estim.epoch);  // sampling without replacement
      for (j=1; j<=estim.epoch; j++)  // construct epoch set
      {
	for (k=1; k<=nin; k++)    
	  ffnet_h.x[j][k] = x(I(j),k);
	for (k=1; k<=nout; k++)    
	  ffnet_h.y[j][k] = y(I(j),k);
      }	       
      if (estim.type == GRDDSC)  // train on the given epoch set
	grddsc (w.elem(), w.rows(), estim.eta, estim.alpha, estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itepoch);
      else if (estim.type == STPDSC)
	stpdsc (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itepoch);
      else if (estim.type == FRPRMN)
	frprmn (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itepoch);
      else if (estim.type == DFPMIN)
	dfpmin (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itepoch);
      else 
	RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
		 "const intset& fixed_weights, int objective_function, int regularizer," 
		 "double reg_control, const estim_proc& estim)", "2");
    }
    if (estim.trace)
    {
      x_temp = ffnet_h.x; y_temp = ffnet_h.y;  // set handler values for whole data set
      nr_temp = ffnet_h.nr; 
      ffnet_h.x = x.elem(); ffnet_h.y = y.elem();
      ffnet_h.nr = x.rows(); 
      cout << endl << "-- final     value " << ffnet_err_func (w.elem());
      cout << " --" << endl << endl;
      cout << "Stopped after " << estim.itmax << " iterations" << endl;
      ffnet_h.x = x_temp; ffnet_h.y = y_temp;  // reset handler values for epoch data set
      ffnet_h.nr = nr_temp; 
    }
  }
  else  // batch mode estimation
  {
    ffnet_h.x = x.elem(); ffnet_h.y = y.elem();  // set batch handler
    ffnet_h.nr = x.rows(); 
    if ((GRDDSC <= estim.type) && (estim.type <= DFPMIN))  // deterministic estimation
    {
      if (estim.type == GRDDSC)
	grddsc (w.elem(), w.rows(), estim.eta, estim.alpha, estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itmax);
      else if (estim.type == STPDSC)
	stpdsc (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itmax);
      else if (estim.type == FRPRMN)
	frprmn (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itmax);
      else if (estim.type == DFPMIN)
	dfpmin (w.elem(), w.rows(), estim.tol, &iter, &errv, 
		ffnet_err_func, ffnet_derr_func, estim.trace, estim.itmax);
      else
	RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
		 "const intset& fixed_weights, int objective_function, int regularizer," 
		 "double reg_control, const estim_proc& estim)", "3"); 
    }
    else if ((SANN <= estim.type) && (estim.type <= NRSANN))  // stochastic estimation
    {
      if (!(*ffnet_h.set).empty())  // some weights are fixed
      {
	if (w_new_n > 0)  // not all are fixed
	{
	  j = 1; 
	  for (i=1; i<=w.rows(); i++)  // w_new contains the weights for estimation
	    if (!(*ffnet_h.set).member(i))
	    {
	      w_new(j) = w(i);
	      j++;
	    }
	  if (estim.type == SANN)
	    sann (w_new.elem(), &errv, w_new_n, estim.itmax, estim.kmax, 
		  estim.ti, ffnet_ann_err_func, estim.trace);
	  else if (estim.type == NRSANN)
	    nrsann (w_new.elem(), &errv, w_new_n, estim.itmax, estim.kmax, 
		    estim.ti, ffnet_ann_err_func, estim.trace);
	  else 
	    RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
		     "const intset& fixed_weights, int objective_function, int regularizer," 
		     "double reg_control, const estim_proc& estim)", "4"); 
	  j = 1; 
	  for (i=1; i<=w.rows(); i++)  // restore new weights into weights
	    if (!(*ffnet_h.set).member(i))
	    {
	      w(i) = w_new(j);
	      j++;
	    }
	}
      }
      else  // estimate all weights
      {
	if (estim.type == SANN)
	  sann (w.elem(), &errv, w.rows(), estim.itmax, estim.kmax, 
		 estim.ti, ffnet_err_func, estim.trace);
	else if (estim.type == NRSANN)
	  nrsann (w.elem(), &errv, w.rows(), estim.itmax, estim.kmax, 
		   estim.ti, ffnet_err_func, estim.trace);
	else
	  RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
		   "const intset& fixed_weights, int objective_function, int regularizer," 
		   "double reg_control, const estim_proc& estim)", "5"); 
      }
    }
    else
      RTerror ("neuro.cc", "void ffnet::train (const mat& x, const mat& y,"
	       "const intset& fixed_weights, int objective_function, int regularizer," 
	       "double reg_control, const estim_proc& estim)", "6"); 
  }
  free_dvector (ffnet_h.hid, 1, nhid); free_dvector (ffnet_h.ynet, 1, nout); 
  free_dmatrix (ffnet_h.cov, 1, nout, 1, nout); free_dvector (ffnet_h.e, 1, nout);
  return errv;
}

/* Handler function used to estimate the 2nd derivatives 
   by numerical approximation */

static long icom, jcom;  // communication variables for numdf
static double *wscom, *dwcom;

double num_sec_deriv (double arg)  // used by numdf
{
  double temp;

  temp = wscom[jcom];
  wscom[jcom] = arg;
  ffnet_derr_func (wscom, dwcom);
  wscom[jcom] = temp;
  return dwcom[icom];
}

mat ffnet::hess (const mat& x, const mat& y, const intset& fixed_weights, 
		 int objective_function, int regularizer, double reg_control)
  /* estimate hessian matrix of given objective function at the current weight vector
     by numerical approximation, c.f. "NRC, 186, 187", "Dennis and Schnabel (1989): 
     A View of Unconstrained Optimization. In Nemhauser, Rinnooy Kan, Todd (eds.), 
     Optimization, Vol. 1, North-Holland, Amsterdam, pp. 17-22." */
{
  long i, j;
  double df;
  mat h(w.rows(),w.rows());
  intset& fixed_weights_nc = const_cast <intset&> (fixed_weights);  // non-const reference
  
#ifdef CHECK
  if ((x.columns() != nin) || (y.columns() != nout) || (x.rows() != y.rows()))
    RTerror ("neuro.cc", "mat ffnet::hess (const mat& x, const mat& y,"
	     "const intset& fixed_weights, int objective_function, int regularizer,"
	     "double reg_control)");
#endif  
  ffnet_h.ynet = dvector (1, nout); ffnet_h.hid = dvector (1, nhid);  // set handler values
  ffnet_h.e = dvector (1, nout); ffnet_h.cov = dmatrix (1, nout, 1, nout);
  ffnet_h.nin = nin; ffnet_h.nhid = nhid; ffnet_h.nout = nout; ffnet_h.nw = w.rows(); 
  ffnet_h.objective_function = objective_function; ffnet_h.regularizer = regularizer; 
  ffnet_h.hid_t = hid_t; ffnet_h.out_t = out_t; ffnet_h.shortc = shortc; 
  ffnet_h.reg_control = reg_control;
  ffnet_h.set = &fixed_weights_nc; 
  ffnet_h.x = x.elem(); ffnet_h.y = y.elem(); 
  ffnet_h.nr = x.rows();
  wscom = w.elem(); dwcom = dvector (1, ffnet_h.nw);
  for (i=1; i<=w.rows(); i++)  // compute upper triangle and diagonal of hessian
  {
    icom = i;
    for (j=1; j<=w.rows(); j++)
    {
      jcom = j;
      df = numdf (num_sec_deriv, w(j));
      h(i,j) = df;
    }
  }
  for (i=1; i<=w.rows(); i++)  // let h = (h+h')/2 be the Frobenius norm projection of
  {
    for (j=i+1; j<=w.rows(); j++)  // h into the subspace of all symmetric matrices
    {
      h(i,j) = (h(i,j)+h(j,i))/2.0;
      h(j,i) = h(i,j);
    }
  }
  if (!fixed_weights.empty())  // set 2nd derivatives of fixed weights to zero
    for (i=1; i<=w.rows(); i++)
      for (j=1; j<=w.rows(); j++)
	if ((fixed_weights.member(i)) || (fixed_weights.member(j))) h(i,j) = 0.0;
  free_dvector (ffnet_h.hid, 1, nhid); free_dvector (ffnet_h.ynet, 1, nout); 
  free_dmatrix (ffnet_h.cov, 1, nout, 1, nout); free_dvector (ffnet_h.e, 1, nout);
  free_dvector (dwcom, 1, ffnet_h.nw);
  return h;  // inefficient, 2 copies of h!
}

// C++ style

istream& operator>> (istream& s, ffnet& net)  // deallocate and read  
{
  str buf1, buf2;

  if (!(s >> buf1 >> buf2 >> net.nin) || (buf1 != "#") || (buf2 != "nin:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "1");
  if (!(s >> buf1 >> buf2 >> net.nhid) || (buf1 != "#") || (buf2 != "nhid:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "2");
  if (!(s >> buf1 >> buf2 >> net.nout) || (buf1 != "#") || (buf2 != "nout:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "3");
  if (!(s >> buf1 >> buf2 >> net.hid_t) || (buf1 != "#") || (buf2 != "hid_t:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "4");
  if (!(s >> buf1 >> buf2 >> net.out_t) || (buf1 != "#") || (buf2 != "out_t:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "5");
  if (!(s >> buf1 >> buf2 >> net.shortc) || (buf1 != "#") || (buf2 != "shortc:"))
    RTerror ("neuro.cc", "istream& operator>> (istream& s, ffnet& net)", "6");
  s >> net.w;
  return s;
}

ostream& operator<< (ostream& s, const ffnet& net)             
{
  s << "# nin: " << net.nin << endl << endl;
  s << "# nhid: " << net.nhid << endl << endl;
  s << "# nout: " << net.nout << endl << endl;
  s << "# hid_t: " << net.hid_t << endl << endl;
  s << "# out_t: " << net.out_t << endl << endl;
  s << net.w << endl;
  return s;
}

ffnet::ffnet (long input, long hidden, long output, int hid_type, int out_type, 
	      int shortcut) 
  : w (shortcut
       ? (input+1)*hidden+(hidden+1)*output+input*output
       : (input+1)*hidden+(hidden+1)*output)
{
#ifdef CHECK
  if ((input <= 0) || (hidden < 0) || (output <= 0) || ((output == 1) && (out_type == SOFT)))
    RTerror ("neuro.cc", "ffnet::ffnet (long input, long hidden, long output, int hid_type,"
	     "int out_type, int shortcut)");  
#endif
  nin = input; nhid = hidden; nout = output; hid_t = hid_type; 
  out_t = out_type; shortc = shortcut;
  w.rand (GAUSS, 0.0, 1.0);  // default initialization is randomly standard Normal
}

ffnet& ffnet::set_weights (const vec& weight)  // set weight vector
{
#ifdef CHECK
  if (w.rows() != weight.rows())
    RTerror ("neuro.cc", "ffnet& ffnet::set_weights (const vec& weight)");
#endif 
  w = weight;
  return *this;
}

ffnet& ffnet::operator= (const ffnet& net)  // assignment 
{
  if (this != &net)  // beware of net = net
  {
    nin = net.nin; nhid = net.nhid; nout = net.nout; 
    hid_t = net.hid_t; out_t = net.out_t; shortc = net.shortc;
    w = net.w;
  }
  return *this;
}

ffnet::ffnet (const ffnet& net) : w(net.w)  // copy constructor
{
  nin = net.nin; nhid = net.nhid; nout = net.nout; 
  hid_t = net.hid_t; out_t = net.out_t; shortc = net.shortc;
}

vec ffnet::predict (const vec& x) const  // prediction from given input
{
  vec h(nhid), y(nout);
  
#ifdef CHECK
  if (x.rows() != nin)
    RTerror ("neuro.cc", "vec& ffnet::predict (const vec& x) const");
#endif  
  forward_propagate (x.elem(), h.elem(), y.elem(), w.elem(), 
		     nin, nhid, nout, hid_t, out_t, shortc);
  return y;  // inefficient, for an assignment z = ffnet.predict(), y is copied twice!
}

mat ffnet::predict (const mat& x) const
  /* prediction from given input array of observations;
     each row equals one observation */
{
  long i;
  vec h(nhid);
  mat y(x.rows(),nout); 

#ifdef CHECK
  if (x.columns() != nin)
    RTerror ("neuro.cc", "mat ffnet::predict (const mat& x) const");
#endif  
  for (i=1; i<=x.rows(); i++)  // loop over observations
    forward_propagate (&(x.elem())[i][0], h.elem(), &(y.elem())[i][0], w.elem(), 
		       nin, nhid, nout, hid_t, out_t, shortc);     
  return y;  // inefficient, for an assignment z = ffnet.predict(), y is copied twice!
}

double ffnet::train (const mat& x, const mat& y, int objective_function, int regularizer, 
		     double reg_control, const estim_proc& estim)
  // as above, but without fixing any weights 
{
  intset fixed_weights;
  
  return (*this).train (x, y, fixed_weights, objective_function, regularizer, reg_control, estim);
}

static str key = "AT26081967";

mat ffnet::hess (const mat& x, const mat& y, int objective_function, 
		 int regularizer, double reg_control)
  // compute hessian matrix, but without fixing any weights
{
  intset fixed_weights;
  
  return (*this).hess (x, y, fixed_weights, objective_function, regularizer, reg_control);  
}


// class estim_proc: container for the information about an estimation procedure 

void estim_proc::init (int ty, int tr, int itm, double to)  
  // Initializer for FRPRMN and DFPMIN in batch mode
{
  type = ty; trace = tr; itmax = itm; tol = to;
  kmax = 0; epoch = 0; itepoch = 0; 
  eta = 0.0; alpha = 0.0; ti = 0.0;
}

void estim_proc::init (int ty, int tr, int itm, double to, int ep, int itep) 
  // Initializer for FRPRMN and DFPMIN in epoch based mode
{
  type = ty; trace = tr; itmax = itm; tol = to;
  kmax = 0; epoch = ep; itepoch = itep;
  eta = 0.0; alpha = 0.0; ti = 0.0;
}

void estim_proc::init (int ty, int tr, int itm, double to, double et, double al)
  // Initializer for GRDDSC in batch mode
{
  type = ty; trace = tr; itmax = itm; tol = to; 
  kmax = 0; epoch = 0; itepoch = 0;
  eta = et; alpha = al; ti = 0.0;
}

void estim_proc::init (int ty, int tr, int itm, double to, double et, double al, int ep, int itep)
  // Initializer for GRDDSC in epoch based mode
{
  type = ty; trace = tr; itmax = itm; tol = to;
  kmax = 0; epoch = ep; itepoch = itep;
  eta = et; alpha = al; ti = 0.0;
}

void estim_proc::init (int ty, int tr, int itm, int km, double t)
  // Initializer for SANN and NRSANN
{
  type = ty; trace = tr; itmax = itm; tol = 0.0;
  kmax = km; epoch = 0; itepoch = 0; 
  eta = 0.0; alpha = 0.0; ti = t;
}
