/*
 * Portable Memory Management Interface for ANSI C,
 * with Extension for Win32 platforms supporting
 * multiple heaps.
 *
 * Copyright (C) 2000 Federico Spinazzi <fspinazzi@hotmail.com>
 *
 * Free Software License:
 *
 * All rights are reserved by the author, with the following exceptions:
 * Permission is granted to freely reproduce and distribute this software,
 * possibly in exchange for a fee, provided that this copyright notice appears
 * intact.
 *
 * Permission is also granted to adapt this software to produce
 * derivative works, as long as the modified versions carry this copyright
 * notice and additional notices stating that the work has been modified.
 *
 * This source code may be translated into executable form and incorporated
 * into proprietary software; there is no requirement for such software to
 * contain a copyright notice related to this source.
 *
 */

#define IMEM_IMPLEMENTATION
#include "imem.h"

#ifdef WITH_RCSID
static char *rcsid = "$Id: imem.c 1.19 2000/12/13 23:53:27 federico Exp $";
#endif

#ifdef MSS
#include <mss.h>
#endif


/*
 * TODO
 * - rename from a_ to imem_
 * - log_info should describe each memory space
 * - document (expecially random failure)
 * - fprintf formats (expecially for unsigned and size_t ...)
 * - verify correct use of nodepool: there was a point where the presence
 *   of the pool wasn't checked !
 *   - A nodepool is used by the hash to allocate its nodes
 *   - A nodepool is (optionally) used by amb_list structure to allocate its
 *     nodes
 * - error messages associated with numerical error codes
 * - return codes for errors.
 * - support for debugging: list allocated block by memory space,
 * - ??? there are incompatibilities between logging and keeping track
 *   of memory blocks via KEEP_AMB ???
 *
 * OPEN
 * - potential problems with alignement in nodepool ?
 *
 * MAJOR CHANGES OTHER:
 * - move the amb stuff away as one can decide to use
 *   binary trees instead of hash tables
 * - keep an histogram of allocated block sizes or write a log to a file,
 *   or write a program to extract this information ...
 * - inlining when allowed by the compiler ?
 * - instead of checking if the node_pool field is NULL,
 *   unify calls for allocations of nodes for amb_list indipendent
 *   on using/not using node pools (i.e. encapsulate amb blocks)
 * - multilingual
 * - serialize with multiple threads with the same interface (NO: its
 *   an user task)
 * - keep a name of the callback functions (using a macro to initialize
 *   the whole thing ...)
 *
 * $Log: imem.c $
 * Revision 1.19  2000/12/13 23:53:27  federico
 * Changhed MAKE_NAME macros to IMEM_MAKE_NAME.
 * Extended imem_log_info to print the name of the functions and the
 * list of initialization values
 * Added name field to imem_t and imem_funcs_t; added IMEM_INIT_FUNCS macro.
 * Exported imem_log_info
 * Removed some assertions as they contained non conditional code
 * (that code will not be executed with NDEBUG defined)
 * Added an imem_init_funcs function to support IMEM_INIT_FUNCS macro, of
 * course.
 *
 * Revision 1.18  2000/12/13 00:39:29  federico
 * little modification to a_log_start.
 * Added a flag (unused for the moment) to log random generated
 * numbers to the log file.
 *
 * Revision 1.17  2000/07/01 01:54:53  federico
 * fixed bugs in amb_list_free_mem_space: forgotten an IMEM_DEBUG_FREE call.
 *
 * Revision 1.16  2000/07/01 00:44:52  federico
 * Debug message in random_failed
 * IMEM_DEBUG_* macros to use MSS on internal data structures.
 * Fixed wrong calls to node_pool_* functions.
 * amb_list_deinit only sees the list of allocated functions.
 * SPACE substitued by A_LINE.
 * Added a message in imem_log_end for when there aren't info.
 *
 * Revision 1.15  2000/06/30 00:19:59  federico
 * amb_list_deinit no longer frees the memory pointed by allocated memory
 * block: it will leak memory: thetask has been given to amb_list_free_mem_space
 * OPEN bugs: WITH_LOG && ! KEEP_AMB.
 * added node_pool* calls where incorrectly missing.
 * fixed other calls to node_pool* functions.
 *
 * Revision 1.14  2000/06/29 01:05:11  federico
 * Changed random generator.
 * removed any attempt to use MSS.
 * Added logging to amb_list_deinit.
 *
 * Revision 1.13  2000/06/26 23:09:28  federico
 * fixed random failure handling.
 * added imem_random_test.
 *
 * Revision 1.12  2000/05/22 23:49:09  federico
 * Updated to imem_*_t types.
 * Function and file names are not allocated dinamically.
 *
 * Revision 1.11  2000/05/21 12:18:21  federico
 * Added node pool for mb
 *
 * Revision 1.10  2000/05/20 10:01:53  federico
 * Renamed two amb fields.
 * Added two amb fields.
 * The renamed fields are the node pools for tha hash table.
 * The added fields are the node pools for the memory blocks.
 *
 * Revision 1.9  2000/05/14 16:37:23  federico
 * added support for node pools.
 * early tests show a speed up of about 40%.
 * to be debugged for memory leaks.
 *
 * Revision 1.8  2000/05/08 21:12:32  federico
 * code cleanup and reformatting.
 *
 * Revision 1.7  2000/04/22 14:57:18  federico
 * Fixed wrong tests in realloc function regarding imem->log_stream.
 *
 * Revision 1.6  2000/03/27 23:30:42  federico
 * Dropped lock support.
 *
 * Revision 1.5  2000/03/25 19:06:44  federico
 * Added mutex/critical section like management
 * It needs to be tested !!
 *
 * Revision 1.4.1.1  2000/03/25 16:05:30  federico
 * Added some stuff to support locking of the imem struc access.
 * Useful for thread sincronization.
 * Untested, uncomplete.
 *
 * Revision 1.4  2000/03/23 22:44:14  federico
 * FIxed bugs in realloc.
 * Added some support for logging operations.
 *
 * Revision 1.3  2000/03/14 00:02:06  federico
 * Many changes to function names.
 * Fixes to _free function.
 * Size maintenance performed if actualy keeping memory block list.
 *
 * Revision 1.1  2000/02/26 21:28:01  federico
 * Initial revision
 *
 */

enum
  {
    PARTIALLY, COMPLETELY
  };

/*
 * GLobal to be used in single threaded programs
 */

imem_t a_global_s;
imem_t *a_global = &a_global_s;

#define A_LINE "\n"

/*
   * public ansi wrappers
 */

/**[txh]*******************************************************
Function: imem_ansi_malloc
Include: imem.h
Module: IMEM
Prototype:
void * imem_ansi_malloc (size_t size, void *op)
Description:
Simply wraps the ANSI C malloc function for use with the IMEM interface
Return:
A pointer to the allocated memory if possible, else NULL, as malloc does.
Example:
@<pre>
imem_t imem;
imem_funcs_t mf = {imem_ansi_malloc, imem_ansi_calloc, ...};

imem_init(&imem, mf, ...);
...
imem_deinit(&imem);

@</pre>
**************************************************************/
void *
a_ansi_malloc (size_t size, void *op)
{
  return malloc (size);
}


/**[txh]*******************************************************
Function: imem_ansi_calloc
Include: imem.h
Module: IMEM
Prototype:
void * imem_ansi_calloc (size_t num, size_t size, void *op)
Description:
Simply wrap the ANSI C calloc function for use with the IMEM
interface.
Return:
A pointer to the allocated memory, else NULL, as calloc does
Example:

@<pre>
imem_t imem;
char * p;
imem_funcs_t mf = {imem_ansi_malloc, imem_ansi_calloc, ...};

imem_init(&imem, mf, ...);

@</pre>
**************************************************************/
void *
a_ansi_calloc (size_t num, size_t size, void *op)
{
  return calloc (num, size);
}


/**[txh]*******************************************************
Function: imem_ansi_realloc
Include: imem.h
Module: IMEM
Prototype:
void *imem_ansi_realloc (void *ptr, size_t size, void *op)
Description:
From the ANSI C realloc function description:
the realloc function changes the size of the object pointed to by
ptr to the size specified by size.
The contents of the object shall be unchanged up to the lesser of
the new and old sizes.
If the new size is larger, the value of the newly allocated
portion of the object is indeterminate.
If ptr is a null pointer, the realloc function behaves
like the malloc function for the specified size.
Otherwise, if ptr does not match a pointer earlier
returned by the calloc, malloc, or realloc function, or if the
space has been deallocated by a call to the free or realloc
function, the behavior is undefined.
If the space cannot be allocated, the object pointed to by ptr
is unchanged.
If the realloc function returns a null pointer when size is zero
and ptr is not a null pointer, the object it pointed to has been freed.

Returns
The realloc function returns either a null pointer or a pointer
to the possibly moved allocated space.
If the object has moved, ptr is a pointer that refers to freed space.
**************************************************************/
void *
a_ansi_realloc (void *p, size_t size, void *op)
{
  return realloc (p, size);
}


/**[txh]*******************************************************
Function: imem_ansi_free
Module: IMEM
Prototype:
void imem_ansi_free (void * p, void * op)
Description:
Simply wraps the ANSI C free function.
**************************************************************/
void
a_ansi_free (void *p, void *op)
{
  free (p);
}

static int
random_failed (imem_t * imem)
{
  DEBUGGING (
      fprintf (stderr, "imem: random failure !!\n");
      fprintf (stderr, "      probability: %.3f, hit: %.3f\n",
               imem->failing_probability, imem->hprob);
    );

  if (imem->log_stream != NULL)
    {
      fputs ("\n", imem->log_stream);
      fprintf (imem->log_stream, "imem: random failure !!\n");
      fprintf (imem->log_stream, "      probability: %.3f, hit: %.3f\n",
               imem->failing_probability, imem->hprob);
      fputs ("\n", imem->log_stream);
    }

  return 1;
}



/*
   FROM GSL GNU Scientific Library 0.41

   This is an implementation of the algorithm used in Numerical
   Recipe's ran0 generator. It is the same as MINSTD with an XOR mask
   of 123459876 on the seed.

   The period of this generator is 2^31.

   Note, if you choose a seed of 123459876 it would give a degenerate
   series 0,0,0,0, ...  I've made that into an error. */

static const long int m = 2147483647, a = 16807, q = 127773, r = 2836;
static const unsigned long int mask = 123459876;

typedef struct
  {
    unsigned long int x;
  }
ran0_state_t;

static unsigned long int
ran0_get (void *vstate)
{
  ran0_state_t *state = (ran0_state_t *) vstate;

  const unsigned long int x = state->x;

  const long int h = x / q;
  const long int t = a * (x - h * q) - h * r;

  if (t < 0)
    {
      state->x = t + m;
    }
  else
    {
      state->x = t;
    }

  return state->x;
}

static double
ran0_get_double (void *vstate)
{
  return ran0_get (vstate) / 2147483647.0 ;
}

static void
ran0_set (void *vstate, unsigned long int s)
{
  ran0_state_t *state = (ran0_state_t *) vstate;

  assert (s != mask);

  state->x = s ^ mask;

  return;
}

static double
random_prob (imem_t * imem)
{
  double d = 0.0;

	imem->hprob = d = ran0_get_double (&imem->seed);

	if (imem->log_stream && (imem->flags & LOG_RANDOM_PROB))
		{
			fprintf (imem->log_stream, "imem: random_prob: %f\n", d);
      fputs ("\n", imem->log_stream);
		}

  return d;
}

void
a_random_test (imem_t * imem, size_t n)
{
  FILE *fp;

  if (imem->log_stream != NULL)
    fp = imem->log_stream;
  else
    fp = fopen ("imem.rnd", "w");
  while (n--)
    fprintf (fp, "[random.test]\t%f\n", random_prob (imem));

  if (imem->log_stream == NULL)
	  fclose (fp);
}

/**[txh]*******************************************************
Function:
unregister_block
Module: IMEM
Prototype:
static int unregister_block (imem_t * imem, const void *p, int how)
Description:
Unregister_block is responsible to unregister an allocated memory block
from the table of allocated blocks and the associated memory.
If the IMEM interface was initialized with the IMEM_WITH_LOG
flags on, the unregistration will be logged as succesfull or not.
HOW controls if the memory held by the block should be free'd or
not: when reallocating a memory region elsewhere, we consider the
pointer to the old memory as free'd.
Return:
IMEM_OK an successfull completion, otherwise (!IMEM_OK)
**************************************************************/
static int
unregister_block (imem_t * imem, const void *p, int how)
{
  hnode_t *node = NULL;
  amb_t *mb = NULL;

  assert (imem->flags & KEEP_AMB);

  if ((node = hash_lookup (imem->amb_list->table, p)) == NULL)
    {
      return (!OK);             /* TODO: NODE_NOT_FOUND */
    }

  mb = hnode_get (node);

  assert (mb != NULL);

  if (how == COMPLETELY)
    IMEM_DEBUG_FREE(mb->ptr, imem->opaque,
			imem->pfree (mb->ptr, imem->opaque));

	if (imem->amb_list->node_pool == NULL)
		{
			IMEM_DEBUG_FREE (mb, imem->opaque, imem->pfree (mb, imem->opaque));
		}
	else
  	node_pool_free_node (imem->amb_list->node_pool, mb);

  hash_delete_free (imem->amb_list->table, node);
  assert (hash_lookup (imem->amb_list->table, p) == NULL);

  /*
   * TODO:
   * - log, if verbosely;
   */

  return OK;
}


/**[txh]*******************************************************
Function: mb_lookup
Module: IMEM
Prototype:
static amb_t * mb_lookup (const amb_list_t * list, const void *p)
Description:
This function seeks for p in the table of the allocated memory
blocks
Return:
NULL if the pointer p isn't in the table of the allocated blocks,
else a pointer to the block carrying its information
**************************************************************/
static amb_t *
mb_lookup (const amb_list_t * list, const void *p)
{
  hnode_t *node;

  if ((node = hash_lookup (list->table, p)) != NULL)
    {
      return (amb_t *) hnode_get (node);
    }
  else
    {
      return NULL;
    }
}

static const char *
mb_function (const amb_t * mb)
{
  return mb->function;
}

static const char *
mb_file (const amb_t * mb)
{
  return mb->file;
}

static int
mb_line (const amb_t * mb)
{
  return mb->line;
}

static size_t
mb_size (const amb_t * mb)
{
  return mb->size;
}


/*
   zero on failure,
   if a block exists, change its attributes ...
 */

/**[txh]*******************************************************
Function: register_block
Module: IMEM
Prototype:
static int
register_block (imem_t * imem, void *p, size_t size,
const int by_who, const char *file, const int line,
const char *function)
Description:
This function tries to insert P in the table of the allocated
memory blocks, adding informations about the function, the file and the
line where the allocation firstly come.
Zero dimensioned blocks are ignored as it is possible that some
allocators returns the same pointer twice if the first time it is
zero sized.
Return:
IMEM_OK on success, (!IMEM_OK) else
**************************************************************/
static int
register_block (imem_t * imem, void *p, size_t size,
                const int by_who, const char *file, const int line,
                const char *function)
{
  amb_t *mb;

  assert (imem != NULL);
  assert (imem->flags & KEEP_AMB);

  /* ignore zero sized blocks */
  if (size == 0 && (imem->flags & WITH_ZB))
    return OK;

  if (imem->amb_list->node_pool != NULL)
    {
      mb = node_pool_get_node (imem->amb_list->node_pool);
    }
  else
    {
      /*
       * we'll get out of this call only if the pmalloc doesn't throw
       * any exception ...
       */
			mb = IMEM_DEBUG_MALLOC (sizeof (*mb),
				imem->opaque, imem->pmalloc (sizeof (*mb), imem->opaque));
    }


  if (mb == NULL)
    return (!OK);

  memset (mb, (char) 0, sizeof (*mb));

  mb->ptr = p;
  mb->size = size;
  mb->mem_space = imem->curr_mem_space;
  mb->how = by_who;
  mb->function = function;
  mb->file = file;
  mb->line = line;

  /*
   * TODO: check for size of hash table
   * TODO: it is illegal to insert a key more than once ...
   *        this can happen with realloc ...
   */

  DEBUGGING (
              if (hash_lookup (imem->amb_list->table, p) != NULL)
         				fprintf (stderr,
									"imem: memory address already in hash table: %p\n"
                  "called from function \"%s()\", file \"%s\", line %d\n"
                  "(file \"%s\", line %d)\n", p, function, file, line,
                  __FILE__, __LINE__);
    );

  assert (hash_lookup (imem->amb_list->table, p) == NULL);

  if (hash_alloc_insert (imem->amb_list->table, p, mb) == 0)
    {
      /* failed ! */

      DEBUGGING (
                  fprintf (stderr, "imem: warning: register_block failed!");
        );

      return (!OK);
    }

  return OK;
}

static int
reregister_block (imem_t * imem, void *old_p, void *new_p, size_t size,
                  const int by_who, const char *file, const int line,
                  const char *function)
{
  /*
   * used by a_realloc: it should unregister old_p in any case ...
   * TODO: here a little optimization is possible ..
   */
  assert (imem->flags & KEEP_AMB);
  unregister_block (imem, old_p, PARTIALLY);
  return register_block (imem, new_p, size, by_who, file, line, function);
}


static hash_val_t
amb_hash_func (const void * key)
{
  static unsigned long randbox[] =
  {
    0x49848f1bU, 0xe6255dbaU, 0x36da5bdcU, 0x47bf94e9U,
    0x8cbcce22U, 0x559fc06aU, 0xd268f536U, 0xe10af79aU,
    0xc1af4d69U, 0x1d2917b5U, 0xec4c304dU, 0x9ee5016cU,
    0x69232f74U, 0xfead7bb3U, 0xe9089ab6U, 0xf012f6aeU,
  };

  /*
   * TODO:
   * - Why unsigned char?
   * - is the size of str_ correct?
   */
  char str_[sizeof (void *) * 8 + 1] = "";
  char *str = str_;
  hash_val_t acc = 0;

  sprintf (str, "%p", key);
  while (*str)
    {
      acc ^= randbox[(*str + acc) & 0xf];
      acc = (acc << 1) | (acc >> 31);
      acc &= 0xffffffffU;
      acc ^= randbox[((*str++ >> 4) + acc) & 0xf];
      acc = (acc << 2) | (acc >> 30);
      acc &= 0xffffffffU;
    }

  return acc;
}


static int
amb_comp_func (const void *key1, const void *key2)
{
  /*
   * key1 and key2 should be just memory addresses
   */

  return (int) key1 - (int) key2;
}


static size_t
amb_get_size (amb_list_t * l, void *p)
{
  amb_t *mb;
  hnode_t *n;

  if ((n = hash_lookup (l->table, p)) == NULL)
    {
      return 0;
    }
  else
    {
      mb = (amb_t *) hnode_get (n);
      return mb->size;
    }
}

/*
 * count amb in memory space ms
 */
static size_t
amb_list_count (amb_list_t * list, size_t ms)
{
  hscan_t hs;
  hnode_t *node = NULL;
  size_t n = 0;

  assert (list != NULL);
  assert (list->table != NULL);

  hash_scan_begin (&hs, list->table);
  while ((node = hash_scan_next (&hs)) != NULL)
    {
      if (((amb_t *) hnode_get (node))->mem_space == ms || ms == 0)
        n++;
    }

  return n;
}

/*
 * while amb is not an 'object by it own I cannot iterate throught
 * ambs and handle the informations to be logged at higher level, so pass
 * a complete imem_t pointer ...
 */
static void
amb_list_deinit (amb_list_t *list, imem_free_t dealloc, void *op)
{
  hscan_t hs;
  amb_t *mb;
  hnode_t *node;

  /*
   * op can be NULL
   */
  assert (dealloc != NULL);
  assert (list != NULL);

  /*
   * recover the data associated with each node and free it,
   * then free the hash table
   */
  hash_scan_begin (&hs, list->table);
  while ((node = hash_scan_next (&hs)) != NULL)
    {
      /*
       * free an allocated memory block
       */
      mb = (amb_t *) hnode_get (node);

			/*
			 * Maybe mb->ptr is a pointer to an allocated memory
			 * area, but actually we leaks: it is not a task for
			 * this function ...
			 */

			/*
			 * UNIFY
			 */
      if (list->node_pool == NULL)
				{
					IMEM_DEBUG_FREE (mb, op, dealloc (mb, op));
				}
      else
				{
					node_pool_free_node (list->node_pool, mb);
				}
		}
  hash_free (list->table);
  list->table = NULL;

	/*
	 * node pool for the hash table is always used
	 */
  node_pool_destroy (list->table_node_pool);

  if (list->node_pool != NULL)
    node_pool_destroy (list->node_pool);
}

/*
 * prototype needed by hashing functions
 */
static void *
amb_node_pool_get_node (size_t size, void *context)
{
  return node_pool_get_node ((node_pool_t *) context);
}

static void
amb_node_pool_free_node (void *node, void *context)
{
  node_pool_free_node ((node_pool_t *) context, node);
}

static int
amb_list_init (amb_list_t * list, int use_pool, imem_malloc_t pmalloc,
               imem_calloc_t pcalloc, imem_realloc_t prealloc,
               imem_free_t pfree, void *opaque)
{
  /*
     * TODO:
     * It would be more correct if the functions used by the hash
     * table wouldn't register memory blocks ...
   */
  list->table = hash_create (HASHCOUNT_T_MAX, amb_comp_func, amb_hash_func,
                             pmalloc, prealloc, pfree, opaque);
  if (list->table == NULL)
    return (!OK);

  list->table_node_pool = &list->_table_node_pool;

  /*
   * initialization below is always OK ...
   */
  node_pool_init (list->table_node_pool, pmalloc, pfree, opaque);
  if (node_pool_create (list->table_node_pool,
                        AMB_IN_NODE_POOL, sizeof (hnode_t)) != NODE_POOL_OK)
    {
      /* free hash table and return error  */
      hash_free (list->table);
      return (!OK);
    }

	/*
	 * Node pool for the hash table
	 */
  hash_set_allocator (list->table, amb_node_pool_get_node,
                      amb_node_pool_free_node, list->table_node_pool);


	/*
	 * Node pool for the list nodes
	 */
  if (use_pool)
    {
      list->node_pool = &list->_node_pool;
      node_pool_init (list->node_pool, pmalloc, pfree, opaque);

      if (node_pool_create (list->node_pool,
                          AMB_IN_NODE_POOL, sizeof (amb_t)) != NODE_POOL_OK)
        {
          /* free hash table, node pool and return error  */
					node_pool_destroy (list->table_node_pool);
          hash_free (list->table);
          return (!OK);
        }
    }
  else
    {
      list->node_pool = NULL;
    }

  return OK;
}

/*
 * Returns the number of deallocated bytes
 */
static int
amb_list_free_mem_space (imem_t * imem, imem_free_t dealloc, void *op,
                         size_t mem_space, size_t * size, int *n)
{
  hscan_t hs;
  amb_t *mb;
  hnode_t *node;
  size_t ntotal = 0, nms = 0;
	amb_list_t * list;
	int i = 0;

	assert (imem != NULL);
  assert (dealloc != NULL);
  assert (size != NULL);
  assert (n != NULL);

	list = imem->amb_list;
  assert (list != NULL);

  *size = 0;
  *n = 0;
  ntotal = amb_list_count (list, 0);
	nms = amb_list_count (list, mem_space);

	if (imem->log_stream != NULL)
		{
			fprintf (imem->log_stream,
				A_LINE
				"imem: amb_list_free_mem_space: there are %ld allocated blocks\n"
				"      in memory space %ld, on a total of %ld\n"
				A_LINE, nms, mem_space, ntotal);

		}

  hash_scan_begin (&hs, list->table);
  while ((node = hash_scan_next (&hs)) != NULL)
    {
      mb = (amb_t *) hnode_get (node);
			assert (mb != NULL);
      if (mem_space == 0 || mb->mem_space == mem_space)
        {
          (*n)++;
          (*size) += mb->size;

      		if (imem->log_stream != NULL)
        		{
          		fprintf (imem->log_stream,
                   A_LINE
                   "imem: amb_list_free_mem_space: freeing memory block %d at %p\n"
                   "      sized %ld bytes, memory space %ld\n"
                   "      allocated by \"%s()\",\n"
                   "      file %s, line %d\n"
                   A_LINE, ++i, mb->ptr, mb->size, mem_space,
									 mb->function, mb->file, mb->line);
        		}

					/*
					 * free the memory block using the supplied deallocator
					 */
          dealloc (mb->ptr, op);

		      if (list->node_pool == NULL)
						{
							IMEM_DEBUG_FREE (mb, op, dealloc (mb, op));
						}
		      else
		        node_pool_free_node (list->node_pool, mb);

          hash_scan_delete (list->table, node);
          IMEM_DEBUG_FREE (node, list->table->node_context,
						list->table->hash_freenode (node, list->table->node_context));
        }

      DEBUGGING (
                  if (amb_list_count (list, 0) != ntotal - (*n))
        	          {
        		          fprintf (stderr, "In list: %d, erased: %d of %d\n",
                           amb_list_count (list, 0), *n, ntotal);
        	          }
      );

      assert (amb_list_count (list, 0) == ntotal - (*n));

    }
  return OK;
}


static int
amb_list_mem_space_nbytes (amb_list_t * list, size_t mem_space,
                           size_t * size, int *n)
{
  hscan_t hs;
  amb_t *mb;
  hnode_t *node;

  assert (n != NULL);

  *size = 0;
  *n = 0;

  hash_scan_begin (&hs, list->table);
  while ((node = hash_scan_next (&hs)) != NULL)
    {
      mb = (amb_t *) hnode_get (node);
      if (mem_space == 0 || mb->mem_space == mem_space)
        {
          (*n)++;
          (*size) += mb->size;
        }
    }
  return OK;
}

static void *
raw_malloc (imem_t * imem, size_t size, const char *file,
            const int line, const char *function, int by)
{
  /*
   * - Lock the access to imem if needed;
   * - Try to allocate size bytes:
   *   if fails check the on_out_of_memory field and throw exception
   *   if needed, else return NULL
   * - if the request is successfull register the allocated block
   */
  void *p = NULL;

  assert (imem != NULL);
  assert (imem->pmalloc != NULL);
  /* assert (imem->zb != NULL); */

  /*
	 * ignore zero sized blocks if requested
	 */
  if (size == 0 && (imem->flags & WITH_ZB))
    return imem->pmalloc (0, imem->opaque);

	/*
	 * Hack to avoid a complicate condition in the if expression
	 * below using shot-circuit-evaluation
	 */
  if ((imem->failing_probability > 0.0
       && random_prob (imem) <= imem->failing_probability
       && random_failed (imem)))
    {
      goto RANDOM_FAILED;
    }

  p = imem->pmalloc (size, imem->opaque);

  /*
   * out of memory, or failed registration
   */
  if (p == NULL || (imem->flags & KEEP_AMB &&
                 !register_block (imem, p, size, by, file, line, function)))
    {
    RANDOM_FAILED:
      if (p != NULL)            /* registration/random failed */
        imem->pfree (p, imem->opaque);

      switch (imem->on_failure)
        {
        case NO_THROW:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to allocate %ld bytes\n"
                       "               request by \"%s()\" with \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "---------------\n"
                       A_LINE,
                       size, function,
                       by == BY_CALLOC ? IMEM_MAKE_NAME (imem.calloc) : IMEM_MAKE_NAME (imem.malloc),
                       file, line);
              fflush (imem->log_stream);
            }
          return NULL;          /* as ANSI malloc ... */
        case THROW:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to allocate %ld bytes\n"
                       "               request by \"%s()\" with \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "               THROWING EXCEPTION\n"
                       "---------------\n"
                       A_LINE,
                       size, function,
                       by == BY_CALLOC ?
                       IMEM_MAKE_NAME (imem.calloc) :
                       IMEM_MAKE_NAME (imem.malloc),
                       file, line);
              fflush (imem->log_stream);
            }
          except_throw (imem->except_p, XCEPT_BAD_ALLOC, 1, "Out of memory");
          assert (1 == 0);      /* never here ! */
          break;
        default:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to allocate %ld bytes\n"
                       "               request by \"%s()\" with \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "               THROWING EXCEPTION\n"
                       "---------------\n"
                       A_LINE,
                       size, function,
                       by == BY_CALLOC ?
                       IMEM_MAKE_NAME (imem.calloc) :
                       IMEM_MAKE_NAME (imem.malloc),
                       file, line);
              fflush (imem->log_stream);
            }
          except_throw (imem->except_p, XCEPT_BAD_ALLOC, 1,
                        "Out of memory and bad parameter"
                        "for memory interface (allocation)");
          assert (1 == 0);      /* never here ! */
          break;
        }

    }

  return p;                     /* keep going, all right ! */
}



void *
a_realloc (imem_t * imem, void *p, size_t size, const char *file,
           const int line, const char *function)
{
  /*
   * try realloc, iff OK reregister allocated block,
   * changing its attribute.
   * Handles realloc (p, 0)
   */
  const char *Ltext =
  "imem: imem_realloc: free'd %p (size: %d) at request of \"%s()\",\n"
  "      file \"%s\", line %d\n"
  "      this pointer was allocated by \"%s()\"\n"
  "      in file \"%s\", line %d ...\n"
  A_LINE
   ;

  const char *Rtext =
  "imem: imem_realloc: reallocated a block at %p sized %d bytes\n"
  "      into a block at %p sized %d\n"
  "      at request of \"%s()\", file %s, line %d\n"
  "      original allocation in function \"%s()\",\n"
  "      file \"%s\", line %d ..."
   ;
  void *new_p = NULL, *old_p = p;
  size_t old_size = 0;
  const amb_t *mb = NULL;

  assert (imem != NULL);

  if (p != NULL && imem->flags & KEEP_AMB)
    {
      old_size = amb_get_size (imem->amb_list, p);
      mb = mb_lookup (imem->amb_list, p);
    }


  if (imem->failing_probability > 0.0
      && random_prob (imem) <= imem->failing_probability
      && random_failed (imem))
    {
      /*
       * bypass all the stuff below ...
       */
      goto RANDOM_FAILED;
    }

  new_p = imem->prealloc (old_p, size, imem->opaque);
  /*
   * TODO: here behaves like free: check and document,
   *       allow customization: consider using the client
   *       function
   */
	/*
	 * should be (size == 0 && old_p != NULL && new_p == NULL)
	 * according to lcc realloc documentation .. but in some
	 * cases (eg. DJGPP v2.02) new_p is not NULL!
	 */
	if (size == 0 && old_p != NULL && new_p != NULL
		&& imem->log_stream != NULL)
		{
			fprintf (imem->log_stream,
				A_LINE
				"imem: friendly warning: the re-allocator doesn't seem to\n"
				"                        return NULL when reallocating a \n"
				"                        non-NULL pointer to a size\n"
				"                        of zero bytes\n"
				A_LINE);
		}
  if (size == 0 && old_p != NULL && new_p == NULL)
    {

      /*
         * log before unregister ...
         * maybe we can't find it if it was a zero sized block
         * not registered
       */

      if (imem->log_stream != NULL)
        {
          fprintf (imem->log_stream, Ltext, p,
                   mb != NULL ? mb_size (mb) : 0,
                   function, file, line,
                   mb != NULL ? mb_function (mb) : "???",
                   mb != NULL ? mb_file (mb) : "???",
                   mb != NULL ? mb_line (mb) : 0);
          fflush (imem->log_stream);
        }
      /*
         * consider the pointer as free'd ..
         * so unregister it if the old size was != 0
         * TODO: is this consistent ?
       */
      if ((imem->flags & KEEP_AMB) && OK != unregister_block (imem, p, COMPLETELY)
          && old_size != 0)
        {
          /*
           * TODO:
           * - what to do ?
           */
          DEBUGGING (
              fprintf (stderr, "imem: warning: unregister_block failed!\n");
                fprintf (stderr, "file: %s, line %d\n", __FILE__, __LINE__);
            );
        }

      assert (imem->size >= old_size);

      imem->size -= old_size;

      assert (new_p == NULL);
      return new_p;             /* should be NULL */
    }

  /*
   * LOG
   */
  if (imem->log_stream != NULL && new_p != NULL)
    {
      fprintf (imem->log_stream, Rtext,
               old_p, old_size, new_p, size, function, file, line,
               mb != NULL ? mb_function (mb) : "???",
               mb != NULL ? mb_file (mb) : "???",
               mb != NULL ? mb_line (mb) : 0);
      fflush (imem->log_stream);
    }


  /*
   * out of memory, or failed registration
   */
  if (new_p == NULL || (imem->flags & KEEP_AMB
                     && !reregister_block (imem, p, new_p, size, BY_REALLOC,
                                           file, line, function)))
    {
    RANDOM_FAILED:
      /*
       * HERE: failed to reregister, to reallocate
       * or by 'random cause' ...
       *
       * TODO: verify this one below
       * reregistration will fail only if realloc is
       * behaving like malloc (p == NULL), so free new_p safely,
       * or on out of memory.
       */
      if (new_p != NULL)
        imem->pfree (new_p, imem->opaque);

      switch (imem->on_failure)
        {
        case NO_THROW:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to reallocate %ld bytes\n"
                       "               request by \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "---------------\n"
                       A_LINE,
                       old_size, function, file, line);
            }
          return NULL;          /* as ANSI remalloc ... */
        case THROW:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to reallocate %ld bytes\n"
                       "               request by \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "               TRHOWING EXCEPTION!\n"
                       "---------------\n"
                       A_LINE,
                       old_size, function, file, line);
							fflush (imem->log_stream);
            }
          except_throw (imem->except_p, XCEPT_BAD_ALLOC, 1, "Out of memory");
          break;
        default:
          if (imem->log_stream != NULL)
            {
              fprintf (imem->log_stream,
                       A_LINE
                       "---------------\n"
                       "imem: WARNING: failed to reallocate %ld bytes\n"
                       "               request by \"%s()\"\n"
                       "               file \"%s\", line %d\n"
                       "               TRHOWING EXCEPTION!\n"
                       "---------------\n"
                       A_LINE,
                       old_size, function, file, line);
							fflush (imem->log_stream);
            }
          except_throw (imem->except_p, XCEPT_BAD_ALLOC, 1,
                        "Out of memory and bad parameter"
                        "for memory interface (rellocation)");
          break;
        }
    }

  if (imem->flags & KEEP_AMB)
    {
      /*
			 * assert (imem->size == 0 || imem->size >= (size - old_size));
			 */
      imem->size += (size - old_size);
    }

  /*
   * TODO: log, if requested
   */

  if (imem->log_stream != NULL)
    {
      fprintf (imem->log_stream, " OK !\n" A_LINE);
      fflush (imem->log_stream);
    }

  return new_p;                 /* keep going, all right ! */
}

void *
a_malloc (imem_t * imem, size_t size, const char *file,
          const int line, const char *function)
{
  void *p = NULL;

  assert (imem != NULL);

  p = raw_malloc (imem, size, file, line, function, BY_MALLOC);

  /*
   * LOG
   */
  if (imem->log_stream != NULL)
    {
      fprintf (imem->log_stream,
               "imem: imem_malloc: allocated %ld bytes at %p\n"
               "      request by \"%s()\", file \"%s\", line %d\n"
               A_LINE,
               size, p, function, file, line);
      fflush (imem->log_stream);
    }

  if (imem->flags & KEEP_AMB)
    {
			/*
			 * TODO:
			 * HOW to chek for overflow ?
			 */
      imem->size += size;
    }

  return p;
}

void *
a_calloc (imem_t * imem, size_t num, size_t size,
          const char *file, const int line, const char *function)
{
  void *p = NULL;

  assert (imem != NULL);

  p = raw_malloc (imem, size, file, line, function, BY_CALLOC);
  if (p != NULL)
    memset (p, 0, size * num);

  /*
   * LOG
   */
  if (imem->log_stream != NULL)
    {
      fprintf (imem->log_stream,
               "imem: imem_calloc: allocated %ld bytes at %p\n"
               "      request by \"%s()\", file \"%s\", line %d\n"
               A_LINE,
               size, p, function, file, line);
      fflush (imem->log_stream);
    }

  if (imem->flags & KEEP_AMB)
    {
      imem->size += size;
      assert (imem->size > 0);
    }

  return p;
}

void
a_free (imem_t * imem, void *ptr, const char *file,
        const int line, const char *function)
{
  /*
   * search the block, if it is not in the table of allocated blocks,
   * issue a warning, throw exception or what ?
   * if the block is in the table, remove the element who hold its property
   * and free the block as requested.
   */
  const amb_t *mb = NULL;
  size_t size;
  const char *Ltext =
  "imem: imem_free: freeing %p (size: %d) at request of \"%s()\"\n"
  "      file \"%s\", line %d\n"
  "      this pointer was allocated by \"%s()\"\n"
  "      in file \"%s\", line %d ...";

  if (ptr == NULL)
    return;

  if (imem->log_stream != NULL)
    {

      if (imem->flags & KEEP_AMB)
        mb = mb_lookup (imem->amb_list, ptr);

      if (imem->log_stream)
        {
          fprintf (imem->log_stream, Ltext, ptr,
                   mb != NULL ? mb_size (mb) : 0,
                   function, file, line,
                   mb != NULL ? mb_function (mb) : "???",
                   mb != NULL ? mb_file (mb) : "???",
                   mb != NULL ? mb_line (mb) : 0);
          fflush (imem->log_stream);
        }
    }


  if (imem->flags & KEEP_AMB)
    {
      size = amb_get_size (imem->amb_list, ptr);
      if ((mb != NULL) && OK != unregister_block (imem, ptr, COMPLETELY))
        {
          DEBUGGING (
              fprintf (stderr, "imem: warning: unregister_block failed!\n");
                      fprintf (stderr, "at file: %s, line %d\n",
                               __FILE__, __LINE__);
            );
          /*
           * What to do ?
           * - 1) LOG THE FAILED unregistration
           * - 2) ?
           */
          if (imem->flags & WITH_LOG && imem->log_stream != NULL)
            {
							fflush (imem->log_stream);
              fprintf (imem->log_stream,
								A_LINE
                "imem: WARNING: failed to unregister block at (%p) "
								"sized %ld bytes\n", ptr, size);
              fprintf (imem->log_stream,
                "      request in function \"%s()\" at file: %s, line %d\n",
                function, file, line);
              fprintf (imem->log_stream,
                "      [from \"%s()\", file: %s, line %d]\n"
		A_LINE, "imem_free", __FILE__, __LINE__);
              fprintf (imem->log_stream, A_LINE);
              fflush (imem->log_stream);
            }
        }

      assert (imem->size >= size);
      imem->size -= size;
    }

  imem->pfree (ptr, imem->opaque);
  if (imem->log_stream != NULL)
    {
      fputs (" OK\n" A_LINE, imem->log_stream);
      fflush (imem->log_stream);
    }

  return;
}


void *
a_get_opaque (imem_t * imem)
{
  return imem->opaque;
}

/*
 * undefined behaviour if request to
 * set_mem_space with NO_KEEP_AMB
 */
size_t
a_set_mem_space (imem_t * imem, size_t mem_space)
{
  /*
   * Sets the current memory space index,
   * returns the previous memory space;
   */
  size_t old = imem->curr_mem_space;

  imem->curr_mem_space = mem_space;

  return old;
}

size_t
a_get_mem_space (const imem_t * imem)
{
  return imem->curr_mem_space;
}


void
a_free_mem_space (imem_t * imem, size_t mem_space)
{
  /*
   * Frees all the blocks associated with mem_space memory space
   * or with a memory space greater than that. The idea is that
   * a memory space can own the others, gerarchically 'inferior'.
   * Usefull after thread cancellation, if another thread has kept
   * property of an imem interface (:->)
   * A memory space is identified by a number greater than 0.
   * A number equals to 0 means all the memory spaces.
   * The IMEM interface doesn't know the implementation of amb_list !
   */
  size_t lsize = 0;
  int n = 0, n1;

  assert (imem != NULL);
  assert (imem->flags & KEEP_AMB);
  assert (mem_space <= a_get_mem_space (imem));


  if (mem_space == 0)
    DEBUGGING (
                fprintf (stderr, "imem: size to free: %d, n. blocks: %d\n",
                         a_mem_space_nbytes (imem, 0),
                         a_mem_space_nblocks (imem, 0));
    );

  n1 = a_mem_space_nblocks (imem, 0);

  amb_list_free_mem_space (imem, imem->pfree,
                           imem->opaque, mem_space, &lsize, &n);

  if (imem->flags & KEEP_AMB)
    imem->size -= lsize;

  if (mem_space == 0)
    {
      DEBUGGING (
                  fprintf (stderr, "imem: imem size: %d, size freed: %d, "
                           "n.freed: %d, were: %d\n",
                           imem->size, lsize, n, n1);
        );
      assert (imem->size == 0);
    }

	if (imem->log_stream != NULL)
		fprintf (imem->log_stream,
			"imem: imem_free_mem_space: free'd mem_space number %.0f\n"
			A_LINE , (double) mem_space);
}

/*
 * when called by imem_deinit:
 * - if imem->deinit != NULL then the deinit function has been called,
 */
void
a_log_info (imem_t * imem)
{
  static const char *info_fmt =
  "imem: Information about the memory manager:\n"
  "      name: %s\n"
  "      allocated blocks(max): %ld\n"
  "      allocated bytes(max): %ld\n"
  "      options: %s\n"
  "      malloc:  %s (%p)\n"
  "      calloc:  %s (%p)\n"
  "      realloc: %s (%p)\n"
  "      free:    %s (%p)\n"
  "      opaque:  %p\n"
  "      current memory space: %d\n"
  "      size: %ld\n"
  "      failing probability: %f\n"
  A_LINE;

  static const char *no_info =
  "imem: no information available about the memory manager\n"
  A_LINE;

  assert (imem->log_stream != NULL);

  if (imem->log_stream != NULL)
    {
      size_t size;
      int n;
      char *options_string = imem->flags_name;

			/*
			 * if we are keeping propery of memory block,
			 * give an appropriate (?)information,
			 * otherwise ignore ...
			 */
			if (imem->flags & KEEP_AMB)
				{
	      	amb_list_mem_space_nbytes (imem->amb_list, 0, &size, &n);
		      fprintf (imem->log_stream, info_fmt,
               imem->name, n, size,
               options_string,
               imem->malloc_name, imem->pmalloc,
							 imem->calloc_name, imem->pcalloc,
							 imem->realloc_name, imem->prealloc,
							 imem->free_name, imem->pfree,
							 imem->opaque, imem->curr_mem_space, imem->size,
							 imem->failing_probability);
				}
			else
				{
		      fprintf (imem->log_stream, no_info);
				}
    }
}

static int
a_log_start (imem_t * imem)
{
  static const char *header =
  A_LINE
  "IMEM: memory management interface for C\n"
	A_LINE
  "      This is free software, it comes with ABSOLUTELY NO WARRANTY\n"
  "      Copyright (C) 2000 Federico Spinazzi <fspinazzi@hotmail.com>\n"
  "      Portions Copyright (C) 1997 Kaz Kylheku <kaz@ashi.footprints.net>\n"
  "      Portions Copyright (C) 1995 DJ Delorie <dj@delorie.com>\n"
  "      Portions Copyright (C) ???? Thomas Niemann <niemann@yahoo.com>\n"
  "      This version compiled " __DATE__ ", " __TIME__ "\n"
  "      Log started at %s"
	A_LINE
  " Free Software License:\n"
  "      All rights are reserved by the author, with the\n"
  "      following exceptions:\n"
	"      Permission is granted to freely reproduce and distribute this\n"
	"      software possibly in exchange for a fee, provided that this\n"
	"      copyright notice appears intact.\n"
	"      Permission is also granted to adapt this software to produce\n"
  "      derivative works, as long as the modified versions carry\n"
	"      this copyright notice and additional notices stating that the\n"
	"      work has been modified.\n"
	"      The source code may be translated into executable form\n"
	"      and incorporated into proprietary software; there is no\n"
	"      requirement for such software to contain a copyright notice\n"
	"      related to this source.\n"
  "-------------------------------------\n"
   ;

  time_t now;
  time (&now);

  assert (imem->log_stream != NULL);


  fprintf (imem->log_stream, header, asctime (localtime (&now)));

	/* General information */
	a_log_info (imem);

  if (ferror (imem->log_stream))
    return (!OK);
  else
    return OK;
}

static int
a_log_end (imem_t * imem)
{
  static const char *footer =
  A_LINE
  "IMEM :Log ended at %s"
  "-------------------------------------------------\n"
   ;


  time_t now;
  time (&now);

  assert (imem->log_stream != NULL);
  fprintf (imem->log_stream, footer, asctime (localtime (&now)));

  if (ferror (imem->log_stream))
    return (!OK);
  else
    return OK;
}

int
a_init (imem_t * imem, const imem_funcs_t * imem_funcs,
        struct except_params *except, const char *name)
{
  int use_pool = NO;

  assert (imem != NULL);
  assert (imem_funcs != NULL);
  assert (name != NULL);

  imem->flags = imem_funcs->flags;
  /* imem->zb = &imem->_zb; */
	/* imem->zb = imem_funcs->pmalloc (0, imem_funcs->opaque); */
  imem->name = name;

  if (imem->flags & THROW)
    {
      assert (except != NULL);
      imem->on_failure = THROW;
    }
  else
    {
      imem->on_failure = NO_THROW;
    }

  imem->pmalloc = imem_funcs->pmalloc;
  imem->pcalloc = imem_funcs->pcalloc;
  imem->pfree = imem_funcs->pfree;
  imem->prealloc = imem_funcs->prealloc;
  imem->opaque = imem_funcs->opaque;

	imem->malloc_name = imem_funcs->malloc_name;
	imem->calloc_name = imem_funcs->calloc_name;
	imem->realloc_name = imem_funcs->realloc_name;
	imem->free_name = imem_funcs->free_name;
	imem->flags_name = imem_funcs->flags_name;

  if (imem->flags & WITH_CUSTOM_CLEANUP)
    {
      imem->pdeinit = imem_funcs->pdeinit;
    }

  imem->curr_mem_space = 0;
  imem->size = 0;

  /*
   * exceptions
   */
  imem->except_p = except;

  /*
   * node pool
   */
  if (imem->flags & USE_POOL)
    {
      use_pool = YES;
    }

  /*
   * allocated memory blocks list
   */
  imem->amb_list = &imem->_amb_list;
  if (imem->flags & KEEP_AMB
      && (OK != amb_list_init (imem->amb_list, use_pool,
                               imem_funcs->pmalloc, imem_funcs->pcalloc,
                               imem_funcs->prealloc, imem_funcs->pfree,
                               imem_funcs->opaque)))
    {
      /*
       * cannot initialize the list of allocated blocks
       */
      memset (imem, (char) 0, sizeof (*imem));
      return (!OK);
    }

  /*
   * for now every allocation is succesfull (if there is space
   * in the heap)
   */
  imem->failing_probability = 0.0;

  /* TODO: issue a warning if the file cannot be opened */
  if (imem->flags & WITH_LOG)
    {
      imem->log_stream_name = name;
      imem->log_stream = fopen (name, "w");
      a_log_start (imem);
    }
  else
    {
      imem->log_stream_name = NULL;
      imem->log_stream = NULL;
    }

  return OK;
}



int
a_deinit (imem_t * imem, int flags)
{
  /*
   * free the memory needed by the interface,
   * silently frees the memory hold by this interface (this is
   * a problem if the memory is shared with other owners,
   * but this last thing is not very good, I think),
   * set all the fields to NULL,
   *
   * currently ignore flags ...
   *
   */

  int should_continue = 1;

  if (imem->log_stream != NULL)
    {
      a_log_info (imem);
    }

  /*
   * call an user defined function
   */
  if (imem->flags & WITH_CUSTOM_CLEANUP)
    {
      assert (imem->pdeinit != NULL);
			/*
			 * TODO:
			 * - the client code cannot access imem field so:
			 * -    1) maybe we'll pass only imem->opaque;
			 * ---> 2) we'll give a complete interface to imem fields:
			 *      imem_get_allocator, ...
			 */
      should_continue = imem->pdeinit (imem);
		  if (imem->log_stream != NULL)
		    {
		      fprintf (imem->log_stream,
						A_LINE
						"imem: imem_deinit: custom cleanup called\n"
						A_LINE);
		    }
    }

  /*
   * destroy the amb_list if present and if the client cleanup
   * handler request this ... maybe the custom cleanup has
	 * deleted the heap ...
   */
  if (should_continue)
    {
      if (imem->flags & KEEP_AMB)
        {
          amb_list_deinit (imem->amb_list, imem->pfree, imem->opaque);
        }

    }

  if (imem->log_stream != NULL)
    {
      a_log_end (imem);
      fclose (imem->log_stream);
    }

  memset (imem, (char) 0, sizeof (imem_t));

#ifdef MSS
 MSS_LOG_BLOCK_LIST;
#endif

  return OK;
}

int
a_init_ansi (imem_t * imem)
{
  /* uses the malloc family functions, wrapped here for convenience */
  /* TODO: allow to keep track of memory blocks ... */
  static const imem_funcs_t mf =
  {NO_KEEP_AMB | NO_THROW | NO_USE_POOL,
   a_ansi_malloc, a_ansi_calloc,
   a_ansi_realloc, a_ansi_free, NULL};

  return a_init (imem, &mf, NULL, IMEM_MAKE_LOG_NAME ("ansi"));
}

size_t
a_nbytes (imem_t * imem)
{
  return imem->size;
}

size_t
a_mem_space_nblocks (imem_t * imem, size_t ms)
{
  size_t n;

  n = amb_list_count (imem->amb_list, ms);

  return n;
}


size_t
a_mem_space_nbytes (imem_t * imem, size_t ms)
{
  int n = 0;
  size_t n2 = 0;

  amb_list_mem_space_nbytes (imem->amb_list, ms, &n2, &n);

  return n2;
}

void
a_set_random_fail (imem_t * imem, double prob, unsigned long int seed)
{
  assert (imem != NULL);
  assert (prob >= 0.0 && prob <= 100.0);
  imem->failing_probability = prob;
	imem->seed = seed;

	ran0_set (&imem->seed, seed);
}

void
a_init_funcs (imem_funcs_t * mf, int flags,
	imem_malloc_t fmalloc, imem_calloc_t fcalloc,
	imem_realloc_t frealloc, imem_free_t ffree, void *opaque, \
	char * nflags, char * nmalloc, char * ncalloc,
	char * nrealloc, char * nfree)
{
	assert (mf != NULL);
	mf->pmalloc = fmalloc;
	mf->pcalloc = fcalloc;
	mf->prealloc = frealloc;
	mf->pfree = ffree;
	mf->opaque = opaque;
	mf->flags = flags;

	mf->malloc_name = nmalloc;
	mf->calloc_name = ncalloc;
	mf->realloc_name = nrealloc;
	mf->free_name = nfree;
	mf->flags_name = nflags;
}

#if 0
/*
 * WORDWRAPPING function stolen from MSS code ...
 */
const char *
mss_word_wrap (const char *label, const char *str)
{
  int label_len = 0;
  int position = 0;
  int last_space = 0;
  const char *src, *src_temp;
  char *dest;
  const int PAGE_WIDHT = 77;
  char ww_str[1024];


  if (!MSS_DO_WORD_WRAP)
    {
      if (strlen (label) > 0)
        {
          strcpy (ww_str, label);
          strcat (ww_str, ": ");
          strcat (ww_str, str);
        }
      else
        strcpy (ww_str, str);

      return ww_str;
    }

  /* Now we start word wrapping the text. */
  if (label != NULL)
    {
      label_len = strlen (label) + 2;
      if (strlen (label) > 0)
        {
          strcpy (ww_str, label);
          strcat (ww_str, ": ");
        }
      else
        ww_str[0] = '\0';
    }
  else
    label_len = 0;

  position = label_len;
  src = str;
  dest = ww_str + label_len;

  while (*src != '\0')
    {
      position = label_len;
      last_space = -1;
      src_temp = src;
      while (position < PAGE_WIDHT - 1)
        {
          if (*src == ' ')
            last_space = position;

          if (*src == '\0')
            {
              last_space = position;
              break;
            }
          src++;
          position++;
        };
      src = src_temp;
      position = last_space;
      if (last_space == -1)
        last_space = PAGE_WIDHT - 1;

      position = label_len;
      while (position <= last_space)
        {
          *dest++ = *src++;
          position++;
        };
      if (*(src - 1) == '\0')
        return ww_str;

      *dest++ = '\n';

      position = 0;
      while (position < label_len)
        {
          *dest++ = ' ';
          position++;
        };
    };
  *dest = '\0';


  return ww_str;
}

#endif /* 0 */
