/*******************************************************************************************************
/  Translation of Michael Prager's For2R routines
/   Author:
/         Andi Stephens
/         NOAA, Beaufort, NC
/         andi.stephens@noaa.gov
/         jennifer.martin@noaa.gov
/         mike.prager@noaa.gov
/ Date annotated:
/          Dec. 28, 2005
/ Language: C
/ Purpose:
/          This package has routines for writing R-compatible data output.
/          Output is written into a file that R can read with the dget() function.
/          Example from R prompt:
/                > myvar = dget("myfile.txt")
********************************************************************************************************/
/* CHANGELOG OF C2R
/ Changed names of 2 routines from ...info_vector to ...info_list
/ to correspond with R structure. MHP, 8/8/2006
/ Fixed position of comma in reading comment, MHP, 8/9/2006.
/ Changed version to 1.00 for release.  MHP, 8/9/2006.
/ 1.00 Fixed mismatched braces, MHP/AS, 9/7/06.
/ 1.00 Added (void) to argument list of two functions, MHP, 9/8/06.
/ 1.01 Write default row.names for data frames with no specified row names. JLM, 3/9/07.
/ 1.02 Fixed when to print (or not print) commas before object names. JLM, 9/18/07.
/    - Standardized the printing of blank lines after each object completes. JLM, 9/18/07.
/    - INFO object can now be other than the first sub-object. JLM, 9/18/07. 
********************************************************************************************************/

#include <time.h>       // for info date and time
#include <string.h>     // for string manipulation
#include <stdio.h>      // input, output
#include <stdarg.h>     // variable arguments
#include <stdlib.h>     // miscellaneous

#define EOF (-1)        // Probably now predefined in most compilers.

#ifndef TRUE
    #define TRUE (1==1)
    #define FALSE (1==0)
#endif


/*********************************************************************************************************
/*  GLOBALS
/       NCOMP       Number of components and sumcomponents written
/       LEVEL       Current nesting level
/       MAXLEVEL    Max number of levels allowed
/       MAXCOMP     Max number of components per level allowed
/       NAMES       Array of character strings containing component names
/       DFLEN       Used by data-frame routines to store working column length
**********************************************************************************************************/

typedef char **namelist;
typedef char r_name[128];
typedef int *numnames;

int dflen=0, level=0, prevlevel=0, maxlevel=128, maxcomp=128, ndigits=9;

namelist *names;
numnames nnames;

FILE *OUTFILE;

char *nachar = "NA";
char *version = "1.02";
r_name float_fmt = "%.6e";

/*****************************************************************************************************
/ WRT_R_COMMENT
/
/  Write a comment to the ouput file
*****************************************************************************************************/
void wrt_r_comment(char *text){

    fprintf(OUTFILE, "### %s\n", text);

}// End WRT_R_COMMENT

/**********************************************************************************************************
/ OPEN_R_FILE
/   Open a file to hold an S data object and initialize the object
/   Allocate array to hold component levels
/   ARGUMENTS
/       fname - output file name
/       mxlevel - max nesting level of subcomponents
/       mxcomp - max components within a level (e.g., cols within dataframe)
/       ndigits - digits after decimal point in real format for writing
**********************************************************************************************************/
void open_r_file(char *fname, int max_comp, int max_level, int ndigits){
    r_name    string1, string2, string3;
    int     i, j;

    sprintf(string1, "This file written with C2R version %s ", version);
    sprintf(string2, "Read this file into R or S with x=dget(\"%s\")", fname);
    sprintf(string3,
            "C2R originated by Mike.Prager@noaa.gov."
            "  Please credit author and report bugs/improvements.");

    // Initialize level variables
    level = 0;
    prevlevel = -1;

    if (max_comp > 0) {
        maxcomp = max_comp;
    }
    if (max_level > 0) {
        maxlevel = max_level;
    }
    if (ndigits > 0) {
        sprintf(float_fmt, "%s%de", "%.", ndigits);
    }

    // Allocate 2-d array to hold names

    names = (namelist *)malloc(maxlevel * sizeof(namelist *));
    for (i=0; i < maxlevel; i++) {
        names[i] = (char **)malloc(maxcomp * sizeof(char **));
        for (j=0; j < maxcomp; j++) {
            names[i][j] = (char *)malloc(sizeof(r_name));
            names[i][j] = "";
        }
    } // for i

    // Allocate array to hold number of names per level
//    nnames = (int *)malloc(sizeof(int(maxlevel)));
    nnames = (int *)malloc(sizeof(int)*maxlevel);
    for (i=0; i < maxlevel; i++) {
        nnames[i] = -1;
    }

    OUTFILE = fopen(fname, "w+");
    if (OUTFILE == NULL) {
        printf("Unable to write to file %s\n");
        exit(1);
    }
    wrt_r_comment(string1);
    wrt_r_comment(string2);
    wrt_r_comment(string3);
    fprintf(OUTFILE,"\nstructure(list(\n\n");
} // end OPEN_R_FILE

/*************************************************************************************
/ REG_RNAMES
/ Subroutine to keep track of names of the components in the S structure.
/
**************************************************************************************/
void reg_rnames(char *newname) {

	// Check for invalid nesting levels
    if (level >= maxlevel) {
        printf("Error: Too many levels in REG_RNAMES.  Level= %d\n", level);
        exit(2);
    }
    if (level < 0) {
        printf("Error:  Level can\'t be zero in REG_RNAMES.\n");
        exit(2);
    }
    // See if level has changed, and if so, take appropriate action:
    if (level == prevlevel) {
        ;
    }
    else if (level < prevlevel) {
        prevlevel = level;
    }
    else if (level == prevlevel + 1) {
        // Initialize new level
        nnames[level] = -1;
        prevlevel = level;
    }
    else {
        printf("Note: level change unexpected in REG_RNAMES.  Current level= %d, and previous level = %d\n",
                level+1, prevlevel+1);
        exit(3);
    }
    // Keep count of the number of names at this level:
    nnames[level]++;

    // Store current name in the NAMES array:
    names[level][nnames[level]] = newname;
}

/*************************************************************************************
/ OPEN_R_INFO_LIST
/ Initialze an INFO object and write its DATE subobject.
/ All main S objects are assumed to begin with an INFO object.
/
/ The INFO object contains descriptive information about the data structure.
/ It ALWAYS contains the date as the first item, and it MUST contain
/ at least one other item.
/ Changed INFO_VECTOR to INFO_LIST ... MHP 8/8/06
***************************************************************************************/
void open_r_info_list(char *newname) {

    time_t  ltime;
    struct  tm *today;
    char    tmpbuf[50];

    reg_rnames(newname);

	// If nnames (level) == 0, this is the first item at this level,
    // and we omit the comma.
	if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
	// Write output to start the info subobject.
	fprintf(OUTFILE, "info=structure(list\n");

    level++;

	// Automatically write the first subobject, the date and time
    time(&ltime);               // Gets time as a long integer
    today = localtime(&ltime);  //Converts to local time
    strftime(tmpbuf, 50, "%A, %d %b %Y at %H:%M:%S", today);  // apply formatting
    // Write date and times
    reg_rnames("Date");
    fprintf(OUTFILE, "(date=\"%s\"\n", tmpbuf);
}// end OPEN_R_INFO_LIST

/*************************************************************************************
/ OLD_WRT_R_ITEM
/ Write a name-value pair to the INFO object.
/ ARGUMENTS:
/       newname - name of data subobject (character)
**************************************************************************************/
void wrt_r_info_item(char *newname, char *value){

	// Register (save) the name of the item
    reg_rnames(newname);

    // Write the VALUE of the info entry.  If nnames (level) == 0, this is
    // the first item at this level, and we omit the comma.
    if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
    fprintf(OUTFILE, "%s=\"%s\"\n", newname, value);

} // End OLD_WRT_R_ITEM

/*************************************************************************************
/ CLOSE_R_INFO_LIST
/ Write the terminal name-value pair to the INFO object and close the object.
/ ARGUMENTS:
/       newname - name of data subobject (character)
/       value - corresponding data (character)
**************************************************************************************/
void close_r_info_list(char *newname, char *value){

    int     i;

    // Register (save) the name of the item
    reg_rnames(newname);

    // Write the VALUE of the info entry.  If nnames (level)== 0, this is
    // the first item at this level, and we omit the comma.
    if (nnames[level] > 0) {
        fprintf(OUTFILE, "%s", ",");
    }
    fprintf(OUTFILE, "%s=\"%s\"", newname, value);

    // Write the NAMES of the information items
    fprintf(OUTFILE, "),\n.Names = c(");
    for (i=0; i < nnames[level]; i++) {
        fprintf(OUTFILE, "\"%s\",", names[level][i]);
    }
    fprintf(OUTFILE, "\"%s\"))\n\n", names[level][i]);

    // Reset level since this is done.
    level--;

} // End CLOSE_R_INFO_LIST

/********************************************************************************************************
/ OPEN_R_VECTOR
/
/  Initialize a vector object
/  Argument
/       newname - name of the vector object (character)
********************************************************************************************************/
void open_r_vector(char *newname){

    // Register name of vector
    reg_rnames(newname);

	// If nnames (level) == 0, this is the first item at this level,
    // and we omit the comma.
	if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
	// Write output to start the vector subobject.
	fprintf(OUTFILE, "%s=structure(c\n(", newname);
    level++;
} // End OPEN_R_VECTOR

/******************************************************************************************************
/ WRT_R_ITEM
/
/   Write one element of a vector.
/   The element must have a name.
/   ARGUMENTS:
/       newname - name of the data item (character)
/       spec - the type of the item: one of {%s,%d,%f} if the item is char, integer, float
/              following the printf convention.
/              optionally flags "NA" or "LAST" if missing value or last element of vector.
/       x - the datum itself unless NA
*******************************************************************************************************/
void wrt_r_item(char *newname, char *spec, ...){

    int i, last = FALSE;
    va_list ap;  // Special pointer type for variable argument list.

    // Register (save) the name of the item.  This is done first so
    // that REG_RNAMES can initialize this level's name count.
    reg_rnames(newname);

    // If this isn't the first item at this level, prefix it with a comma.
    if (nnames[level] > 0) {
        fprintf(OUTFILE, "%s", ",");
    }

    va_start (ap, spec);         // Initialize the argument list.  Var args begin after spec.

    if (strstr(spec, "LAST") != NULL) {
        last = TRUE;
    }
    if (strstr(spec, "NA") != NULL){
        fprintf(OUTFILE, "%s", nachar);
    }
    else if (strstr(spec, "%s") != NULL) {
        fprintf(OUTFILE, "\"%s\"", va_arg(ap, char *));
    }
    else if (strstr(spec, "%d") != NULL){
        fprintf(OUTFILE, "%d", va_arg(ap, int));
    }
    else if (strstr(spec, "%f") != NULL){
        fprintf(OUTFILE, float_fmt, va_arg(ap, double));
    }
    else {
        printf("%s", "Error in type specification calling WRT_R_ITEM\n");
        exit(4);
    }

    va_end (ap);                  // Clean up argument list.

    if (last) {
        // Write the NAMES of the information items
        fprintf(OUTFILE, "%s%s%s", "),", "\n.Names = c", "(");
        for(i=0; i< nnames[level]; i++){
            fprintf(OUTFILE, "\"%s\",", names[level][i]);
        }
        fprintf(OUTFILE, "\"%s\"))\n\n", names[level][i]);

        // Reset level since this is done.
        level--;

    } //End if last

} // End WRT_R_ITEM

/**********************************************************************************************
/ WRT_R_MATRIX
/
/  Write a matrix subobject to the S data object
/  Note:  No corresponding INIT subroutine is necessary for a matrix!
/
/  ARGUMENTS
/   newname : name of the matrix
/   spec    : one of {%d,%f} indicating integer or real values
/             optional flags NA, in which case a missing-value vector
/             follows the matrix argument.  If Rownames or ids or
/             column names or ids are supplied, flags Rn or Ri, Cn or Ci
/             must be used to read them.
/   nrow    : number of rows
/   ncol    : number of columns
/   x       : matrix of integer or real values
/   na      : matrix of logical values the same shape and size as x
/   rownames: array of row names (character)
/   colnames: array of column names
/   rowids  : array of row names as integers (e.g., years)
/   colids  : array of integer column identifiers
/
/   NOTE:  the NA matrix, and all row and column identifiers are optional
/
/***********************************************************************************************/
void wrt_r_matrix(char *newname, char *spec, int nrow, int ncol, ... ){

    int **isna=NULL, *colids=NULL, *rowids=NULL;
    char **cname=NULL, **rname=NULL;

    r_name xtype = "none";
    int **ix = NULL;
    double **x = NULL;

    int i,j;

    // Read arguments
    va_list ap;             // Special pointer type for variable argument list.
    va_start (ap, ncol);    // Initialize the argument list.  Var args begin after spec.

    // Register (store) name of matrix
    reg_rnames(newname);


    if (strstr(spec, "%d") != NULL) {
        strcpy(xtype, "integer");
        ix = va_arg(ap, int **);
    }
    else if (strstr(spec, "%f") != NULL) {
        strcpy(xtype, "real");
        x = va_arg (ap, double **);
    } // Integer or float

    if (strstr(spec, "NA") != NULL) { // There's a NA mask
        isna = va_arg (ap, int **);
    }
    if (strstr(spec, "Rn") != NULL) { // There are row names
        rname = va_arg (ap, char **);
    }
    if (strstr(spec, "Cn") != NULL) { // There are col names
        cname = va_arg (ap, char **);
    }
    if (strstr(spec, "Ri") != NULL) { // There are row ids
        rowids = va_arg (ap, int *);
    }
    if (strstr(spec, "Ci") != NULL) { // There are col ids
        colids = va_arg (ap, int *);
    }
    va_end (ap);                  // Clean up argument list.

    if (!strcmp(xtype, "none")) {
        printf("Error: no data supplied to WRT_R_MATRIX for object name %s \n", newname);
        exit(5);
    }

    // Check availability & compatibility of missing-value mask

    if (isna != NULL) {
        for (i=0; i<nrow; i++) {
            for (j=0; j<ncol; j++) {
                if ((isna[i][j] != 0) && (isna[i][j] != 1)) {
                    printf("Error: missing-value mask contains illegal values or does not %s \n",
                           "match the size of the data.");
                    printf("I is %d and J is %d and the value is %d\n", i,j,isna[i][j]);
                    exit(6);
                }
            }// End for j
        }// End for i
    }// End if napresent

    // Write output to start the matrix
	// If nnames (level) == 0, this is the first item at this level,
    // and we omit the comma.
	if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
    fprintf(OUTFILE, "%s=structure(c(\n", newname);

    // Write the data
    for (i=0; i < nrow; i++) {
        for (j=0; j < ncol; j++) {
            if (isna != NULL && isna[i][j]) {
                fprintf(OUTFILE, "%s", nachar);
            }
            else {
                if (!strcmp(xtype, "real")) {
                    fprintf(OUTFILE, float_fmt, x[i][j]);
                }
                else {
                    fprintf(OUTFILE, "%d", ix[i][j]);
                }
            }// Print value or nachar
            if (i < nrow-1 || j < ncol-1) {
                // not the last matrix value
                fprintf(OUTFILE, ",");
            }
            else {
                // It's the last matrix value
                fprintf(OUTFILE, "),");
            }
        } // End for columns
    } // End for rows

    // Write the dimension information:
    fprintf(OUTFILE, "\n.Dim = c(%d,%d),", nrow, ncol);

    // Write heading for the dimension names:
    fprintf(OUTFILE, ".Dimnames = list(");

    // Write the row names
    if (rname != NULL) {
        fprintf(OUTFILE, "c(");
        for (i=0; i < nrow-1; i++) {
            fprintf(OUTFILE, "\"%s\",", rname[i]);
        }
        fprintf(OUTFILE, "\"%s\"),", rname[i]);
    }
    else if (rowids != NULL){
        fprintf(OUTFILE, "c(");
        for (i=0; i < nrow-1; i++) {
            fprintf(OUTFILE, "\"%d\",", rowids[i]);
        }
        fprintf(OUTFILE, "\"%d\"),", rowids[i]);
    }
    else {
        fprintf(OUTFILE, "NULL,");
    } // End if wrtrownames

    // Write the column names
    if (cname != NULL) {
        fprintf(OUTFILE, "c(");
        for (i=0; i < ncol-1; i++) {
            fprintf(OUTFILE, "\"%s\",", cname[i]);
        }
        fprintf(OUTFILE, "\"%s\")))\n", cname[i]);
    }
    else if (colids != NULL) {
        fprintf(OUTFILE, "c(");
        for (i=0; i < ncol-1; i++) {
            fprintf(OUTFILE, "\"%d\",", colids[i]);
        }
        fprintf(OUTFILE, "\"%d\")))\n", colids[i]);
    }
    else{
        fprintf(OUTFILE, "NULL))\n");
    } // End if wrtrowcolnames

	// Append a blank line for consistency in layout.
	fprintf(OUTFILE, "\n");

}// END WRT_R_MATRIX

/*****************************************************************************************************
/ OPEN_R_DF
/
/  Initialize a data frame
/  ARGUMENT:
/       newname - name of data frame
*****************************************************************************************************/
void open_r_df(char *newname){

    reg_rnames(newname);     // Register name of data frame

	// If nnames (level) == 0, this is the first item at this level,
    // and we omit the comma.
	if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
    // Write output to start the data frame subobject
	fprintf(OUTFILE, "%s=structure(list\n", newname);

    level++;                 // We are up one level
    dflen = 0;               // Initialize number of rows in DF
    
} // End OPEN_R_DF

/*****************************************************************************************************
/ WRT_R_DF_COL
/
/  Write a numeric column to a data frame.
/  ARGUMENTS:
/       newname - name to use for this column of the data frame.
/       nrow - length of column.
/       x    - vector of real values to write to DF column.
/       spec - one of {%d,%f} indicating integer or real values
/              optional flag NA, in which case a missing-value vector
/              follows the matrix argument.  If Rownames or ids or
/              Rn or Ri must be used to read them.
/       na   - optional vector of logical values the same length as x.
/              where TRUE, value in x is missing.
/       rownames - optional vector of character strings
/       rowids - optional array of integers
*****************************************************************************************************/
void wrt_r_df_col(char *newname, char *spec, int nrow, ...){

    int *isna=NULL, *rowids=NULL;
    char **rname=NULL;

    r_name xtype = "none";
    int *ix = NULL;
    double *x = NULL;
    char **ax = NULL;

    int i, last = FALSE;

    // Read arguments
    va_list ap;             // Special pointer type for variable argument list.
    va_start (ap, nrow);    // Initialize the argument list.  Var args begin after nrow.

    // Register (store) name of DF
    reg_rnames(newname);

    if (strstr(spec, "LAST") != NULL) {
        last = TRUE;
    }
    if (strstr(spec, "%d") != NULL) {
        ix = va_arg(ap, int *);
        strcpy(xtype, "integer");
    }
    else if (strstr(spec, "%f") != NULL) {
        x = va_arg (ap, double *);
        strcpy(xtype, "real");
    }
    else {
        ax = va_arg (ap, char **);
        strcpy(xtype, "char");
    }
    if (strstr(spec, "NA") != NULL) {
        isna = va_arg (ap, int *);
    }
    if (last) { // Check for row names or ids.
        if (strstr(spec, "Rn") != NULL) {
            rname = va_arg (ap, char **);
        }
        if (strstr(spec, "Ri") != NULL) {
            rowids = va_arg (ap, int *);
        }
    }
    va_end (ap);                  // Clean up argument list.

    if (!strcmp(xtype, "none")) {
        printf("Error: no data supplied to WRT_R_DF_COL for object name %s \n", newname);
        exit(5);
    }

    // Check availability & compatibility of missing-value mask
    if (isna != NULL) {
        for (i=0; i<nrow; i++) {
            if ((isna[i] != 0) && (isna[i] != 1)) {
                printf("Error: missing-value mask contains illegal values or does not %s \n",
                       "match the length of the column.");
                exit(8);
                }
        }// End for i
    }// End if napresent

    // Store column length if this is the first col; otherwise check against first col.
    if (nnames[level] == 0) {
        dflen = nrow;
    }
    else if (nrow != dflen) {
        printf("Error:  Column lengths do not match in S_DF_WRT.\n",
               "Length of column 1 is %d and current length is %d\n",
               dflen, nrow);
        exit(9);
    }

    // If first column in the d.f., start with left paren; otherwise, comma.
    if (nnames[level] == 0) {
        fprintf(OUTFILE, "(");
    }
    else {
        fprintf(OUTFILE, "\n,");
    }

    // Initialize the column:
    fprintf(OUTFILE, "%s=c(", newname);

    // Write the data
    for (i=0; i < nrow; i++) {
        if (isna != NULL && isna[i]) {
            fprintf(OUTFILE, "%s", nachar);
        }
        else {
            if (!strcmp(xtype, "real")) {
                fprintf(OUTFILE, float_fmt, x[i]);
            }
            else if (!strcmp(xtype, "integer")) {
                fprintf(OUTFILE, "%d", ix[i]);
            }
            else {
                fprintf(OUTFILE, "\"%s\"", ax[i]);
            }
        }// End print a value or nachar
        if (i < nrow-1) {
            // not the last column value
            fprintf(OUTFILE, ",");
        }
        else {
            // It's the last column value
            fprintf(OUTFILE, ")");
        }
    } // End for rows

    // For the last column only
    if (last) {
        fprintf(OUTFILE, "),\n.Names = c(");

        // Write the column names
        for (i=0; i < nnames[level]; i++) {
            fprintf(OUTFILE, "\"%s\",", names[level][i]);
        }
        fprintf(OUTFILE, "\"%s\"),\n", names[level][i]);

        // Write the row names
        if (rname != NULL) {
            fprintf(OUTFILE, "row.names = c(");
            for (i=0; i < nrow-1; i++) {
                fprintf(OUTFILE, "\"%s\",", rname[i]);
            }
            fprintf(OUTFILE, "\"%s\"),\n", rname[i]);
        }
        else if (rowids != NULL) {
            fprintf(OUTFILE, "row.names = c(");
            for (i=0; i < nrow-1; i++) {
                fprintf(OUTFILE, "\"%d\",", rowids[i]);
            }
            fprintf(OUTFILE, "\"%d\"),\n", rowids[i]);
		}
		else {
			fprintf(OUTFILE, "row.names = c(NA,");
            fprintf(OUTFILE, "%d),\n", dflen);
        } // End if wrtrownames

        // Write the closing information:
        fprintf(OUTFILE, "class = \"data.frame\")\n\n");
        level--;
    } // End if last

 } // End WRT_R_DF_COL

/*****************************************************************************************************
/ OPEN_R_LIST
/
/  Initilize a LIST object
/
*****************************************************************************************************/
void open_r_list(char *newname){


    // Register name of list
    reg_rnames(newname);

	// If nnames (level) == 0, this is the first item at this level,
    // and we omit the comma.
	if (nnames[level] > 0) {
        fprintf(OUTFILE, ",");
    }
	// Write output to start the list subobject
    fprintf(OUTFILE, "%s=structure(list(\n", newname);

	// Insert blank line for consistency in layout.
	fprintf(OUTFILE, "\n");
    
	level++;

}// End OPEN_R_LIST

/*****************************************************************************************************
/ CLOSE_R_LIST
/
/  Write component names to finalize the list
/
*****************************************************************************************************/
void close_r_list(void){
    int i;

    // Write output to close the vector subobject:
    fprintf(OUTFILE, "),.Names = c(");

    // Write the names of the components of the list
    for (i=0; i < nnames[level]; i++) {
        fprintf(OUTFILE, "\"%s\",", names[level][i]);
    }
    fprintf(OUTFILE, "\"%s\"))\n\n", names[level][i]);
    level--;
}// End CLOSE_R_LIST

/*****************************************************************************************************
/ CLOSE_R_FILE
/
/  Write component names to finalize the object and close the file
*****************************************************************************************************/
void close_r_file(void){

    int i;

    fprintf(OUTFILE, "),.Names=c(");
    for (i=0; i<nnames[0]; i++) {
        fprintf(OUTFILE, "\"%s\",", names[0][i]);
    }
    fprintf(OUTFILE, "\"%s\"))\n", names[0][i]);
    fclose(OUTFILE);
    level--;
}// End CLOSE_R_FILE

// END MODULE C2R
