#' Prior plotting
#'
#' Plots priors for free model parameters in a ctStanModel.
#' 
#' @param x ctStanModel object as generated by \code{\link{ctModel}} with type='stanct' or 'standt'.
#' @param rows vector of integers denoting which rows of ctstanmodel$pars to plot priors for. 
#' Character string 'all' plots all rows with parameters to be estimated.
#' @param wait If true, user is prompted to continue before plotting next graph.
#' @param nsamples Numeric. Higher values increase fidelity (smoothness / accuracy) of density plots, at cost of speed.
#' @param rawpopsd Either 'marginalise' to sample from the specified (in the ctstanmodel) 
#' prior distribution for the raw population standard deviation, or a numeric value to use for the raw population standard deviation
#' for all subject level prior plots - the plots in dotted blue or red.
#' @param inddifdevs numeric vector of length 2, setting the means for the individual differences distributions.
#' @param ... not used.
#' @details Plotted in black is the prior for the population mean. In red and blue are the subject level priors that result
#' given that the population mean is estimated as 1 std deviation above the mean of the prior, or 1 std deviation below. 
#' The distributions around these two points are then obtained by marginalising over the prior for the raw population std deviation - 
#' so the red and blue distributions do not represent any specific subject level prior, but rather characterise the general amount
#' and shape of possible subject level priors at the specific points of the population mean prior.
#' @method plot ctStanModel
#' @export
#' @examples
#' model <- ctModel(type='omx', Tpoints=50,
#' n.latent=2, n.manifest=1, 
#' manifestNames='sunspots', 
#' latentNames=c('ss_level', 'ss_velocity'),
#' LAMBDA=matrix(c( 1, 'ma1' ), nrow=1, ncol=2),
#' DRIFT=matrix(c(0, 1,   'a21', 'a22'), nrow=2, ncol=2, byrow=TRUE),
#' MANIFESTMEANS=matrix(c('m1'), nrow=1, ncol=1),
#' # MANIFESTVAR=matrix(0, nrow=1, ncol=1),
#' CINT=matrix(c(0, 0), nrow=2, ncol=1),
#' DIFFUSION=matrix(c(
#'   0, 0,
#'   0, "diffusion"), ncol=2, nrow=2, byrow=TRUE))
#' 
#' stanmodel=ctStanModel(model)
#' plot(stanmodel,rows=8)

plot.ctStanModel<-function(x,rows='all',wait=FALSE,nsamples=1e6, rawpopsd='marginalise',inddifdevs=c(-1,1),...){
  if(class(x)!='ctStanModel') stop('not a ctStanModel object!')
  m<-x$pars
  highmean=inddifdevs[2]
  lowmean= inddifdevs[1]
  if(rows=='all') rows<-1:nrow(m)
  for(rowi in rows){
    if(is.na(m$value[rowi])){
      #rawpopsd
      if(rawpopsd[1]=='marginalise'){
        rawpopsdbase<-  stats::rnorm(nsamples)
        if(!is.na(x$rawpopsdbaselowerbound)) rawpopsdbase <- rawpopsdbase[rawpopsdbase>x$rawpopsdbaselowerbound]
        sdscale <- m$sdscale[rowi]
        sdtform <- gsub('.*', '*',x$rawpopsdtransform,fixed=TRUE)
        rawpopsdprior<-eval(parse(text=sdtform)) * sdscale

      } else if(is.na(as.numeric(rawpopsd))) stop('rawpopsd argument is ill specified!') else {
        rawpopsdprior <- rep(rawpopsd,nsamples)
      }
      denslist<-list()
      #mean

      rawpopmeans=stats::rnorm(length(rawpopsdprior))
      # xmean=eval(parse(text=paste0(m$transform[rowi])))
      denslist[[1]]=tform(rawpopmeans,m$transform[rowi], m$multiplier[rowi], m$meanscale[rowi], m$offset[rowi],m$inneroffset[rowi])
      leg <- c('Pop. mean prior')
      colvec <- c(1)
      
      if(m$indvarying[rowi]){
        
        if(inddifdevs[1]=='marginalise'){
          param=stats::rnorm(length(rawpopsdprior),rawpopmeans,rawpopsdprior)
          denslist[[2]]=tform(param,m$transform[rowi], m$multiplier[rowi], m$meanscale[rowi], m$offset[rowi],m$inneroffset[rowi])
          leg <- c('Pop. mean prior', paste0('Subject prior',lowmean))
          colvec <- c(1,2)
        }
        if(inddifdevs[1]!='marginalise'){
        #high
        param=stats::rnorm(length(rawpopsdprior),highmean,rawpopsdprior)
        denslist[[2]]=tform(param,m$transform[rowi], m$multiplier[rowi], m$meanscale[rowi], m$offset[rowi],m$inneroffset[rowi])
        
        #low
        param=stats::rnorm(length(rawpopsdprior),lowmean,rawpopsdprior)
        denslist[[3]]=tform(param,m$transform[rowi], m$multiplier[rowi], m$meanscale[rowi], m$offset[rowi],m$inneroffset[rowi])
  
        leg <- c('Pop. mean prior', paste0('Subject prior | mean = ',lowmean),paste0('Subject prior | mean = ',highmean))
        colvec <- c(1,2,4)
        }
      }

        ctDensityList(denslist,plot = TRUE, lwd=2,probs=c(.01,.99),main=m$param[rowi],
        cex=.8,cex.main=.8,cex.axis=.8,cex.lab=.8,cex.sub=.8,
        legend = leg,
        legendargs=list(cex=.8),
        colvec = colvec)

      if(wait==TRUE & rowi != utils::tail(rows,1)){
        message("Press [enter] to display next plot")
        readline()
      }
    }
  }
}
