% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tq_mutate.R
\name{tq_mutate}
\alias{tq_mutate}
\alias{tq_mutate_}
\alias{tq_mutate_fun_options}
\alias{tq_mutate_xy}
\alias{tq_mutate_xy_}
\title{Mutates quantitative data (adds new variables to existing tibble)}
\usage{
tq_mutate(data, x_fun = OHLCV, mutate_fun, ...)

tq_mutate_(data, x_fun = "OHLCV", mutate_fun, ...)

tq_mutate_xy(data, .x, .y = NULL, mutate_fun, ...)

tq_mutate_xy_(data, .x, .y = NULL, mutate_fun, ...)

tq_mutate_fun_options()
}
\arguments{
\item{data}{A \code{tibble} (tidy data frame) of data from \code{\link{tq_get}}.}

\item{x_fun}{The \code{quantmod} function that identifes columns to pass to
the mutatation function. OHLCV is \code{quantmod} terminology for
open, high, low, close, and volume. Options include c(Op, Hi, Lo, Cl, Vo, Ad,
HLC, OHLC, OHLCV).}

\item{mutate_fun}{The mutation function from either the \code{xts},
\code{quantmod}, or \code{TTR} package. Execute \code{tq_mutate_fun_options()}
to see the full list of options by package.}

\item{...}{Additional parameters passed to the appropriate mutatation
function.}

\item{.x, .y}{Column names of variables to be passed to the mutatation
function (instead of OHLC functions).}
}
\value{
Returns data in the form of a \code{tibble} object.
}
\description{
Mutates quantitative data (adds new variables to existing tibble)
}
\details{
\code{tq_mutate} is a very flexible wrapper for various \code{xts},
\code{quantmod} and \code{TTR} functions. The main advantage is the
results are returned as a \code{tibble} and the
function can be used with the \code{tidyverse}.

\code{x_fun} is one of the various \code{quantmod} Open, High, Low, Close (OHLC) functions.
The function returns a column or set of columns from \code{data} that are passed to the
\code{mutate_fun}. In Example 1 below, \code{Cl} returns the "close" price and sends
this to the mutate function, \code{periodReturn}.

\code{mutate_fun} is the function that performs the work. In Example 1, this
is \code{periodReturn}, which calculates the period returns. The \code{...}
functions are additional arguments passed to the \code{mutate_fun}. Think of
the whole operation in Example 1 as the close price, obtained by \code{x_fun = Cl},
is being sent to the \code{periodReturn} function along
with additional arguments defining how to perform the period return, which
includes \code{period = "daily"} and \code{type = "log"}.

\code{tq_mutate_xy} is designed to enable working with (1) mutatation
functions that require two primary inputs (e.g. EVWMA, VWAP, etc) and (2) data
that is not in OHLC format. Example 2 shows the first benefit in action:
using the EVWMA function that uses volume to defind the moving average period.
The two variables do not fall into a single OHLC code (i.e. CV does not exist).
The xy form gets us out of this problem. Example 3 shows the second benefit
in action: Some functions are useful to non-OHLC data, and defining .x = price
allows us to mutate WTI crude prices from daily to monthly periodicity.

\code{tq_mutate_} and \code{tq_mutate_xy_} are setup for Non-Standard
Evaluation (NSE). This enables programatically changing column names by modifying
the text representations. Example 4 shows the difference in implemenation.
Note that character strings are being passed to the variables instead of
unquoted variable names. See \code{vignette("nse")} for more information.
}
\examples{
# Load libraries
library(tidyverse)
library(tidyquant)

##### Basic Functionality

fb_stock_prices  <- tq_get("FB", get = "stock.prices")

# Example 1: Return logarithmic daily returns using periodReturn()
fb_stock_prices \%>\%
    tq_mutate(x_fun = Cl, mutate_fun = periodReturn,
                 period = "daily", type = "log")

# Example 2: Use tq_mutate_xy to use functions with two columns required
fb_stock_prices \%>\%
    tq_mutate_xy(.x = close, .y = volume, mutate_fun = EVWMA)

# Example 3: Using tq_mutate_xy to work with non-OHLC data
tq_get("DCOILWTICO", get = "economic.data") \%>\%
    tq_mutate_xy(.x = price, mutate_fun = lag.xts, k = 1, na.pad = TRUE)

# Example 4: Non-standard evaluation:
# Programming with tq_mutate_() and tq_mutate_xy_()
col_name <- "adjusted"
mutate <- c("MACD", "SMA")
tq_mutate_xy_(fb_stock_prices, .x = col_name, mutate_fun = mutate[[1]])
}
\seealso{
\code{\link{tq_transform}}
}

