#  File src/library/tools/R/build.R
#  Part of the R package, http://www.R-project.org
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  A copy of the GNU General Public License is available at
#  http://www.r-project.org/Licenses/

#### R based engine for R CMD build

### emulation of Perl Logfile.pm

newLog <- function(filename = "")
{
    con <- if (nzchar(filename)) file(filename, "wt") else 0L
    list(filename = filename, con = con, stars = "*", warnings = 0L)
}

closeLog <- function(Log) if (Log$con > 2) close(Log$con)

printLog <- function(Log, ...) {
    cat(..., sep = "")
    if (Log$con > 0L) cat(..., file = Log$con, sep = "")
}

## unused
setStars <- function(Log, stars) {Log$stars <- stars; Log}

checkingLog <- function(Log, ...)
    printLog(Log, Log$stars, " checking ", ..., " ...")

creatingLog <- function(Log, text)
    printLog(Log, Log$stars," creating ", text, " ...")

messageLog <- function(Log, ...)
{
    text <- paste(..., sep="")
##     cat(Log$stars, " ",
##         gsub("\n", paste("\n", Log$stars, " ", sep = ""), text, fixed = TRUE),
##         sep = "\n", file = Log$con)
    printLog(Log, Log$stars, " ", ..., "\n")
}
resultLog <- function(Log, text) printLog(Log, " ", text, "\n")

errorLog <- function(Log, ...)
{
    resultLog(Log, "ERROR")
    text <- paste(..., sep="")
    if (length(text) && nzchar(text)) printLog(Log, ..., "\n")
}

warningLog <- function(Log, text="")
{
    resultLog(Log, "WARNING")
    if (nzchar(text)) messageLog(Log, text)
    Log$warnings <- Log$warnings+1L
    invisible(Log)
}

noteLog <- function(Log, text="")
{
    resultLog(Log, "NOTE")
    if (nzchar(text)) messageLog(Log, text)
}

summaryLog <- function(Log)
{
    if (Log$warnings > 1)
        printLog(Log,
                 sprintf("WARNING: There were %d warnings, see\n  %s\nfor details\n",
                         Log$warnings, sQuote(Log$filename)))
    else if (Log$warnings == 1)
        printLog(Log,
                 sprintf("WARNING: There was 1 warning, see\n  %s\nfor details\n",
                         sQuote(Log$filename)))
}



### formerly Perl R::Utils::get_exclude_patterns

## Return list of file patterns excluded by R CMD build and check.
## Kept here so that we ensure that the lists are in sync, but not exported.
## Has Unix-style '/' path separators hard-coded, but that is what dir() uses.
get_exclude_patterns <- function()
    c("^\\.Rbuildignore$",
      "(^|/)\\.DS_Store$",
      "^\\.(RData|Rhistory)$",
      "~$", "\\.bak$", "\\.swp$",
      "(^|/)\\.#[^/]*$", "(^|/)#[^/]*#$",
      ## Outdated ...
      "^TITLE$", "^data/00Index$",
      "^inst/doc/00Index\\.dcf$",
      ## Autoconf
      "^config\\.(cache|log|status)$",
      "^autom4te\\.cache$",
      ## Windows dependency files
      "^src/.*\\.d$", "^src/Makedeps$",
      ## IRIX, of some vintage
      "^src/so_locations$",
      ## Sweave detrius
      "^inst/doc/Rplots\\.(ps|pdf)$"
      )


### based on Perl build script

.build_packages <- function(args = NULL)
{
    ## this requires on Windows sh make tar gzip

    WINDOWS <- .Platform$OS.type == "windows"

    Sys.umask("022") # Perl version did not have this.

    writeLinesNL <- function(text, file)
    {
        ## a version that uses NL line endings everywhere
        con <- file(file, "wb")
        on.exit(close(con))
        writeLines(text, con)
    }

    ## This version of shell_with_capture merges stdout and stderr
    ## Used to install package and build vignettes.
    shell_with_capture <- function (command, args) {
        outfile <- tempfile("xshell")
        on.exit(unlink(outfile))
        status <- system2(command, args, outfile, outfile)
        list(status = status, stdout = readLines(outfile, warn = FALSE))
    }
    ## Run silently
    Ssystem <- function(command, args = character(), ...)
        system2(command, args, stdout = NULL, stderr = NULL, ...)


    .file_test <- function(op, x)
        switch(op,
               "-f" = !is.na(isdir <- file.info(x)$isdir) & !isdir,
               "-x" = (file.access(x, 1L) == 0L),
               stop(sprintf("test '%s' is not available", op), domain = NA))

    dir.exists <- function(x) !is.na(isdir <- file.info(x)$isdir) & isdir

    do_exit <- function(status = 1L) q("no", status = status, runLast = FALSE)

    env_path <- function(...) file.path(..., fsep = .Platform$path.sep)

    parse_description_field <-
        function(desc, field, default = TRUE, logical = TRUE)
    {
        tmp <- desc[field]
        if (is.na(tmp)) default
        else if(logical)
            switch(tmp,
                   "yes"=, "Yes" =, "true" =, "True" =, "TRUE" = TRUE,
                   "no" =, "No" =, "false" =, "False" =, "FALSE" = FALSE,
                   default)
        else tmp
    }

    Usage <- function() {
        cat("Usage: R CMD build [options] pkgdirs",
            "",
            "Build R packages from package sources in the directories specified by",
            sQuote("pkgdirs"),
            "",
            "Options:",
            "  -h, --help		print short help message and exit",
            "  -v, --version		print version info and exit",
            "",
            "  --force               force removal of INDEX file",
            "  --keep-empty-dirs     do not remove empty dirs",
            "  --no-vignettes        do not rebuild package vignettes",
            "  --no-manual           do not build the PDF manual even if \\Sexprs are present",
            "  --resave-data=        re-save data files as compactly as possible:",
            '                        "no", "best", "gzip" (default)',
            "  --resave-data         same as --resave-data=best",
            "  --no-resave-data      same as --resave-data=no",
            "  --compact-vignettes   try to compact PDF files under inst/doc (using qpdf)",
            "",
            "  --binary              build pre-compiled binary packages, with option:",
            "  --install-args=       command-line args to be passed to INSTALL,",
            "                        separated by spaces",
            "",
            "Report bugs to <r-bugs@r-project.org>.", sep="\n")
    }


    add_build_stamp_to_description_file <- function(ldpath)
    {
        lines <- readLines(ldpath, warn = FALSE)
        lines <- lines[nzchar(lines)] # Remove blank lines.
        ## Do not keep previous build stamps.
        lines <- lines[!grepl("^Packaged:", lines, useBytes = TRUE)]
        ## this is an optional function, so could fail
        user <- Sys.info()["user"]
        if (user == "unknown") user <- Sys.getenv("LOGNAME")
        lines <- c(lines,
                   paste("Packaged: ",
                         format(Sys.time(), '', tz='UTC', usetz=TRUE),
                         ";", " ", user, sep = ""))
        writeLinesNL(lines, ldpath)
    }

    temp_install_pkg <- function(pkgdir, libdir) {
	dir.create(libdir, mode = "0755", showWarnings = FALSE)
        ## assume vignettes only need one arch
        if (WINDOWS) {
            cmd <- file.path(R.home("bin"), "Rcmd.exe")
            args <- c("INSTALL -l", shQuote(libdir),
                      "--no-multiarch", shQuote(pkgdir))
        } else {
            cmd <- file.path(R.home("bin"), "R")
            args <- c("CMD", "INSTALL -l", shQuote(libdir),
                      "--no-multiarch", shQuote(pkgdir))
        }
	res <- shell_with_capture(cmd, args)
	if (res$status) {
	    printLog(Log, "      -----------------------------------\n")
	    printLog(Log, paste(c(res$stdout, ""),  collapse="\n"))
	    printLog(Log, "      -----------------------------------\n")
	    unlink(libdir, recursive = TRUE)
	    printLog(Log, "ERROR: package installation failed\n")
	    do_exit(1)
	}
	TRUE
    }

    prepare_pkg <- function(pkgdir, desc, Log)
    {
        owd <- setwd(pkgdir); on.exit(setwd(owd))
        pkgname <- basename(pkgdir)
        checkingLog(Log, "DESCRIPTION meta-information")
        res <- try(.check_package_description("DESCRIPTION"))
        if (inherits(res, "try-error")) {
            resultLog(Log, "ERROR")
            messageLog(Log, "running .check_package_description failed")
        } else {
            if (any(sapply(res, length))) {
                resultLog(Log, "ERROR")
                print(res) # FIXME print to Log?
                do_exit(1L)
            } else resultLog(Log, "OK")
        }
        cleanup_pkg(pkgdir, Log)

        libdir <- tempfile("Rinst")

        pkgInstalled <- build_Rd_db(pkgdir, libdir)

        if (file.exists("INDEX")) update_Rd_index("INDEX", "man", Log)
        doc_dir <- file.path("inst", "doc")
        if ("makefile" %in% dir(doc_dir)) { # avoid case-insensitive match
            messageLog(Log, "renaming 'inst/doc/makefile' to 'inst/doc/Makefile'")
            file.rename(file.path(doc_dir, "makefile"),
                        file.path(doc_dir, "Makefile"))
        }
        if (vignettes && dir.exists(doc_dir) &&
           length(list_files_with_type(doc_dir, "vignette")) &&
            parse_description_field(desc, "BuildVignettes", TRUE)) {
            if (!pkgInstalled) {
		messageLog(Log, "installing the package to re-build vignettes")
		pkgInstalled <- temp_install_pkg(pkgdir, libdir)
	    }

            ## Good to do this in a separate process: it might die
            creatingLog(Log, "vignettes")
            R_LIBS <- Sys.getenv("R_LIBS", NA_character_)
            if (!is.na(R_LIBS)) {
                on.exit(Sys.setenv(R_LIBS = R_LIBS), add=TRUE)
                Sys.setenv(R_LIBS = env_path(libdir, R_LIBS))
            } else {
                on.exit(Sys.unsetenv("R_LIBS"), add=TRUE)
                Sys.setenv(R_LIBS = libdir)
            }
            cmd <- file.path(R.home("bin"), "Rscript")
            args <- c("--vanilla",
                      "--default-packages=", # some vignettes assume methods
                      "-e", shQuote("tools::buildVignettes(dir = '.')"))
            ## since so many people use 'R CMD' in Makefiles,
            oPATH <- Sys.getenv("PATH")
            Sys.setenv(PATH = paste(R.home("bin"), oPATH,
                                    sep = .Platform$path.sep))
            res <- shell_with_capture(cmd, args)
            Sys.setenv(PATH = oPATH)
            if (res$status) {
                resultLog(Log, "ERROR")
                printLog(Log, paste(c(res$stdout, ""),  collapse="\n"))
                do_exit(1L)
            } else {
                ## Do any of the .R files which will be generated
                ## exist in inst/doc?  If so the latter should be removed.
                sources <- basename(list_files_with_exts(doc_dir, c("r", "s", "R", "S")))
                if (length(sources)) {
                    vf <- list_files_with_type(doc_dir, "vignette")
                    td <- tempfile()
                    dir.create(td)
                    file.copy(doc_dir, td, recursive = TRUE)
                    od <- setwd(file.path(td, "doc"))
                    unlink(list_files_with_exts(".", c("r", "s", "R", "S")))
                    for(v in vf) tryCatch(utils::Stangle(v, quiet = TRUE),
                                          error = function(e) {})
                    new_sources <- basename(list_files_with_exts(".", c("r", "s", "R", "S")))
                    setwd(od)
                    dups <- sources[sources %in% new_sources]
                    if(length(dups)) {
                        warningLog(Log)
                        printLog(Log,
                                 paste(c("  Unused files in inst/doc which are pointless or misleading",
                                         "  as they will be re-created from the vignettes on installation:",
                                         paste("  ", dups),
                                         "  have been removed", ""),
                                       collapse = "\n"))
                        unlink(file.path(doc_dir, dups))
                    } else resultLog(Log, "OK")
                } else resultLog(Log, "OK")
            }
        }
        if (compact_vignettes &&
            length(pdfs <- dir(doc_dir, pattern = "\\.pdf", recursive = TRUE,
                               full.names = TRUE))
            && nzchar(Sys.which(qpdf <-Sys.getenv("R_QPDF", "qpdf")))) {
            messageLog(Log, "compacting vignettes and other PDF files")
            compactPDF(pdfs, qpdf, "")
        }
        if (pkgInstalled) {
            unlink(libdir, recursive = TRUE)

	    ## And finally, clean up again.
            cleanup_pkg(pkgdir, Log)
        }
    }

    cleanup_pkg <- function(pkgdir, Log)
    {
        owd <- setwd(pkgdir); on.exit(setwd(owd))
        pkgname <- basename(pkgdir)
        if (dir.exists("src")) {
            setwd("src")
            messageLog(Log, "cleaning src")
            if (WINDOWS) {
                if (file.exists("Makefile.win")) {
                    Ssystem(Sys.getenv("MAKE", "make"),
                            "-f Makefile.win clean")
                } else {
                    if (file.exists("Makevars.win")) {
                        makefiles <- paste()
                        makefiles <- paste("-f",
                                           shQuote(file.path(R.home("share"), "make", "clean.mk")),
                                           "-f Makevars.win")
                        Ssystem(Sys.getenv("MAKE", "make"),
                                c(makefiles, "clean"))
                    }
                    ## Also cleanup possible Unix leftovers ...
                    unlink(c(Sys.glob(c("*.o", "*.sl", "*.so", "*.dylib")),
                             paste(pkgname, c(".a", ".dll", ".def"), sep="")))
                    if (dir.exists(".libs")) unlink(".libs", recursive = TRUE)
                    if (dir.exists("_libs")) unlink("_libs", recursive = TRUE)
                }
            } else {
                makefiles <- paste("-f",
                                   shQuote(file.path(R.home("etc"),
                                                     Sys.getenv("R_ARCH"),
                                                     "Makeconf")))
                if (file.exists("Makefile")) {
                    makefiles <- paste(makefiles, "-f", "Makefile")
                    Ssystem(Sys.getenv("MAKE", "make"), c(makefiles, "clean"))
                } else {
                    if (file.exists("Makevars")) {
                        ## ensure we do have a 'clean' target.
                        makefiles <- paste(makefiles, "-f",
                                       shQuote(file.path(R.home("share"), "make", "clean.mk")),
                                           "-f Makevars")
                        Ssystem(Sys.getenv("MAKE", "make"),
                                c(makefiles, "clean"))
                    }
                    ## Also cleanup possible Windows leftovers ...
                    unlink(c(Sys.glob(c("*.o", "*.sl", "*.so", "*.dylib")),
                             paste(pkgname, c(".a", ".dll", ".def"), sep="")))
                    if (dir.exists(".libs")) unlink(".libs", recursive = TRUE)
                    if (dir.exists("_libs")) unlink("_libs", recursive = TRUE)
                }
            }
        }
        setwd(owd)
        ## It is not clear that we want to do this: INSTALL should do so.
        ## Also, certain environment variables should be set according
        ## to 'Writing R Extensions', but were not in Perl version (nor
        ## was cleanup.win used).
        if (WINDOWS) {
            if (file.exists("cleanup.win")) {
                Sys.setenv(R_PACKAGE_NAME = pkgname)
                Sys.setenv(R_PACKAGE_DIR = pkgdir)
                Sys.setenv(R_LIBRARY_DIR = dirname(pkgdir))
                messageLog(Log, "running cleanup.win")
                Ssystem("sh", "./cleanup.win")
            }
        } else if (.file_test("-x", "cleanup")) {
            Sys.setenv(R_PACKAGE_NAME = pkgname)
            Sys.setenv(R_PACKAGE_DIR = pkgdir)
            Sys.setenv(R_LIBRARY_DIR = dirname(pkgdir))
            messageLog(Log, "running cleanup")
            Ssystem("./cleanup")
        }
    }

    update_Rd_index <- function(oldindex, Rd_files, Log)
    {
        newindex <- tempfile()
        res <- try(Rdindex(Rd_files, newindex))
        if (inherits(res, "try-error")) {
            errorLog(Log, "computing Rd index failed")
            do_exit(1L)
        }
        checkingLog(Log, "whether ", sQuote(oldindex), " is up-to-date")
        if (file.exists(oldindex)) {
            ol <- readLines(oldindex, warn = FALSE) # e.g. BaM had missing final NL
            nl <- readLines(newindex)
            if (!identical(ol, nl)) {
                resultLog(Log, "NO")
               if (force) {
                    messageLog(Log, "removing ", sQuote(oldindex),
			      " as '--force' was given")
                    unlink(oldindex)
                } else {
                    messageLog(Log, "use '--force' to remove ",
			      "the existing ", sQuote(oldindex))
                    unlink(newindex)
                }
            } else {
                resultLog(Log, "OK")
                unlink(newindex)
            }
        } else {
            resultLog(Log, "NO")
            messageLog(Log, "creating new ", sQuote(oldindex))
            file.rename(newindex, oldindex)
        }
    }

    build_Rd_db <- function(pkgdir, libdir) {
    	db <- .build_Rd_db(pkgdir, stages=NULL, os=c("unix", "windows"), step=1)
    	if (!length(db)) return(FALSE)

    	# Strip the pkgdir off the names
    	names(db) <- substring(names(db), nchar(file.path(pkgdir, "man", ""))+1)

	containsSexprs <-
            which(sapply(db, function(Rd) getDynamicFlags(Rd)["\\Sexpr"]))
	if (!length(containsSexprs)) return(FALSE)

	messageLog(Log, "installing the package to process help pages")

        dir.create(libdir, mode = "0755", showWarnings = FALSE)
        savelib <- .libPaths()
        .libPaths(libdir)
        on.exit(.libPaths(savelib), add = TRUE)

        temp_install_pkg(pkgdir, libdir)

	containsBuildSexprs <-
            which(sapply(db, function(Rd) getDynamicFlags(Rd)["build"]))

	if (length(containsBuildSexprs)) {
	    for (i in containsBuildSexprs)
		db[[i]] <- prepare_Rd(db[[i]], stages = "build",
                                      stage2 = FALSE, stage3 = FALSE)
	    messageLog(Log, "saving partial Rd database")
	    partial <- db[containsBuildSexprs]
	    dir.create("build", showWarnings = FALSE)
	    saveRDS(partial, file.path("build", "partial.rdb"))
	}
	needRefman <- manual && any(sapply(db, function(Rd) any(getDynamicFlags(Rd)[c("install", "render")])))
	if (needRefman) {
	    messageLog(Log, "building the PDF package manual")
	    dir.create("build", showWarnings = FALSE)
	    refman <- file.path(pkgdir, "build",
                                paste(basename(pkgdir), ".pdf", sep = ""))
	    ..Rd2dvi(c("--pdf", "--force", "--no-preview",
	               paste("--output=", refman, sep=""),
	               pkgdir), quit = FALSE)
        }
	return(TRUE)
    }

    ## These also fix up missing final NL
    fix_nonLF_in_source_files <- function(pkgname, Log)
    {
        if (!dir.exists(file.path(pkgname, "src"))) return()
        src_files <- dir(file.path(pkgname, "src"),
                         pattern = "\\.([cfh]|cc|cpp)$",
                         full.names=TRUE, recursive = TRUE)
        for (ff in src_files) {
            lines <- readLines(ff, warn = FALSE)
            writeLinesNL(lines, ff)
        }
    }

    fix_nonLF_in_make_files <- function(pkgname, Log)
    {
        if (!dir.exists(file.path(pkgname, "src"))) return()
         for (f in c("Makefile", "Makefile.in", "Makefile.win",
                     "Makevars", "Makevars.in", "Makevars.win")) {
             if (!file.exists(ff <- file.path(pkgname, "src", f))) next
             lines <- readLines(ff, warn = FALSE)
             writeLinesNL(lines, ff)
         }
     }

    find_empty_dirs <- function(d)
    {
        ## dir(recursive = TRUE) did not include directories, so
        ## we needed to do this recursively
        files <- dir(d, all.files = TRUE, full.names = TRUE)
        isdir <- file.info(files)$isdir
        for (dd in files[isdir]) {
            if (grepl("/\\.+$", dd)) next
            find_empty_dirs(dd)
        }
        ## allow per-package override
        keep_empty1 <- parse_description_field(desc, "BuildKeepEmpty",
                                               keep_empty)
        if (!keep_empty1) # might have removed a dir
            files <- dir(d, all.files = TRUE, full.names = TRUE)
        if (length(files) <= 2L) { # always has ., ..
            if (keep_empty1) {
                printLog(Log, "WARNING: directory ", sQuote(d), " is empty\n")
            } else {
                unlink(d, recursive = TRUE)
                printLog(Log, "Removed empty directory ", sQuote(d), "\n")
            }
        }
    }

    fixup_R_dep <- function(pkgname, ver="2.10")
    {
        desc <- .read_description(file.path(pkgname, "DESCRIPTION"))
        Rdeps <- .split_description(desc)$Rdepends2
        for(dep in Rdeps) {
            if(dep$op != '>=') next
            if(dep$version >= package_version(ver)) return()
        }
        on.exit(Sys.setlocale("LC_CTYPE", Sys.getlocale("LC_CTYPE")))
        Sys.setlocale("LC_CTYPE", "C")
        flatten <- function(x) {
            if(length(x) == 3L)
                paste(x$name, " (", x$op, " ", x$version, ")", sep = "")
            else x[[1L]]
        }
        deps <- desc["Depends"]
        desc["Depends"] <- if(!is.na(deps)) {
            deps <- .split_dependencies(deps)
            deps <- deps[names(deps) != "R"] # could be more than one
            paste(c(sprintf("R (>= %s)", ver), sapply(deps, flatten)),
                  collapse = ", ")
        } else sprintf("R (>= %s)", ver)
        write.dcf(t(as.matrix(desc)), file.path(pkgname, "DESCRIPTION"))
        printLog(Log,
                 "  NB: this package now depends on R (>= ", ver, ")\n")
    }

    resave_data_rda <- function(pkgname, resave_data)
    {
        if (resave_data == "no") return()
        ddir <- file.path(pkgname, "data")
        if(resave_data == "best") {
            files <- Sys.glob(c(file.path(ddir, "*.rda"),
                                file.path(ddir, "*.RData"),
                                file.path(pkgname, "R", "sysdata.rda")))
            messageLog(Log, "re-saving image files")
            resaveRdaFiles(files)
            rdas <- checkRdaFiles(files)
            if(any(rdas$compress %in% c("bzip2", "xz")))
                fixup_R_dep(pkgname, "2.10")
        } else {
            rdas <- checkRdaFiles(ddir)
            if(nrow(rdas)) {
                update <- with(rdas, ASCII | compress == "none" | version < 2)
                if(any(update)) {
                    messageLog(Log, "re-saving image files")
                    resaveRdaFiles(row.names(rdas)[update], "gzip")
                }
            }
            if(file.exists(f <- file.path(pkgname, "R", "sysdata.rda"))) {
                rdas <- checkRdaFiles(f)
                update <- with(rdas, ASCII | compress == "none" | version < 2)
                if(any(update)) {
                    messageLog(Log, "re-saving sysdata.rda")
                    resaveRdaFiles(f, "gzip")
                }
            }
        }
    }


    resave_data_others <- function(pkgname, resave_data)
    {
        if (resave_data == "no") return()
        ddir <- file.path(pkgname, "data")
        dataFiles <- grep("\\.(rda|RData)$",
                          list_files_with_type(ddir, "data"),
                          invert = TRUE, value = TRUE)
        if (!length(dataFiles)) return()
        Rs <- grep("\\.[Rr]$", dataFiles, value = TRUE)
        if (length(Rs)) { # these might use .txt etc
            messageLog(Log, "re-saving .R files as .rda")
            ## ensure utils is visible
            library("utils")
            lapply(Rs, function(x){
                envir <- new.env(hash = TRUE)
                sys.source(x, chdir = TRUE, envir = envir)
                save(list = ls(envir, all.names = TRUE),
                     file = sub("\\.[Rr]$", ".rda", x),
                     compress = TRUE, compression_level = 9,
                     envir = envir)
                unlink(x)
            })
            printLog(Log,
                     "  NB: *.R converted to .rda: other files may need to be removed\n")
        }
        tabs <- grep("\\.(CSV|csv|TXT|tab|txt)$", dataFiles, value = TRUE)
        if (length(tabs)) {
            messageLog(Log, "re-saving tabular files")
            if (resave_data == "gzip") {
                lapply(tabs, function(nm) {
                    ## DiceDesign/data/greenwood.table.txt is missing NL
                    x <- readLines(nm, warn = FALSE)
                    con <- gzfile(paste(nm, "gz", sep = "."), "wb")
                    writeLines(x, con)
                    close(con)
                    unlink(nm)
                })
            } else {
                OK <- TRUE
                lapply(tabs, function(nm) {
                    x <- readLines(nm, warn = FALSE)
                    nm3 <- paste(nm, c("gz", "bz2", "xz"), sep = ".")
                    con <- gzfile(nm3[1L], "wb", compression=9); writeLines(x, con); close(con)
                    con <- bzfile(nm3[2L], "wb", compression=9); writeLines(x, con); close(con)
                    con <- xzfile(nm3[3L], "wb", compression=9); writeLines(x, con); close(con)
                    sizes <- file.info(nm3)$size * c(0.9, 1, 1)
                    ind <- which.min(sizes)
                    if(ind > 1) OK <<- FALSE
                    unlink(c(nm, nm3[-ind]))
                })
                if (!OK) fixup_R_dep(pkgname, "2.10")
            }
        }
    }

    force <- FALSE
    vignettes <- TRUE
    binary <- FALSE
    manual <- TRUE  # Install the manual if Rds contain \Sexprs
    INSTALL_opts <- character()
    pkgs <- character()
    options(showErrorCalls = FALSE, warn = 1)

    ## read in ~/.R/build.Renviron[.rarch]
    rarch <- .Platform$r_arch
    if (nzchar(rarch) &&
        file.exists(Renv <- paste("~/.R/build.Renviron", rarch, sep = ".")))
        readRenviron(Renv)
    else if (file.exists(Renv <- "~/.R/build.Renviron")) readRenviron(Renv)

    ## Configurable variables.
    compact_vignettes <-
        config_val_to_logical(Sys.getenv("_R_BUILD_COMPACT_VIGNETTES_",
                                         "FALSE"))
    resave_data <-
        Sys.getenv("_R_BUILD_RESAVE_DATA_", "gzip")

    keep_empty <-
        config_val_to_logical(Sys.getenv("_R_BUILD_KEEP_EMPTY_DIRS_", "FALSE"))

    if (is.null(args)) {
        args <- commandArgs(TRUE)
        ## it seems that splits on spaces, so try harder.
        args <- paste(args, collapse=" ")
        args <- strsplit(args,'nextArg', fixed = TRUE)[[1L]][-1L]
    }

    while(length(args)) {
        a <- args[1L]
        if (a %in% c("-h", "--help")) {
            Usage()
            do_exit(0L)
        }
        else if (a %in% c("-v", "--version")) {
            cat("R add-on package builder: ",
                R.version[["major"]], ".",  R.version[["minor"]],
                " (r", R.version[["svn rev"]], ")\n", sep = "")
            cat("",
                "Copyright (C) 1997-2011 The R Core Development Team.",
                "This is free software; see the GNU General Public License version 2",
                "or later for copying conditions.  There is NO warranty.",
                sep="\n")
            do_exit(0L)
        } else if (a == "--force") {
            force <- TRUE
        } else if (a == "--keep-empty-dirs") {
            keep_empty <- TRUE
        } else if (a == "--no-vignettes") {
            vignettes <- FALSE
        } else if (a == "--binary") {
            binary <- TRUE
            message("--binary is deprecated")
        } else if (substr(a, 1, 15) == "--install-args=") {
            INSTALL_opts <- c(INSTALL_opts, substr(a, 16, 1000))
        } else if (a == "--resave-data") {
            resave_data <- "best"
        } else if (a == "--no-resave-data") {
            resave_data <- "no"
        } else if (substr(a, 1, 14) == "--resave-data=") {
            resave_data <- substr(a, 15, 1000)
        } else if (WINDOWS && a == "--auto-zip") {
            warning("use of '--auto-zip' is defunct")
        } else if (a == "--use-zip-data") {
            warning("use of '--use-zip-data' is defunct")
        } else if (a == "--no-docs") {
            warning("use of '--no-docs' is deprecated: use '--install-args' instead")
            INSTALL_opts <- c(INSTALL_opts, "--no-docs")
        } else if (a == "--no-manual") {
            manual <- FALSE
        } else if (a == "--compact-vignettes") {
            compact_vignettes <- TRUE
        } else if (substr(a, 1, 1) == "-") {
            message("Warning: unknown option ", sQuote(a))
        } else pkgs <- c(pkgs, a)
        args <- args[-1L]
    }
    if (!binary && length(INSTALL_opts))
        message("** Options ",
                sQuote(paste(INSTALL_opts, collapse=" ")),
                " are only for '--binary'  and will be ignored")

    Sys.unsetenv("R_DEFAULT_PACKAGES")

    startdir <- getwd()
    if (is.null(startdir))
        stop("current working directory cannot be ascertained")
    R_platform <- Sys.getenv("R_PLATFORM", "unknown-binary")
    libdir <- tempfile("Rinst")

    if (WINDOWS) {
        ## Some people have *assumed* that R_HOME uses / in Makefiles
        ## Spaces in paths might still cause trouble.
        rhome <- chartr("\\", "/", R.home())
        Sys.setenv(R_HOME = rhome)
    }

    for(pkg in pkgs) {
        Log <- newLog() # if not stdin; on.exit(closeLog(Log))
        ## remove any trailing /, for Windows' sake
        pkg <- sub("/$", "", pkg)
        ## 'Older versions used $pkg as absolute or relative to $startdir.
        ## This does not easily work if $pkg is a symbolic link.
        ## Hence, we now convert to absolute paths.'
        setwd(startdir)
        res <- try(setwd(pkg), silent = TRUE)
        if (inherits(res, "try-error")) {
            errorLog(Log, "cannot change to directory ", sQuote(pkg))
            do_exit(1L)
        }
        pkgdir <- getwd()
        pkgname <- basename(pkgdir)
        checkingLog(Log, "for file ", sQuote(file.path(pkg, "DESCRIPTION")))
        f <- file.path(pkgdir, "DESCRIPTION")
        if (file.exists(f)) {
            desc <- try(read.dcf(f))
            if (inherits(desc, "try-error") || !length(desc)) {
                resultLog(Log, "EXISTS but not correct format")
                do_exit(1L)
            }
            desc <- desc[1L, ]
            resultLog(Log, "OK")
        } else {
            resultLog(Log, "NO")
            do_exit(1L)
        }
        intname <- desc["Package"]
        ## make a copy, cd to parent of copy
        setwd(dirname(pkgdir))
        filename <- paste(intname, "_", desc["Version"], ".tar", sep="")
        filepath <- file.path(startdir, filename)
        Tdir <- tempfile("Rbuild")
        dir.create(Tdir, mode = "0755")
        ## on Windows we will not be able to delete read-only files in .svn
        if (!file.copy(pkgname, Tdir, recursive = TRUE,
                       copy.mode = !WINDOWS)) {
            errorLog(Log, "copying to build directory failed")
            do_exit(1L)
        }
        setwd(Tdir)

        ## Now correct the package name (PR#9266)
        if (pkgname != intname) {
            if (!file.rename(pkgname, intname)) {
                message("Error: cannot rename directory to ", sQuote(intname))
                do_exit(1L)
            }
            pkgname <- intname
        }

        ## prepare the copy
        messageLog(Log, "preparing ", sQuote(pkgname), ":")
        prepare_pkg(normalizePath(pkgname, "/"), desc, Log);
        owd <- setwd(pkgname)
        ## remove exclude files
        allfiles <- dir(".", all.files = TRUE, recursive = TRUE,
                        full.names = TRUE, include.dirs = TRUE)
        allfiles <- substring(allfiles, 3L)  # drop './'
        bases <- basename(allfiles)
        exclude <- rep(FALSE, length(allfiles))
        ignore <- get_exclude_patterns()
        ## handle .Rbuildignore:
        ## 'These patterns should be Perl regexps, one per line,
        ##  to be matched against the file names relative to
        ##  the top-level source directory.'
        ignore_file <- file.path(pkgdir, ".Rbuildignore")
        if (file.exists(ignore_file))
            ignore <- c(ignore, readLines(ignore_file, warn = FALSE))
        for(e in ignore[nzchar(ignore)])
            exclude <- exclude | grepl(e, allfiles, perl = TRUE,
                                       ignore.case = WINDOWS)

        isdir <- file_test("-d", allfiles)
        ## old (pre-2.10.0) dirnames
        exclude <- exclude | (isdir & (bases %in%
                                       c("check", "chm", .vc_dir_names)))
        exclude <- exclude | (isdir & grepl("([Oo]ld|\\.Rcheck)$", bases))
        ## FIXME: GNU make uses GNUmakefile (note capitalization)
        exclude <- exclude | bases %in% c("Read-and-delete-me", "GNUMakefile")
        ## Mac resource forks
        exclude <- exclude | grepl("^\\._", bases)
	## Windows DLL resource file
        exclude <- exclude | (bases == paste("src/", pkgname, "_res.rc", sep=""))
        unlink(allfiles[exclude], recursive = TRUE)
        setwd(owd)

        ## Fix up man, R, demo inst/doc directories
        res <- .check_package_subdirs(pkgname, TRUE)
        if (any(sapply(res, length))) {
            messageLog(Log, "excluding invalid files")
            print(res) # FIXME print to Log?
        }
        setwd(Tdir)
        ## Fix permissions for all files to be at least 644, and dirs 755
        ## Not restricted by umask.
        if (!WINDOWS) .Internal(dirchmod(pkgname))
        ## Add build stamp to the DESCRIPTION file.
        add_build_stamp_to_description_file(file.path(pkgname, "DESCRIPTION"))
        messageLog(Log,
                   "checking for LF line-endings in source and make files")
        fix_nonLF_in_source_files(pkgname, Log)
        fix_nonLF_in_make_files(pkgname, Log)
        messageLog(Log, "checking for empty or unneeded directories");
        find_empty_dirs(pkgname)
        for(dir in c("Meta", "R-ex", "chtml", "help", "html", "latex")) {
            d <- file.path(pkgname, dir)
            if (dir.exists(d)) {
                msg <- paste("WARNING: Removing directory",
                             sQuote(d),
                             "which should only occur",
                             "in an installed package")
                printLog(Log, paste(strwrap(msg, indent = 0L, exdent = 2L),
                                    collapse = "\n"), "\n")
                unlink(d, recursive=TRUE)
            }
        }
        ## remove subarch build directories
        unlink(file.path(pkgname,
                         c("src-i386", "src-x64", "src-x86_64", "src-ppc")),
               recursive = TRUE)

        ## work on 'data' directory if present
        if(file_test("-d", file.path(pkgname, "data"))) {
            messageLog(Log, "looking to see if a 'data/datalist' file should be added")
            ## in some cases data() needs the package installed as
            ## there are links to the package's namespace
            tryCatch(add_datalist(pkgname),
                     error = function(e)
                     printLog(Log, "  unable to create a 'datalist' file: may need the package to be installed\n"))
            ## allow per-package override
            resave_data1 <- parse_description_field(desc, "BuildResaveData",
                                                    resave_data, FALSE)
            resave_data_others(pkgname, resave_data1)
            resave_data_rda(pkgname, resave_data1)
        }

        ## Finalize
        if (binary) {
            messageLog(Log, "building binary distribution")
            setwd(startdir)
            libdir <- tempfile("Rinst")
            dir.create(libdir, mode = "0755")
            srcdir <- file.path(Tdir, pkgname)
            cmd <- if (WINDOWS)
                paste(shQuote(file.path(R.home("bin"), "Rcmd.exe")),
                      "INSTALL -l", shQuote(libdir),
                      "--build", paste(INSTALL_opts, collapse = " "),
                      shQuote(pkgdir))
            else
                 paste(shQuote(file.path(R.home("bin"), "R")),
                       "CMD INSTALL -l", shQuote(libdir),
                      "--build", paste(INSTALL_opts, collapse = " "),
                       shQuote(pkgdir))
            if (system(cmd)) {
                errorLog(Log, "Installation failed")
                do_exit(1)
            }
        } else {
            filename <- paste(pkgname, "_", desc["Version"], ".tar.gz", sep="")
            filepath <- file.path(startdir, filename)
            ## NB: naughty reg-packages.R relies on this exact format!
            messageLog(Log, "building ", sQuote(filename))
            ## This should be set on a Unix-alike, but might get set to ""
            TAR <- Sys.getenv("TAR")
            if(!nzchar(TAR)) {
                ## The tar.exe in Rtools has --force-local by default,
                ## but this enables people to use Cygwin or MSYS tar.
                TAR <- if (WINDOWS) "tar --force-local" else "internal"
            }
            res <- utils::tar(filepath, pkgname, compression = "gzip",
                              compression_level = 9, tar = TAR)
            if (res) {
                errorLog(Log, "packaging into .tar.gz failed")
                do_exit(1L)
            }
            message("") # blank line
        }
        setwd(startdir)
        unlink(Tdir, recursive = TRUE)
        on.exit() # cancel closeLog
        closeLog(Log)
    }
    do_exit(0L)
}
