% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/fdaPDE.smoothing.R
\name{smooth.FEM.PDE.basis}
\alias{smooth.FEM.PDE.basis}
\title{Spatial regression with differential regularization: anysotropic case (elliptic PDE)}
\usage{
smooth.FEM.PDE.basis(locations = NULL, observations, FEMbasis,
       lambda, PDE_parameters, covariates = NULL, BC = NULL, GCV = FALSE,
       CPP_CODE = TRUE)
}
\arguments{
\item{locations}{A #observations-by-2 matrix where each row specifies the spatial coordinates \code{x} and \code{y} of the corresponding observations in the vector \code{observations}.
This parameter can be \code{NULL}. In this case the spatial coordinates of the corresponding observations are assigned as specified in \code{observations}.}

\item{observations}{A vector of length #observations with the observed data values over the domain.
The locations of the observations can be specified with the \code{locations} argument.
Otherwise if only the vector of observations is given, these are consider to be located in the corresponding node in the table
\code{nodes} of the mesh. In this last case, an \code{NA} value in the \code{observations} vector indicates that there is no observation associated to the corresponding
 node.
\code{NA} values are admissible to indicate that the node is not associated with any observed data value.}

\item{FEMbasis}{A \code{FEMbasis} object describing the Finite Element basis, as created by \code{\link{create.FEM.basis}}.}

\item{lambda}{A scalar or vector of smoothing parameters.}

\item{PDE_parameters}{A list specifying the parameters of the elliptic PDE in the regularizing term: \code{K}, a 2-by-2 matrix of diffusion coefficients. This induces an anisotropic
smoothing with a preferential direction that corresponds to the first eigenvector of the diffusion matrix K; \code{b}, a vector of length 2 of advection coefficients. This induces a
smoothing only in the direction specified by the vector \code{b}; \code{c}, a scalar reaction coefficient. \code{c} induces a shrinkage of the surface to zero}

\item{covariates}{A #observations-by-#covariates matrix where each row represents the covariates associated with the corresponding observed data value in \code{observations}.}

\item{BC}{A list with two vectors:
\code{BC_indices}, a vector with the indices in \code{nodes} of boundary nodes where a Dirichlet Boundary Condition should be applied;
\code{BC_values}, a vector with the values that the spatial field must take at the nodes indicated in \code{BC_indices}.}

\item{GCV}{Boolean. If \code{TRUE} the following quantities are computed: the trace of the smoothing matrix, the estimated error standard deviation,  and
the Generalized Cross Validation criterion, for each value of the smoothing parameter specified in \code{lambda}.}

\item{CPP_CODE}{Boolean. If \code{TRUE} the computation relies on the C++ implementation of the algorithm. This usually ensures a much faster computation.}
}
\value{
A list with the following variables:
         \item{\code{fit.FEM}}{A \code{FEM} object that represents the fitted spatial field.}
         \item{\code{PDEmisfit.FEM}}{A \code{FEM} object that represents the PDE misfit for the estimated spatial field.}
         \item{\code{beta}}{If covariates is not \code{NULL}, a vector of length #covariates with the regression coefficients associated with each covariate.}
         \item{\code{edf}}{If GCV is \code{TRUE}, a scalar or vector with the trace of the smoothing matrix for each value of the smoothing parameter specified in \code{lambda}.}
         \item{\code{stderr}}{If GCV is \code{TRUE}, a scalar or vector with the estimate of the standard deviation of the error for each value of the smoothing parameter specified in \code{lambda}.}
         \item{\code{GCV}}{If GCV is \code{TRUE}, a  scalar or vector with the value of the GCV criterion for each value of the smoothing parameter specified in \code{lambda}.}
}
\description{
This function implements a spatial regression model with differential regularization; anysotropic case. In particular, the regularizing term involves a second order elliptic PDE, that models the space-variation of the phenomenon. Space-varying covariates can be included in the model. The technique accurately handle data distributed over irregularly shaped domains. Moreover, various conditions can be imposed at the domain boundaries.
}
\examples{
# Load the mesh and plot it
data(mesh.2D.simple)
plot(mesh.2D.simple)
# Create a vector of noisy samples of an underlying spatial field,
# located over the nodes of the mesh
observations = sin(pi*mesh.2D.simple$nodes[,1]) + rnorm(n = nrow(mesh.2D.simple$nodes), sd = 0.1)
# Create the FEM basis object
FEMbasis = create.FEM.basis(mesh.2D.simple, 2)

# Set a vector of smoothing coefficients
lambda = c(10^-4, 1, 10^4)

# Set the anysotropic smoothing matrix K
PDE_parameters_anys = list(K = matrix(c(0.01,0,0,1), nrow = 2), b = c(0,0), c = 0)
# Estimate one field for each smoothing parameter and plot these
FEM_CPP_PDE = smooth.FEM.PDE.basis(observations = observations,
                                   FEMbasis = FEMbasis, lambda = lambda,
                                   PDE_parameters = PDE_parameters_anys)
plot(FEM_CPP_PDE$fit.FEM)

# Evaluate solution in three points
eval.FEM(FEM_CPP_PDE$fit.FEM, locations = rbind(c(0,0),c(0.5,0),c(-2,-2)))
}
\references{
Azzimonti, L., Sangalli, L.M., Secchi, P., Domanin, M., and Nobile, F., 2014. Blood flow velocity field estimation via spatial regression with PDE penalization Blood flow velocity field estimation via spatial regression with PDE penalization. DOI. 10.1080/01621459.2014.946036. \cr
 Azzimonti, L., Nobile, F., Sangalli, L.M., and Secchi, P., 2014. Mixed Finite Elements for Spatial Regression with PDE Penalization. SIAM/ASA Journal on Uncertainty Quantification, 2(1), pp.305-335.
}
\seealso{
\code{\link{smooth.FEM.basis}}, \code{\link{smooth.FEM.PDE.sv.basis}}
}

