% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/has_columns.R
\name{has_columns}
\alias{has_columns}
\title{Determine if one or more columns exist in a table}
\usage{
has_columns(x, columns)
}
\arguments{
\item{x}{The table object.}

\item{columns}{One or more column names that are to be checked for existence
in the table \code{x}.}
}
\value{
A length-1 logical vector.
}
\description{
This utility function can help you easily determine whether a column of a
specified name is present in a table object. This function works well enough
on a table object but it can also be used as part of a formula in any
validation function's \code{active} argument. Using \code{active = ~ . \%>\% has_columns("column_1")} means that the validation step will be inactive if
the target table doesn't contain a column named \code{column_1}. We can also use
multiple columns in \code{vars()} so having \code{active = ~ . \%>\% has_columns(vars(column_1, column_2))} in a validation step will make it
inactive at \code{\link[=interrogate]{interrogate()}} time unless the columns \code{column_1} and \code{column_2}
are both present.
}
\section{Examples}{


The \code{small_table} dataset in the package has the columns \code{date_time}, \code{date},
and the \code{a} through \code{f} columns.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table
#> # A tibble: 13 x 8
#>    date_time           date           a b             c      d e     f    
#>    <dttm>              <date>     <int> <chr>     <dbl>  <dbl> <lgl> <chr>
#>  1 2016-01-04 11:00:00 2016-01-04     2 1-bcd-345     3  3423. TRUE  high 
#>  2 2016-01-04 00:32:00 2016-01-04     3 5-egh-163     8 10000. TRUE  low  
#>  3 2016-01-05 13:32:00 2016-01-05     6 8-kdg-938     3  2343. TRUE  high 
#>  4 2016-01-06 17:23:00 2016-01-06     2 5-jdo-903    NA  3892. FALSE mid  
#>  5 2016-01-09 12:36:00 2016-01-09     8 3-ldm-038     7   284. TRUE  low  
#>  6 2016-01-11 06:15:00 2016-01-11     4 2-dhe-923     4  3291. TRUE  mid  
#>  7 2016-01-15 18:46:00 2016-01-15     7 1-knw-093     3   843. TRUE  high 
#>  8 2016-01-17 11:27:00 2016-01-17     4 5-boe-639     2  1036. FALSE low  
#>  9 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
#> 10 2016-01-20 04:30:00 2016-01-20     3 5-bce-642     9   838. FALSE high 
#> 11 2016-01-26 20:07:00 2016-01-26     4 2-dmx-010     7   834. TRUE  low  
#> 12 2016-01-28 02:51:00 2016-01-28     2 7-dmx-010     8   108. FALSE low  
#> 13 2016-01-30 11:23:00 2016-01-30     1 3-dka-303    NA  2230. TRUE  high
}\if{html}{\out{</div>}}

With \code{has_columns()} we can check for column existence by using it directly
on the table. A column name can be verified as present by using it in double
quotes.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% has_columns(columns = "date")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Multiple column names can be supplied. The following is \code{TRUE} because both
columns are present in \code{small_table}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% has_columns(columns = c("a", "b"))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

It's possible to supply column names in \code{vars()} as well:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% has_columns(columns = vars(a, b))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] TRUE
}\if{html}{\out{</div>}}

Because column \code{h} isn't present, this returns \code{FALSE} (all specified columns
need to be present to obtain \code{TRUE}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{small_table \%>\% has_columns(columns = vars(a, h))
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] FALSE
}\if{html}{\out{</div>}}

The \code{has_columns()} function can be useful in expressions that involve the
target table, especially if it is uncertain that the table will contain a
column that's involved in a validation.

In the following agent-based validation, the first two steps will be 'active'
because all columns checked for in the expressions are present. The third
step becomes inactive because column \code{j} isn't there (without the \code{active}
statement there we would get an evaluation failure in the agent report).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{agent <- 
  create_agent(
    tbl = small_table,
    tbl_name = "small_table"
  ) \%>\%
  col_vals_gt(
    columns = vars(c), value = vars(a),
    active = ~ . \%>\% has_columns(vars(a, c))
  ) \%>\%
  col_vals_lt(
    columns = vars(h), value = vars(d),
    preconditions = ~ . \%>\% dplyr::mutate(h = d - a),
    active = ~ . \%>\% has_columns(vars(a, d))
  ) \%>\%
  col_is_character(
    columns = vars(j),
    active = ~ . \%>\% has_columns("j")
  ) \%>\%
  interrogate()
}\if{html}{\out{</div>}}

Through the agent's x-list, we can verify that no evaluation error (any
evaluation at all, really) had occurred. The third value, representative of
the third validation step, is actually \code{NA} instead of \code{FALSE} because the
step became inactive.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{x_list <- get_agent_x_list(agent = agent)

x_list$eval_warning
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## [1] FALSE FALSE    NA
}\if{html}{\out{</div>}}
}

\section{Function ID}{

13-2
}

\seealso{
Other Utility and Helper Functions: 
\code{\link{affix_datetime}()},
\code{\link{affix_date}()},
\code{\link{col_schema}()},
\code{\link{from_github}()},
\code{\link{stop_if_not}()}
}
\concept{Utility and Helper Functions}
