% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_agent_report.R
\name{get_agent_report}
\alias{get_agent_report}
\title{Get a summary report from an agent}
\usage{
get_agent_report(
  agent,
  arrange_by = c("i", "severity"),
  keep = c("all", "fail_states"),
  display_table = TRUE,
  size = "standard",
  title = ":default:",
  lang = NULL,
  locale = NULL
)
}
\arguments{
\item{agent}{An agent object of class \code{ptblank_agent}.}

\item{arrange_by}{A choice to arrange the report table rows by the validation
step number (\code{"i"}, the default), or, to arrange in descending order by
severity of the failure state (with \code{"severity"}).}

\item{keep}{An option to keep \code{"all"} of the report's table rows (the
default), or, keep only those rows that reflect one or more
\code{"fail_states"}.}

\item{display_table}{Should a display table be generated? If \code{TRUE} (the
default), and if the \strong{gt} package is installed, a display table for the
report will be shown in the Viewer. If \code{FALSE}, or if \strong{gt} is not
available, then a tibble will be returned.}

\item{size}{The size of the display table, which can be either \code{"standard"}
(the default) or \code{"small"}. This only applies to a display table (where
\code{display_table = TRUE}).}

\item{title}{Options for customizing the title of the report. The default is
the keyword \code{":default:"} which produces generic title text that refers to
the \strong{pointblank} package in the language governed by the \code{lang} option.
Another keyword option is \code{":tbl_name:"}, and that presents the name of the
table as the title for the report. If no title is wanted, then the
\code{":none:"} keyword option can be used. Aside from keyword options, text can
be provided for the title and \code{glue::glue()} calls can be used to construct
the text string. If providing text, it will be interpreted as Markdown text
and transformed internally to HTML. To circumvent such a transformation,
use text in \code{\link[=I]{I()}} to explicitly state that the supplied text should not be
transformed.}

\item{lang}{The language to use for automatic creation of briefs (short
descriptions for each validation step) and for the \emph{agent report} (a
summary table that provides the validation plan and the results from the
interrogation. By default, \code{NULL} will create English (\code{"en"}) text. Other
options include French (\code{"fr"}), German (\code{"de"}), Italian (\code{"it"}), Spanish
(\code{"es"}), Portuguese, (\code{"pt"}), Chinese (\code{"zh"}), and Russian (\code{"ru"}).
This \code{lang} option will override any previously set lang value (e.g., by
the \code{\link[=create_agent]{create_agent()}} call).}

\item{locale}{An optional locale ID to use for formatting values in the
\emph{agent report} summary table according the locale's rules. Examples include
\code{"en_US"} for English (United States) and \code{"fr_FR"} for French (France);
more simply, this can be a language identifier without a country
designation, like \code{"es"} for Spanish (Spain, same as \code{"es_ES"}). This
\code{locale} option will override any previously set locale value (e.g., by the
\code{\link[=create_agent]{create_agent()}} call).}
}
\value{
A \strong{gt} table object if \code{display_table = TRUE} or a tibble if
\code{display_table = FALSE}.
}
\description{
We can get an informative summary table from an agent by using the
\code{get_agent_report()} function. The table can be provided in two substantially
different forms: as a \strong{gt} based display table (the default), or, as a
tibble. The amount of fields with intel is different depending on whether or
not the agent performed an interrogation (with the \code{\link[=interrogate]{interrogate()}} function).
Basically, before \code{\link[=interrogate]{interrogate()}} is called, the agent will contain just the
validation plan (however many rows it has depends on how many validation
functions were supplied a part of that plan). Post-interrogation, information
on the passing and failing test units is provided, along with indicators on
whether certain failure states were entered (provided they were set through
\code{actions}). The display table variant of the agent report, the default form,
will have the following columns:

\itemize{
\item i (unlabeled): the validation step number
\item STEP: the name of the validation function used for the validation step
\item COLUMNS: the names of the target columns used in the validation step
(if applicable)
\item VALUES: the values used in the validation step, where applicable; this
could be as literal values, as column names, an expression, a set of
sub-validations (for a \code{\link[=conjointly]{conjointly()}} validation step), etc.
\item TBL: indicates whether any there were any preconditions to apply
before interrogation; if not, a script 'I' stands for 'identity' but, if so,
a right-facing arrow appears
\item EVAL: a character value that denotes the result of each validation
step functions' evaluation during interrogation
\item \emph{N}: the total number of test units for the validation step
\item PASS: the number of test units that received a \emph{pass}
\item FAIL: the fraction of test units that received a \emph{pass}
\item W, S, N: indicators that show whether the \code{warn}, \code{stop}, or \code{notify}
states were entered; unset states appear as dashes, states that are set with
thresholds appear as unfilled circles when not entered and filled when
thresholds are exceeded (colors for W, S, and N are amber, red, and blue)
\item EXT: a column that provides buttons with data extracts for each
validation step where failed rows are available (as CSV files)
}

The small version of the display table (obtained using \code{size = "small"})
omits the \code{COLUMNS}, \code{TBL}, and \code{EXT} columns. The width of the small table
is 575px; the standard table is 875px wide.

If choosing to get a tibble (with \code{display_table = FALSE}), it will have the
following columns:

\itemize{
\item i: the validation step number
\item type: the name of the validation function used for the validation step
\item columns: the names of the target columns used in the validation step
(if applicable)
\item values: the values used in the validation step, where applicable; for
a \code{\link[=conjointly]{conjointly()}} validation step, this is a listing of all sub-validations
\item precon: indicates whether any there are any preconditions to apply
before interrogation and, if so, the number of statements used
\item active: a logical value that indicates whether a validation step is
set to \code{"active"} during an interrogation
\item eval: a character value that denotes the result of each validation
step functions' evaluation during interrogation
\item units: the total number of test units for the validation step
\item n_pass: the number of test units that received a \emph{pass}
\item f_pass: the fraction of test units that received a \emph{pass}
\item W, S, N: logical value stating whether the \code{warn}, \code{stop}, or \code{notify}
states were entered
\item extract: a logical value that indicates whether a data extract is
available for the validation step
}
}
\section{Function ID}{

6-2
}

\examples{
# Create a simple table with a
# column of numerical values
tbl <- 
  dplyr::tibble(a = c(5, 7, 8, 5))

# Validate that values in column
# `a` are always greater than 4
agent <-
  create_agent(tbl = tbl) \%>\%
  col_vals_gt(vars(a), value = 4) \%>\%
  interrogate()

# Get a tibble-based report from the
# agent by using `get_agent_report()`
# with `display_table = FALSE`
agent \%>\%
  get_agent_report(display_table = FALSE)
  
# View a the report by printing the
# `agent` object anytime, but, return a
# gt table object by using this with
# `display_table = TRUE` (the default)
report <- get_agent_report(agent)
class(report)

# What can you do with the report?
# Print it from an R Markdown code,
# use it in an email, put it in a
# webpage, or further modify it with
# the **gt** package

# The agent report as a **gt** display
# table comes in two sizes: "standard"
# (the default) and "small"
small_report <- 
  get_agent_report(
    agent = agent,
    size = "small"
  )

class(small_report)

# The standard report is 875px wide
# the small one is 575px wide

}
\seealso{
Other Interrogate and Report: 
\code{\link{interrogate}()}
}
\concept{Interrogate and Report}
