% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coxpath.R
\name{multiview.cox.path}
\alias{multiview.cox.path}
\title{Fit a Cox regression model with elastic net regularization for a path of
lambda values}
\usage{
multiview.cox.path(
  x_list,
  x,
  y,
  rho = 0,
  weights = NULL,
  lambda = NULL,
  offset = NULL,
  alpha = 1,
  nlambda = 100,
  lambda.min.ratio = ifelse(nobs < nvars, 0.01, 1e-04),
  standardize = TRUE,
  intercept = TRUE,
  thresh = 1e-07,
  exclude = integer(0),
  penalty.factor = rep(1, nvars),
  lower.limits = -Inf,
  upper.limits = Inf,
  maxit = 1e+05,
  trace.it = 0,
  nvars,
  nobs,
  xm,
  xs,
  control,
  vp,
  vnames,
  is.offset
)
}
\arguments{
\item{x_list}{a list of \code{x} matrices with same number of rows
\code{nobs}}

\item{x}{the \code{cbind}ed matrices in \code{x_list}}

\item{y}{the quantitative response with length equal to \code{nobs}, the
(same) number of rows in each \code{x} matrix}

\item{rho}{the weight on the agreement penalty, default 0. \code{rho=0}
is a form of early fusion, and \code{rho=1} is a form of late fusion.
We recommend trying a few values of \code{rho} including 0, 0.1, 0.25,
0.5, and 1 first; sometimes \code{rho} larger than 1 can also be
helpful.}

\item{weights}{observation weights. Can be total counts if
responses are proportion matrices. Default is 1 for each
observation}

\item{lambda}{A user supplied \code{lambda} sequence, default
\code{NULL}. Typical usage is to have the program compute its own
\code{lambda} sequence. This sequence, in general, is different from
that used in the \code{\link[glmnet:glmnet]{glmnet::glmnet()}} call (named \code{lambda})
Supplying a value of \code{lambda} overrides this. WARNING: use with
care. Avoid supplying a single value for \code{lambda} (for
predictions after CV use \code{\link[stats:predict]{stats::predict()}} instead.  Supply
instead a decreasing sequence of \code{lambda} values as \code{multiview}
relies on its warms starts for speed, and its often faster to fit
a whole path than compute a single fit.}

\item{offset}{A vector of length \code{nobs} that is included in
the linear predictor (a \code{nobs x nc} matrix for the
\code{"multinomial"} family).  Useful for the \code{"poisson"}
family (e.g. log of exposure time), or for refining a model by
starting at a current fit. Default is \code{NULL}. If supplied,
then values must also be supplied to the \code{predict} function.}

\item{alpha}{The elasticnet mixing parameter, with
\eqn{0\le\alpha\le 1}.  The penalty is defined as
\deqn{(1-\alpha)/2||\beta||_2^2+\alpha||\beta||_1.}
\code{alpha=1} is the lasso penalty, and \code{alpha=0} the ridge
penalty.}

\item{nlambda}{The number of \code{lambda} values - default is 100.}

\item{lambda.min.ratio}{Smallest value for \code{lambda}, as a
fraction of \code{lambda.max}, the (data derived) entry value
(i.e. the smallest value for which all coefficients are
zero). The default depends on the sample size \code{nobs}
relative to the number of variables \code{nvars}. If \code{nobs >
  nvars}, the default is \code{0.0001}, close to zero.  If
\code{nobs < nvars}, the default is \code{0.01}.  A very small
value of \code{lambda.min.ratio} will lead to a saturated fit in
the \code{nobs < nvars} case. This is undefined for
\code{"binomial"} and \code{"multinomial"} models, and
\code{glmnet} will exit gracefully when the percentage deviance
explained is almost 1.}

\item{standardize}{Logical flag for x variable standardization,
prior to fitting the model sequence. The coefficients are always
returned on the original scale. Default is
\code{standardize=TRUE}.  If variables are in the same units
already, you might not wish to standardize. See details below for
y standardization with \code{family="gaussian"}.}

\item{intercept}{Should intercept(s) be fitted (default \code{TRUE})}

\item{thresh}{Convergence threshold for coordinate descent. Each
inner coordinate-descent loop continues until the maximum change
in the objective after any coefficient update is less than
\code{thresh} times the null deviance. Defaults value is
\code{1E-7}.}

\item{exclude}{Indices of variables to be excluded from the
model. Default is none. Equivalent to an infinite penalty factor
for the variables excluded (next item).  Users can supply instead
an \code{exclude} function that generates the list of indices.  This
function is most generally defined as \verb{function(x_list, y, ...)},
and is called inside \code{multiview} to generate the indices for
excluded variables.  The \code{...} argument is required, the others
are optional.  This is useful for filtering wide data, and works
correctly with \code{cv.multiview}. See the vignette 'Introduction'
for examples.}

\item{penalty.factor}{Separate penalty factors can be applied to
each coefficient. This is a number that multiplies \code{lambda}
to allow differential shrinkage. Can be 0 for some variables,
which implies no shrinkage, and that variable is always included
in the model. Default is 1 for all variables (and implicitly
infinity for variables listed in \code{exclude}). Note: the
penalty factors are internally rescaled to sum to nvars, and the
lambda sequence will reflect this change.}

\item{lower.limits}{Vector of lower limits for each coefficient;
default \code{-Inf}. Each of these must be non-positive. Can be
presented as a single value (which will then be replicated), else
a vector of length \code{nvars}}

\item{upper.limits}{Vector of upper limits for each coefficient;
default \code{Inf}. See \code{lower.limits}}

\item{maxit}{Maximum number of passes over the data for all lambda
values; default is 10^5.}

\item{trace.it}{If \code{trace.it=1}, then a progress bar is
displayed; useful for big models that take a long time to fit.}

\item{nvars}{the number of variables (total)}

\item{nobs}{the number of observations}

\item{xm}{the column means vector (could be zeros if \code{standardize = FALSE})}

\item{xs}{the column std dev vector (could be 1s if \code{standardize = FALSE})}

\item{control}{the multiview control object}

\item{vp}{the variable penalities (processed)}

\item{vnames}{the variable names}

\item{is.offset}{a flag indicating if offset is supplied or not}
}
\value{
An object of class "coxnet" and "glmnet".
\item{a0}{Intercept value, \code{NULL} for "cox" family.}
\item{beta}{A \code{nvars x length(lambda)} matrix of coefficients, stored in
sparse matrix format.}
\item{df}{The number of nonzero coefficients for each value of lambda.}
\item{dim}{Dimension of coefficient matrix.}
\item{lambda}{The actual sequence of lambda values used. When alpha=0, the
largest lambda reported does not quite give the zero coefficients reported
(lambda=inf would in principle). Instead, the largest lambda for alpha=0.001
is used, and the sequence of lambda values is derived from this.}
\item{dev.ratio}{The fraction of (null) deviance explained. The deviance
calculations incorporate weights if present in the model. The deviance is
defined to be 2*(loglike_sat - loglike), where loglike_sat is the log-likelihood
for the saturated model (a model with a free parameter per observation).
Hence dev.ratio=1-dev/nulldev.}
\item{nulldev}{Null deviance (per observation). This is defined to be
2*(loglike_sat -loglike(Null)). The null model refers to the 0 model.}
\item{npasses}{Total passes over the data summed over all lambda values.}
\item{jerr}{Error flag, for warnings and errors (largely for internal
debugging).}
\item{offset}{A logical variable indicating whether an offset was included
in the model.}
\item{call}{The call that produced this object.}
\item{nobs}{Number of observations.}
}
\description{
Fit a Cox regression model via penalized maximum likelihood for a path of
lambda values. Can deal with (start, stop] data and strata, as well as
sparse design matrices.
}
\details{
Sometimes the sequence is truncated before \code{nlambda} values of lambda
have been used. This happens when \code{cox.path} detects that the
decrease in deviance is marginal (i.e. we are near a saturated fit).
}
\examples{
set.seed(2)
nobs <- 100; nvars <- 15
xvec <- rnorm(nobs * nvars)
xvec[sample.int(nobs * nvars, size = 0.4 * nobs * nvars)] <- 0
x <- matrix(xvec, nrow = nobs)
beta <- rnorm(nvars / 3)
fx <- x[, seq(nvars / 3)] \%*\% beta / 3
ty <- rexp(nobs, exp(fx))
tcens <- rbinom(n = nobs, prob = 0.3, size = 1)
jsurv <- survival::Surv(ty, tcens)
fit1 <- glmnet:::cox.path(x, jsurv)

# works with sparse x matrix
x_sparse <- Matrix::Matrix(x, sparse = TRUE)
fit2 <- glmnet:::cox.path(x_sparse, jsurv)

# example with (start, stop] data
set.seed(2)
start_time <- runif(100, min = 0, max = 5)
stop_time <- start_time + runif(100, min = 0.1, max = 3)
status <- rbinom(n = nobs, prob = 0.3, size = 1)
jsurv_ss <- survival::Surv(start_time, stop_time, status)
fit3 <- glmnet:::cox.path(x, jsurv_ss)

# example with strata
jsurv_ss2 <- glmnet::stratifySurv(jsurv_ss, rep(1:2, each = 50))
fit4 <- glmnet:::cox.path(x, jsurv_ss2)
}
