% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bfactor_to_prob.R
\name{bfactor_to_prob}
\alias{bfactor_to_prob}
\title{Turn Bayes Factors Into Posterior Probabilities}
\usage{
bfactor_to_prob(bf, prior_prob = 0.5)
}
\arguments{
\item{bf}{A numeric vector of non-negative values.}

\item{prior_prob}{A numeric vector with values in the [0,1] interval. If \code{length(bf) == 1} then \code{prior_prob} can be of any positive \code{\link[base]{length}}, but if \code{length(bf) > 1} then the \code{\link[base]{length}} of \code{prior_prob} can only be \code{1} or equal to the \code{\link[base]{length}} of \code{bf}.}
}
\value{
If \code{length(bf) > 1} then \code{bfactor_to_prob} returns a numeric vector with the same \code{\link[base]{length}} as \code{bf}, otherwise it returns a numeric vector with the same \code{\link[base]{length}} as \code{prior_prob}.
}
\description{
Update the prior probabilities of models/hypotheses to posterior probabilities using Bayes factors.
}
\details{
\code{bfactor_to_prob} computes the posterior probability of the null hypothesis using the following equation from \insertCite{bergerDelampady1987;textual}{pcal}: \deqn{P(\textrm{null} \, \textrm{hypothesis}|\textrm{data}) = \left(1 + \frac{1 - {null\_prob}}{null\_prob} \times \frac{1}{bf}\right)^{-1}}{P(null hypothesis | data) = (1 + (1 - prior_prob) / prior_prob * (1 / bf)) ^(-1)} where \code{bf} is a Bayes factor if favor of the null hypothesis and \code{prior_prob} is the prior probability of the null hypothesis. The alternative hypothesis has prior probability \code{1 - prior_prob} and posterior probability \code{1 - bfactor_to_prob(bf, prior_prob)}.

The \code{prior_prob} argument is optional and is set to 0.5 by default, implying prior equiprobability of hypotheses. \code{prior_prob} can only be of \code{\link[base]{length}} equal to \code{length(bf)}, in which case each prior probability in \code{prior_prob} will be updated using the corresponding element of \code{bf}, or of \code{\link[base]{length}} \code{1}, in which case it will be recycled (if \code{length(bf) > 1}) and each element of \code{bf} will update the same \code{prior_prob} value.
}
\examples{
# With a Bayes factor that is indifferent between the null and the alternative hypotheses:
bfactor_to_prob(1)

# Same as above but the null hypothesis has high prior probability:
bfactor_to_prob(1, .99)

# Posterior probability of the null as a function of different prior probabilities:
bfactor_to_prob(1, seq(.5, 1, .1))

# With Bayes factors that favor the null hypothesis:
round(bfactor_to_prob(seq(2, 50, 2.5)), 3)

# Same as above but the null hypothesis has low prior probability:
round(bfactor_to_prob(seq(2, 50, 2.5), prior_prob = .01), 3)

# Posterior probabilities obtained with Bayes factors that favor the alternative hypothesis:
round(bfactor_to_prob(seq(0, 1, .05)), 3)

# Same as above but the null hypothesis has high prior probability:
round(bfactor_to_prob(seq(0, 1, .05), prior_prob = .99), 3)

# Application: chi-squared goodness-of-fit test,
# lower bound on the posterior probability of the null hypothesis:
x <- matrix(c(12, 41, 25, 33), ncol = 2)
bfactor_to_prob(bcal(chisq.test(x)[["p.value"]]), prior_prob = .9)

}
\references{
\insertAllCited{}
}
\seealso{
\itemize{
\item \code{\link[pcal]{bfactor_interpret}} and \code{\link[pcal]{bfactor_interpret_kr}} for the interpretation of Bayes factors.
\item \code{\link[pcal]{bfactor_log_interpret}} and \code{\link[pcal]{bfactor_log_interpret_kr}} for the interpretation of the logarithms of Bayes factors.
\item \code{\link[pcal]{bcal}} for a p-value calibration that returns lower bounds on Bayes factors in favor of point null hypotheses.
\item \code{\link[pcal]{pcal}} for a p-value calibration that returns lower bounds on the posterior probabilities of point null hypotheses.
}
}
