#' Repeated Measures (ANOVA & Multiple Regression)
#'
#' @param k number of variables.
#' @param n number of observations.
#' @param mean vector of means.
#' @param sigma vector of standard deviations/covariance-correlation matrix.
#' @param coefvar vector (optional) of coefficients of variation.
#' @param sigmaSup vector (optional) of standard deviations if sigma is a correlation matrix.
#' @param conf.level confidence level for interval in T-Test.
#' @param random a logical indicating whether you want a random covariance/variance matrix.
#' @param dec number of decimals for observations.
#'
#' @usage
#' repeatedm(k, n, mean = 0, sigma = 1, coefvar = NULL,
#'           sigmaSup = NULL, conf.level = 0.95,
#'           random = FALSE, dec = 2)
#'
#' @details Number of variables must be greater than 3, in order to ensure an ANOVA of repeated measures or a multiple Linear Regression.
#'
#'    \code{sigma} can represent a vector or a covariance/correlation matrix. In case \code{sigma} is a vector, independent samples are created. By other hand, if it's a correlation matrix parameter \code{sigmaSup} is required. For covariance matrices, the function does not require any other parameter or special treatment.
#'
#'    If \code{random = TRUE}, a random covariance matrix is generated by using \cite{genpositiveDefMat()}.
#'
#'
#' @seealso \code{[clusterGeneration::genpositiveDefMat()]}
#'
#' @return A data frame.
#'
#' @examples
#' randm <- clusterGeneration::genPositiveDefMat(8, covMethod = "unifcorrmat")
#' mcov <- randm$Sigma
#' Sigma <- cov2cor(mcov)
#' is.corrmatrix(Sigma)
#' repeatedm(k = 8, n = 8, mean = c(20,5, 30, 15),sigma = Sigma, sigmaSup = 2,  dec = 2)
#'
#' repeatedm(k = 5, n = 5, mean = c(8,10,5,14,22.5), random = TRUE)
#' repeatedm(k = 3, n = 8, mean = c(10,5,22.5), sigma = c(3.3,1.5,5), dec = 2)
#'
#' @export


repeatedm <- function(k, n, mean = 0, sigma = 1, coefvar = NULL, sigmaSup = NULL, conf.level = 0.95, random = FALSE, dec = 2){

  if(k < 3 )
    warning("Number of variables must be at least 3 (ANOVA or Regression).")

  else if(all(coefvar) != 0 && any(mean == 0))
    warning("Vector of means cannot have a 0 element.")

  else if(is.matrix(sigma) && (!is.corrmatrix(sigma) && !is.covmatrix(sigma)))
    warning("Given matrix is not a correlation or covariance matrix.")


  mean <- rep(mean, k)[1:k]
  coefvar <- rep(coefvar, k)[1:k]

  if(random == TRUE){
    message("Random covariance matrix generated. sigma omitted")
    randm <- clusterGeneration::genPositiveDefMat(k, covMethod = "unifcorrmat")
    mcov <- randm$Sigma
    sigma <- sqrt(diag(mcov))
  }

  else{

    if(is.vector(sigma)){
      message("Beware! Sigma is a vector, covariance matrix is diagonal.")

      if(!is.null(coefvar))
        sigma <- coefvar * abs(mean)
      else
        sigma <- rep(sigma, k)[1:k]
    }

    else{

      if(is.corrmatrix(sigma))
        sigma <- mCorrCov(mcorr = sigma, sigma = sigmaSup)
    }

  }

  gen <- generator(n, mean, sigma = sigma, dec = dec)

  Values <- gen$Samples
  d <- data.frame(Values)

  return(d)

}
