/*
 * Created on Apr 19, 2004
 *  
 */
package rfdd.chainons;

import java.util.Iterator;

import rfdd.beans.GrandGroupe;
import rfdd.beans.ListeMesures;
import rfdd.beans.ListeMesures2;
import rfdd.beans.Mesure;
import rfdd.beans.PetitGroupe;
import rfdd.beans.PetitGroupe2;

/**
 * Recherche les <code>PetitGroupe</code> adjacents correspondant aux deux
 * extrémités d'un même fragment.
 * 
 * <p>
 * Les critères pour dire si deux <code>PetitGroupe</code> adjacents sont
 * appariés sont:
 * <ul>
 * <li>même triplet de début,
 * <li>même triplet de fin,
 * <li>différence entre les tailles d'amplicon inférieures ou égales à 1.
 * </ul>
 * 
 * <p>
 * Quand on trouve une telle paire de <code>PetitGroupe</code>, on cherche
 * les paires de lignes correspondant aux critères:
 * <ul>
 * <li>tailles expérimentales (tailleRef) très proches (à +/- 0.2%,
 * paramétrable),
 * <li>profils compatibles.
 * </ul>
 * 
 * <p>
 * Si on trouve de telles paires de lignes, on les passe au <code>Chainon</code>
 * suivant (sans oublier d'annoter les lignes avec le profil dans la colonne
 * "profil fiable"), en éliminant PAS les lignes ne correspondant pas au
 * critères. <br>
 * Si on ne trouve pas de paire de lignes, on passe toutes les lignes sans
 * distinction (celles ci, on ne les annote pas) au <code>Chainon</code>
 * suivant, dans l'espoir de pouvoir les traiter plus tard.
 * 
 * <p>
 * Les <code>PetitGroupe</code> dont on a pas trouvé la paire sont passés au
 * <code>Chainon</code> suivant sans traitement.
 * 
 * @author tnguyen
 *  
 */
public class ChainonAppariement extends ChainonSimple
{
	/**
	 * La précision choisie pour comparer des tailles de référence.
	 */
	private final float precision;

	/**
	 * <code>PetitGroupe</code> précédant. Stocké parce qu'on a besoin de
	 * comparer deux <code>PetitGroupe</code> consécutifs.
	 */
	private PetitGroupe2 pgPrecedant;

	/**
	 * <code>PetitGroupe</code> courant. Stocké parce qu'on a besoin de
	 * comparer deux <code>PetitGroupe</code> consécutifs.
	 */
	private PetitGroupe2 pgEnCours;

	private ListeMesures2 ligneEnCours;

	/**
	 * Constructeur.
	 * 
	 * @param precision
	 */
	public ChainonAppariement(float precision)
	{
		this.precision = precision;
	}

	public void startGrandGroupe(GrandGroupe gg)
	{
		assert this.ligneEnCours == null;
		assert this.pgEnCours == null;
		assert pgPrecedant == null;

		super.startGrandGroupe(gg);
	}

	public void startPetitGroupe(PetitGroupe pg)
	{
		assert this.ligneEnCours == null;
		assert this.pgEnCours == null;

		this.pgEnCours = new PetitGroupe2(pg);
	}

	public void startLigne(ListeMesures l)
	{
		assert this.ligneEnCours == null;
		assert this.pgEnCours != null;

		this.ligneEnCours = new ListeMesures2(l);
	}

	public void mesure(Mesure m)
	{
		assert this.ligneEnCours != null;

		// on passe pas les mesures au chainon suivant.
		// à la place, on les stocke pour traitement futur.
		this.ligneEnCours.add(m);
	}

	public void stopLigne()
	{
		// on a fini de stocker les mesures de la ligne en cours,
		// on mémorise cette ligne dans la liste des lignes du PG en cours.
		this.pgEnCours.add(this.ligneEnCours);
		this.ligneEnCours = null;
	}

	public void stopPetitGroupe()
	{
		// on cherche deux PG qui ont mêmes tripletDebut, tripletFin, taille
		// théorique
		if (this.pgPrecedant != null
				&& ((this.pgPrecedant.getTripletDebut().equals(
						this.pgEnCours.getTripletDebut()) && this.pgPrecedant
						.getTripletFin().equals(this.pgEnCours.getTripletFin())) || (this.pgPrecedant
						.getTripletDebut().equals(
								this.pgEnCours.getTripletFin()) && this.pgPrecedant
						.getTripletFin().equals(
								this.pgEnCours.getTripletDebut())))
				&& Math.abs(this.pgPrecedant.getTailleAmplicon()
						- this.pgEnCours.getTailleAmplicon()) <= 1)
		{
			traite2PG();
		}
		else
		{
			// on a pas trouvé une paire de PG qui corresponde aux critères,
			// on passe le PG "previous" au chainon suivant.
			if (this.pgPrecedant != null)
					passePG(this.pgPrecedant);

			// et on garde le PG courant au cas où il aille avec le prochain PG
			this.pgPrecedant = this.pgEnCours;
			this.pgEnCours = null;
		}
	} // end - public void stopPetitGroupe()

	/**
	 * Normalement, là, on a trouvé deux petits groupes qui correspondent aux
	 * deux extrémités d'un même fragment.
	 * 
	 * On va farfouiller dans les lignes de chaque PG pour trouver des tailles
	 * expérimentales particulièrement proches. Si on trouve de telles paires,
	 * on les passe au chainon suivant. Si on n'en trouve pas, on passe tout en
	 * vrac au chainon suivant, dans l'espoir de trouver des correspondances
	 * plus tard.
	 */
	private void traite2PG()
	{
		assert this.pgPrecedant.getTripletDebut().equals(
				this.pgEnCours.getTripletDebut());
		assert this.pgPrecedant.getTripletFin().equals(
				this.pgEnCours.getTripletFin());
		assert Math.abs(this.pgPrecedant.getTailleAmplicon()
				- this.pgEnCours.getTailleAmplicon()) <= 1;

		Iterator i1 = this.pgPrecedant.iterator();
		while (i1.hasNext())
		{
			ListeMesures2 ligne1 = (ListeMesures2) i1.next();

			Iterator i2 = this.pgEnCours.iterator();
			while (i2.hasNext())
			{
				ListeMesures2 ligne2 = (ListeMesures2) i2.next();

				// si les deux valeurs sont égales +/- precision
				// et qu'elles ont le même profil
				if (Math.abs(ligne1.getTailleRef() - ligne2.getTailleRef()) <= this.precision
						&& ProfilUtil.compatibles(ligne1.getProfil(), ligne2
								.getProfil()))
				{
					// On annote les lignes avec leur "profil fiable"
					String pf = ProfilUtil.profilFiable(ligne1.getProfil(),
							ligne2.getProfil());
					ligne1.setProfilFiable(pf);
					ligne2.setProfilFiable(pf);
				}
			} // end - while (i2.hasNext())
		} // end - while (i1.hasNext())

		// à ce point, on doit avoir annoté les paires de lignes
		// correspondant à notre critère.
		
		//
		// on passe les deux PG au chainon suivant.
		//

		//d'abord le PG précédant
		passePG(this.pgPrecedant);

		//et puis le PG courant
		passePG(this.pgEnCours);

		//
		// on a fini de traiter les deux PG
		//
		this.pgPrecedant = null;
		this.pgEnCours = null;

	} // end - private void traite2PG()

	public void stopGrandGroupe()
	{
		assert this.pgEnCours == null;

		// si on a un PG "orphelin" à la fin d'un grand groupe,
		// on le transmet au chainon suivant.
		if (this.pgPrecedant != null)
		{
			passePG(this.pgPrecedant);
			this.pgPrecedant = null;
		}
		assert this.pgPrecedant == null;

		super.stopGrandGroupe();
	}

	/**
	 * Passe un <code>PetitGroupe</code> et ses lignes
	 * <code>ListeMesures2</code> associées au chainon suivant.
	 * 
	 * @param pg
	 * @param lignes
	 */
	private void passePG(PetitGroupe2 pg)
	{
		super.startPetitGroupe(new PetitGroupe(pg));

		Iterator i = pg.iterator();
		while (i.hasNext())
		{
			ListeMesures2 ligne = (ListeMesures2) i.next();
			super.startLigne(new ListeMesures(ligne));

			Iterator iMesures = ligne.iterator();
			while (iMesures.hasNext())
			{
				Mesure m = (Mesure) iMesures.next();
				super.mesure(m);
			}

			super.stopLigne();
		}

		super.stopPetitGroupe();
	}

	public String toString()
	{
		return "{ChainonAppariement: precision=" + this.precision + "}";
	}

}