/*
 * Created on May 27, 2004
 */
package rfdd.chainons;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import rfdd.beans.ListeMesures;
import rfdd.beans.ListeMesures2;
import rfdd.beans.Mesure;

/**
 * Calcule le code et les profils de réponses compatibles avec ce code. (comme
 * <code>ChainonRecalculeProfil2</code>, mais on duplique pas les lignes
 * quand on a plusieurs profils compatibles.)
 * 
 * <p>
 * trois modes de calcul:
 * <ul>
 * <li>mode 1: aucun zéro dans les trois valeurs de fluorescence <br>
 * On fait un calcul "normal" du code bicaractère, et on utilise le "tableau
 * faible" pour faire la correspondace code/profil
 * <li>mode 2: un zéro en extrémité (fluo_Hp=0 ou fluo_N=0) <br>
 * On fait un calcul "normal" du code bicaractère, et on utilise le "tableau
 * fort" pour faire la correspondace code/profil
 * <li>mode 3: un zéro au milieu (fluo_Hm=0) <br>
 * En plus du calcul du code bicaractère, on ajoute un caractère complémentaire
 * indiquant la différence entre les deux valeurs d'extrémité (fluo_Hp et
 * fluo_N). On utilise ce code tricaractère pour faire la correspondance avec
 * les profils dans un troisième tableau spécial pour le mode 3.
 * </ul>
 * 
 * @author tnguyen
 */
public class ChainonRecalculeProfil3 extends ChainonSimple
{
	/**
	 * Tableau de correspondances code/profils du mode 1.
	 */
	private static final Map MODE1 = getTableauFaible();

	/**
	 * Tableau de correspondances code/profils du mode 2.
	 */
	private static final Map MODE2 = getTableauFort();

	/**
	 * Tableau de correspondances code/profils du mode 3.
	 */
	private static final Map MODE3 = getTableauMode3();

	/**
	 * Le seuil de différenciation des valeurs.
	 * <p>
	 * Paramètrable par l'utilisateur.
	 */
	private final float seuilCodeBiChar;

	/**
	 * Stocke les mesures de la ligne en cours le temps du traitement.
	 */
	private ListeMesures2 ligneEnCours;

	/**
	 * La valeur de la première mesure.
	 */
	private float fluo_Hp;

	/**
	 * La valeur de la seconde mesure.
	 */
	private float fluo_Hm;

	/**
	 * Valeur de la troisième mesure.
	 */
	private float fluo_N;

	/**
	 * Constructeur.
	 * 
	 * @param seuilCodeBiChar seuil à partir duquel on considère que deux
	 *            valeurs sont différentes (par ex. 0.3)
	 */
	public ChainonRecalculeProfil3(float seuilCodeBiChar)
	{
		this.seuilCodeBiChar = seuilCodeBiChar;
	}

	public void startLigne(ListeMesures l)
	{
		assert l != null;

		this.ligneEnCours = new ListeMesures2(l);

		// bloque les évènements entrants en attendant d'avoir accumulé toutes
		// les mesures
		this.fluo_Hp = 0f;
		this.fluo_Hm = 0f;
		this.fluo_N = 0f;
	}

	public void mesure(Mesure m)
	{
		if ("Hp".equals(m.getConditions())) this.fluo_Hp = m.getFluo();
		else if ("Hm".equals(m.getConditions())) this.fluo_Hm = m.getFluo();
		else if ("N".equals(m.getConditions())) this.fluo_N = m.getFluo();

		//stocke les mesures
		this.ligneEnCours.add(m);
	}

	public void stopLigne()
	{
		//cherche la fluo max
		float max = Float.MIN_VALUE;
		if (this.fluo_Hp > max) max = this.fluo_Hp;
		if (this.fluo_Hm > max) max = this.fluo_Hm;
		if (this.fluo_N > max) max = this.fluo_N;

		// détermine quel mode et quel tableau de correspondace utiliser
		// et calcule le code bichar au passage.
		String code;
		Map correspondances;
		if (this.fluo_Hp == 0f || this.fluo_N == 0f)
		{
			//on a un zéro en extrémité -> mode 2
			code = calculeCodeBichar(this.fluo_Hp / max, this.fluo_Hm / max,
					this.fluo_N / max);
			correspondances = MODE2;
		}
		else if (this.fluo_Hm == 0f)
		{
			//on a un zéro au milieu -> mode 3
			code = calculeCodeTrichar(this.fluo_Hp / max, this.fluo_Hm / max,
					this.fluo_N / max);
			correspondances = MODE3;
		}
		else
		{
			//aucun zéro -> mode 1
			code = calculeCodeBichar(this.fluo_Hp / max, this.fluo_Hm / max,
					this.fluo_N / max);
			correspondances = MODE1;
		}

		//cherche les profils compatibles
		String profilsCompatibles = (String) correspondances.get(code);

		//modifie le profil
		this.ligneEnCours.setProfil(profilsCompatibles);
		this.ligneEnCours.setCodeBiChar(code);
		super.startLigne(this.ligneEnCours);

		// "libère" toutes les mesures mémorisées
		Iterator im = this.ligneEnCours.iterator();
		while (im.hasNext())
		{
			super.mesure((Mesure) im.next());
		}

		super.stopLigne();

		this.ligneEnCours = null;
	}

	/**
	 * Calcule le code caractère à partir des deux valeurs normalisées.
	 * 
	 * @param mesure1
	 * @param mesure2
	 * @return le code bicaractère
	 */
	private String calculeCodeMonochar(float mesure1, float mesure2)
	{
		assert 0f <= mesure1 && mesure1 <= 1f;
		assert 0f <= mesure2 && mesure2 <= 1f;

		return (mesure1 >= mesure2 + this.seuilCodeBiChar) ? "-" : (mesure1
				+ this.seuilCodeBiChar <= mesure2) ? "+" : "=";
	}

	/**
	 * Calcule le code bicaractère à partir des trois valeurs normalisées.
	 * 
	 * @param mesure1
	 * @param mesure2
	 * @param mesure3
	 * @return le code bicaractère
	 */
	private String calculeCodeBichar(float mesure1, float mesure2, float mesure3)
	{
		assert 0f <= mesure1 && mesure1 <= 1f;
		assert 0f <= mesure2 && mesure2 <= 1f;
		assert 0f <= mesure3 && mesure3 <= 1f;

		return calculeCodeMonochar(mesure1, mesure2)
				+ calculeCodeMonochar(mesure2, mesure3);
	}

	/**
	 * Calcule le code tricaractère à partir des trois valeurs normalisées.
	 * 
	 * @param mesure1
	 * @param mesure2
	 * @param mesure3
	 * @return le code bicaractère
	 */
	private String calculeCodeTrichar(float mesure1, float mesure2,
			float mesure3)
	{
		assert 0f <= mesure1 && mesure1 <= 1f;
		assert 0f <= mesure2 && mesure2 <= 1f;
		assert 0f <= mesure3 && mesure3 <= 1f;

		return calculeCodeBichar(mesure1, mesure2, mesure3) + "("
				+ calculeCodeMonochar(mesure1, mesure3) + ")";
	}

	public String toString()
	{
		return "{RecalculeProfil2: seuil=" + this.seuilCodeBiChar + "}";
	}

	/**
	 * initialisation du tableau du mode 1
	 */
	private static Map getTableauFaible()
	{
		//
		// initialisation du tableau du mode 1
		//
		Map m = new HashMap();

		m.put("--", "\\");
		m.put("-=", "\\ (-)");
		m.put("-+", "V");
		m.put("=-", "\\ (-)");
		m.put("==", "-");
		m.put("=+", "/ (-)");
		m.put("+-", "A");
		m.put("+=", "/ (-)");
		m.put("++", "/");

		return Collections.unmodifiableMap(m);
	}

	/**
	 * initialisation du tableau du mode 2
	 */
	private static Map getTableauFort()
	{
		Map m = new HashMap();

		m.put("--", "\\");
		m.put("-=", "\\");
		m.put("-+", "V");
		m.put("=-", "\\");
		m.put("==", "-");
		m.put("=+", "/");
		m.put("+-", "A");
		m.put("+=", "/");
		m.put("++", "/");

		return Collections.unmodifiableMap(m);
	}

	/**
	 * initialisation du tableau du mode 3
	 */
	private static Map getTableauMode3()
	{
		Map m = new HashMap()
		{
			public Object get(Object key)
			{
				if (!((String) key).matches("[-=+][-=+]\\([-=+]\\)"))
						throw new IllegalArgumentException(key
								+ "n'est pas code tricaractère valide");

				Object val = super.get(key);
				return val == null ? "cas non prévu" : val;
			}
		};

		m.put("-=(-)", "\\");
		m.put("-=(=)", "\\");
		m.put("-=(+)", "cas impossible");

		m.put("-+(-)", "V (\\)");
		m.put("-+(=)", "V");
		m.put("-+(+)", "V (/)");

		m.put("==(-)", "cas impossible");
		m.put("==(=)", "cas impossible");
		m.put("==(+)", "cas impossible");

		m.put("=+(-)", "/");
		m.put("=+(+)", "/");
		m.put("=+(=)", "/");

		return Collections.unmodifiableMap(m);
	}

}