/*
 * Created on Apr 23, 2004
 *  
 */
package rfdd.chainons;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import rfdd.beans.GrandGroupe;
import rfdd.beans.ListeMesures;
import rfdd.beans.ListeMesures2;
import rfdd.beans.Mesure;
import rfdd.beans.PetitGroupe;
import rfdd.beans.PetitGroupe2;

/**
 * Sélectionne les profils compatibles avec le profil majoritaire dans chaque
 * <code>GrandGroupe</code>.
 * 
 * <p>
 * Dans chaque <code>GrandGroupe</code>, calcule le profil majoritaire. Si un
 * profil majoritaire est trouvé, le <code>GrandGroupe</code> est annoté avec
 * ce profil, et seules les lignes compatibles avec ce profil majoritaire sont
 * transmises au <code>Chainon</code> suivant. S'il n'y a pas de profil
 * majoritaire, toutes les lignes sont transmises, sans distinction.
 * 
 * <p>
 * NB: majoritaire veux dire strictement supérieur à 50% du total. Le total
 * étant le nombre de PG dans le GG. <br>
 * NB: les profils sont comptés une seule fois par <code>PetitGroupe</code>.
 * 
 * @author tnguyen
 *  
 */
public class ChainonSelectionneCompatibles extends ChainonSimple
{
	private GrandGroupe ggEnCours;

	/**
	 * PetitGroupe en cours.
	 * 
	 * <p>
	 * Vaut <code>null</code> en dehors des "évènements"
	 * <code>startPetitGroupe()</code> et <code>stopPetitGroupe()</code>.
	 */
	private PetitGroupe2 pgEnCours;

	/**
	 * Liste des petits groupes en cours de traitement.
	 */
	private List petitGroupes;

	private ListeMesures2 ligneEnCours;

	/**
	 * Utilisé pour comptabiliser les occurences de chaque profil dans un
	 * <code>GrandGroupe</code>.
	 * 
	 * <p>
	 * <code>Map&lt;profil, nombre&gt;</code>
	 */
	private Map compteurProfils;

	/**
	 * Utilisé pour comptabiliser les occurences de chaque profil dans un
	 * <code>PetitGroupe</code>.
	 * 
	 * <p>
	 * <code>List&lt;String&gt;</code>
	 */
	private Set compteurProfilsTemp;

	/**
	 * Compte le nombre de petits groupes dans le grand groupe.
	 */
	private int nbPG;

	public void startGrandGroupe(GrandGroupe gg)
	{
		assert this.compteurProfils == null;

		this.ggEnCours = gg;

		this.nbPG = 0;

		this.compteurProfils = new HashMap();
		this.petitGroupes = new LinkedList();
	}

	public void startPetitGroupe(PetitGroupe pg)
	{
		assert this.compteurProfilsTemp == null;

		this.compteurProfilsTemp = new HashSet();
		this.nbPG++;

		this.pgEnCours = new PetitGroupe2(pg);
	}

	public void startLigne(ListeMesures l)
	{
		assert this.ligneEnCours == null;

		//mémorise quelle est la ligne en cours
		this.ligneEnCours = new ListeMesures2(l);

		// mémorise les profils compatibles avec cette ligne
		this.compteurProfilsTemp.addAll(ProfilUtil.asList(this.ligneEnCours
				.getProfil()));
	}

	public void mesure(Mesure m)
	{
		// mémorise les mesures pour les passer plus tard
		this.ligneEnCours.add(m);
	}

	public void stopLigne()
	{
		assert this.ligneEnCours != null;
		assert this.pgEnCours != null;

		// mémorise cette ligne dans le pg en cours
		this.pgEnCours.add(this.ligneEnCours);

		this.ligneEnCours = null;
	}

	public void stopPetitGroupe()
	{
		assert this.pgEnCours != null;
		assert this.ligneEnCours == null;

		// on incrémente le compteur de profils des profils trouvés dans le PG
		Iterator i = compteurProfilsTemp.iterator();
		while (i.hasNext())
		{
			Object profil = i.next();

			//trouve la valeur courante
			Integer compte = (this.compteurProfils.containsKey(profil)) ? (Integer) this.compteurProfils
					.get(profil)
					: new Integer(0);

			//incrémente
			this.compteurProfils
					.put(profil, new Integer(compte.intValue() + 1));
		}

		// on mémorise le PG pour traitement futur
		this.petitGroupes.add(this.pgEnCours);
		this.pgEnCours = null;

		this.compteurProfilsTemp = null;
	}

	public void stopGrandGroupe()
	{
		assert this.ggEnCours != null;
		assert this.pgEnCours == null;

		/*
		 * si on veux changer la définition de la majorité, c'est ici.
		 * 
		 * S'il n'y a qu'un seul petit groupe dans le grand groupe en cours, il
		 * n'y a pas de majorité possible (d'où le majorité = MAX_VALUE).
		 */
		int majorite = (this.nbPG >= 2) ? this.nbPG / 2 : Integer.MAX_VALUE;

		// calcule le profil majoritaire
		StringBuffer profilsMajoritaire = new StringBuffer();

		List sortedEntries = new ArrayList(this.compteurProfils.entrySet());
		Collections.sort(sortedEntries, new Comparator()
		{
			public int compare(Object o1, Object o2)
			{
				Integer compte1 = (Integer) ((Map.Entry) o1).getValue();
				Integer compte2 = (Integer) ((Map.Entry) o2).getValue();
				int comp=compte1.intValue() - compte2.intValue();
				return - comp; // retourne l'opposé pour un tri descendant
			}

			public boolean equals(Object o1, Object o2)
			{
				return compare(o1, o2) == 0;
			}
		});

		Iterator ip = sortedEntries.iterator();
		while (ip.hasNext())
		{
			Map.Entry entry = (Map.Entry)ip.next();
			Character profil = (Character) entry.getKey();
			Integer compte = (Integer) entry.getValue();
			if (compte.intValue() > majorite)
			{
				profilsMajoritaire.append(profil).append("(").append(compte).append(") ");
			}
		}

		// annote le GrandGroupe avec le profil majoritaire
		this.ggEnCours.setProfilMajoritaire(profilsMajoritaire.toString());

		//passe le GG au chainon suivant
		super.startGrandGroupe(this.ggEnCours);

		Iterator ipg = this.petitGroupes.iterator();
		while (ipg.hasNext())
		{
			PetitGroupe2 pg = (PetitGroupe2) ipg.next();
			super.startPetitGroupe(pg);

			Iterator il = pg.iterator();
			while (il.hasNext())
			{
				ListeMesures2 lm = (ListeMesures2) il.next();

				super.startLigne(lm);

				Iterator im = lm.iterator();
				while (im.hasNext())
				{
					Mesure m = (Mesure) im.next();
					super.mesure(m);
				}

				super.stopLigne();
			}

			super.stopPetitGroupe();
		}

		super.stopGrandGroupe();
		this.compteurProfils = null;
	}

	public String toString()
	{
		return "{ChainonSelectionneCompatibles}";
	}

}