/*
 * Created on Apr 5, 2004
 */

package rfdd;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Insets;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;

import rfdd.chainons.LecteurFichier;
import rfdd.chainons.Pipeline;
import rfdd.chainons.SortieFichier;
import rfdd.gui.BadParamException;
import rfdd.gui.ExecutePipelineAction;
import rfdd.gui.FileBox;
import rfdd.gui.FilePrettyfier;
import rfdd.gui.WidgetChainonAppariement;
import rfdd.gui.WidgetChainonCalibration;
import rfdd.gui.WidgetChainonProfilEnClair;
import rfdd.gui.WidgetChainonRealigneur;
import rfdd.gui.WidgetChainonRecalculeProfil;
import rfdd.gui.WidgetChainonRecalculeTailleRef;
import rfdd.gui.WidgetChainonSelectionneCompatibles;

/**
 * Classe principale et point d'entrée dans l'interface graphique du programme.
 * 
 * @author tnguyen
 */
public class GUI extends JFrame
{
	private FileBox inputFileComponent;
	private WidgetChainonRealigneur chainonRealigneurComponent;
	private WidgetChainonRecalculeTailleRef chainonRecalculeTailleRefComponent;
	private WidgetChainonCalibration chainonCalibrationComponent;
	private WidgetChainonRecalculeProfil chainonRecalculeProfilComponent;
	private WidgetChainonAppariement chainonAppariementComponent;
	private WidgetChainonSelectionneCompatibles chainonSelectionneCompatiblesComponent;
	private WidgetChainonProfilEnClair chainonProfilEnClairComponent;
	private FileBox outputFileComponent;

	/**
	 * Point d'entrée dans l'application.
	 * 
	 * @param args
	 */
	public static void main(String[] args)
	{
		SwingUtilities.invokeLater(new Runnable(){

			public void run()
			{
				GUI gui = new GUI();

				gui.setDefaultCloseOperation(EXIT_ON_CLOSE);

				gui.pack();
				gui.setVisible(true);
			}});
	}

	/**
	 * Constructeur.
	 *
	 */
	public GUI()
	{
		this.inputFileComponent = new FileBox("Fichier en entrée", this, FilePrettyfier.READONLY);
		this.chainonRealigneurComponent = new WidgetChainonRealigneur();
		this.chainonRecalculeTailleRefComponent = new WidgetChainonRecalculeTailleRef();
		this.chainonCalibrationComponent = new WidgetChainonCalibration();
		this.chainonRecalculeProfilComponent = new WidgetChainonRecalculeProfil();
		this.chainonAppariementComponent = new WidgetChainonAppariement();
		this.chainonSelectionneCompatiblesComponent = new WidgetChainonSelectionneCompatibles();
		this.chainonProfilEnClairComponent = new WidgetChainonProfilEnClair();
		this.outputFileComponent = new FileBox("Fichier de sortie", this, FilePrettyfier.WRITE);

		setTitle("Réorganisaton + analyse RFDD");

		JPanel contentPane = new JPanel();
		contentPane.setLayout(new BorderLayout());
		contentPane.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

		// petit panneau de présentation de l'appli
		JTextPane haut = new JTextPane();
		haut.setBorder(BorderFactory.createEmptyBorder(20, 20, 20, 20));
		haut.setContentType("text/html");
		try
		{
			haut.setPage(this.getClass().getResource("/description.html"));
		}
		catch (IOException e)
		{
			throw new RuntimeException("ressource description.html introuvable", e);
		}
		haut.setMargin(new Insets(10, 10, 10, 10));
		haut.setEditable(false);

		// panneau d'affichage du "pipeline"
		Box pipelineBox = Box.createVerticalBox();
		pipelineBox.add(this.inputFileComponent);
		pipelineBox.add(this.chainonRealigneurComponent);
		pipelineBox.add(this.chainonRecalculeTailleRefComponent);
		pipelineBox.add(this.chainonCalibrationComponent);
		pipelineBox.add(this.chainonRecalculeProfilComponent);
		pipelineBox.add(this.chainonAppariementComponent);
		pipelineBox.add(this.chainonSelectionneCompatiblesComponent);
		pipelineBox.add(this.chainonProfilEnClairComponent);
		pipelineBox.add(this.outputFileComponent);

		// affiche juste le bouton démarrer
		JPanel bas = new JPanel();
		bas.setLayout(new FlowLayout());
		JButton b = new JButton(new ExecutePipelineAction(this));
		bas.add(b);

		contentPane.add(haut, BorderLayout.WEST);
		contentPane.add(pipelineBox, BorderLayout.CENTER);
		contentPane.add(bas, BorderLayout.SOUTH);
		setContentPane(contentPane);
	}

	/**
	 * Construit un <code>Pipeline</code> avec les composant actifs et les paramètres choisis par l'utilisateur.
	 * 
	 * @return Pipeline
	 * @throws BadParamException si un paramètre est invalide
	 */
	public Pipeline buildPipeline() throws BadParamException
	{
		Pipeline pipeline = new Pipeline();

		// premier étage du pipeline: lecture du fichier en entrée
		File inFile = this.inputFileComponent.getFile();
		if (inFile.isDirectory() || !inFile.canRead())
			throw new BadParamException(
				"Impossible de prendre " + inFile.getName() + " comme fichier d'entrée");
		LecteurFichier entree;
		try
		{
			entree = new LecteurFichier(inFile);
		}
		catch (FileNotFoundException fnfe)
		{
			throw new BadParamException("Problème à l'ouverture de " + inFile.getName(), fnfe);
		}
		pipeline.add(entree);

		// deuxième étage du pipeline: réalignement
		if (this.chainonRealigneurComponent.actif())
			pipeline.add(this.chainonRealigneurComponent.getChainon());

		// troisième étage du pipeline: recalcule taille ref
		if (this.chainonRecalculeTailleRefComponent.actif())
			pipeline.add(this.chainonRecalculeTailleRefComponent.getChainon());

		// quatrième étage du pipeline: calibration
		if (this.chainonCalibrationComponent.actif())
			pipeline.add(this.chainonCalibrationComponent.getChainon());

		//DEBUG
		//pipeline.add(new ChainonDebug(new PrintWriter(System.err, true), "<"));

		// cinquième étage du pipeline: calcul des profils de réponse
		if (this.chainonRecalculeProfilComponent.actif())
			pipeline.add(this.chainonRecalculeProfilComponent.getChainon());

		// sixième étage du pipeline: appariement des extrémités d'un même fragment
		if (this.chainonAppariementComponent.actif())
			pipeline.add(this.chainonAppariementComponent.getChainon());

		// septième étage du pipeline: repérage des profils compatibles
		if (this.chainonSelectionneCompatiblesComponent.actif())
			pipeline.add(this.chainonSelectionneCompatiblesComponent.getChainon());
		
		// huitième étage du pipeline: affichage en clair du profil le plus probable
		if (this.chainonProfilEnClairComponent.actif())
			pipeline.add(this.chainonProfilEnClairComponent.getChainon());
		
		// dernier étage du pipeline: sortie fichier
		File outFile = this.outputFileComponent.getFile();
		if (outFile.exists() && (outFile.isDirectory() || !outFile.canWrite()))
			throw new BadParamException(
				"Impossible de prendre " + outFile.getName() + " comme fichier de sortie");
		SortieFichier sortie;
		try
		{
			sortie = new SortieFichier(outFile, "ISO-8859-1");
		}
		catch (IOException ioe)
		{
			throw new BadParamException("Problème à l'ouverture de " + outFile.getName(), ioe);
		}
		pipeline.add(sortie);

		return pipeline;
	}

}
