/*
 * Created on Apr 4, 2004
 */
package rfdd.chainons;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import rfdd.beans.ListeMesures;
import rfdd.beans.Mesure;
import rfdd.beans.PetitGroupe;


/**
 * Réaligne les mesures petit groupe par petit groupe.
 * 
 * <p>Opère en stockant toutes les mesures colonnes par colonne,
 * Puis en créant les nouvelles lignes à partir de rien.
 * 
 * <p>Pour chaque ligne, on prend la mesure la plus petite
 * et celles qui sont suffisamment proches* dans les autres colonnes.
 * <br>Les mesures qui ne sont pas suffisamment proches sont gardées pour
 * la ligne suivante (ou la suivante ou encore celle d'après ;) )
 * 
 * <p>* <i>La notion de proximité est paramètrée par <code>tolerance</code>.</i>
 *  
 * @author tnguyen
 *
 */
public class ChainonRealigneur extends ChainonSimple
{
	/**
	 * Le paramètre qui permet de distinguer les mesures qui vont --ou non-- dans la même ligne.
	 */
	private final float tolerance;

	/**
	 * Mémorise toues les <code>Mesure</code>s, colonne par colonne.
	 * <p><code>Map</code> de <code>List</code> de <code>Mesure</code>.
	 */
	private Map colonnes;

	/**
	 * Constructeur.
	 * 
	 * @param tolerance
	 */
	public ChainonRealigneur(float tolerance)
	{
		this.tolerance = tolerance;
	}

	/**
	 * Commence le stockage des mesures.
	 */
	public void startPetitGroupe(PetitGroupe pg)
	{
		super.startPetitGroupe(pg);
		this.colonnes = new HashMap();
	}

	/**
	 * Aligne les mesures stockées et transmet ces nouvelles lignes au chainon suivant.
	 */
	public void stopPetitGroupe()
	{
		// tant qu'on a des mesures à mettre dans des lignes
		while (!this.colonnes.isEmpty())
		{
			//commence une ligne
			ListeMesures liste = new ListeMesures();
			liste.setProfil("N/C"); // profil non connu / non calculé
			liste.setTailleRef(Float.NaN); // Nan = Not a Number -> valeur invalide
			super.startLigne(liste);

			// cherche la taille minimum parmis les premières mesures de chaque colonne.
			float tailleMin = chercheTailleMin();

			// extrait les mesures dans l'intervalle de tolerance
			extraitMesuresTolerees(tailleMin);

			// termine la ligne
			super.stopLigne();

		}

		// Toutes les mesures ont été alignes dans des lignes
		super.stopPetitGroupe();
	}

	/**
	 * Parcours les premières mesures de chaque colonne à la recherche de mesures dans l'intervalle de tolérance.
	 * 
	 * <p>Les mesures trouvées sont enlevées de leur colonnes et transmises au chainon suivant.
	 * 
	 * 
	 * @param tailleMin
	 */
	private void extraitMesuresTolerees(float tailleMin)
	{
		Iterator it2 = new LinkedHashSet(this.colonnes.entrySet()).iterator();
		while (it2.hasNext())
		{
			Map.Entry entry = (Map.Entry) it2.next();
			String colName = (String) entry.getKey();
			List colonne = (List) entry.getValue();
			Mesure m = (Mesure) colonne.get(0); //la première mesure de la colonne

			if (m.getTaille() - tailleMin <= 2 * this.tolerance)
			{
				// on a trouvé une mesure qui rentre dans l'intervalle de tolérance

				//
				// transmet la mesure au chainon suivant
				//
				super.mesure(m);

				//on enlève cette mesure de la liste des mesures a aligner
				colonne.remove(m);

				//on enlève la colonne aussi si elle est vide				
				if (colonne.isEmpty())
				{
					// si on a vidé la colonne, on l'enlève de la liste des colonnes
					this.colonnes.remove(colName);
				}

			}
		}
	}

	private float chercheTailleMin()
	{
		//
		// cherche la mesure avec la taille minimum
		//
		float tailleMin = Float.MAX_VALUE;
		Iterator it1 = this.colonnes.values().iterator();
		while (it1.hasNext())
		{
			List colonne = (List) it1.next();
			Mesure m = (Mesure) colonne.get(0); //la première mesure de la colonne

			if (m.getTaille() < tailleMin)
			{
				// on a trouvé une mesure candidate,
				// on mémorise sa taille
				tailleMin = m.getTaille();
			}
		}

		return tailleMin;
	}

	/**
	 * Bloque le passage au chainon suivant.
	 */
	public void startLigne(ListeMesures l)
	{
		;
	}

	/**
	 * Bloque le passage au chainon suivant.
	 */
	public void stopLigne()
	{
		;
	}

	/**
	 * Stocke la mesure pour la réaligner plus tard.
	 */
	public void mesure(Mesure m)
	{
		List colonne = (List) this.colonnes.get(m.getConditions());
		if (colonne == null)
		{
			colonne = new LinkedList();
			this.colonnes.put(m.getConditions(), colonne);
		}

		colonne.add(m);
	}

	public String toString()
	{
		return "{Realigneur: tolerance=" + this.tolerance + "}";
	}

}
