/*
 * Created on Apr 4, 2004
 */
package rfdd.chainons;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import rfdd.beans.ListeMesures;
import rfdd.beans.ListeMesures2;
import rfdd.beans.Mesure;
import rfdd.beans.Profil;

/**
 * Recalcule le profil de chaque ligne.
 * 
 * <p>Opère en deux étapes:
 * <ul>
 * <li>Calcule le "code bi-caractère" associé au triplet de mesures.
 * <li>Associe un ou plusieurs profils à chaque code bi-caractère.
 * </ul>
 * 
 * <h3>Calcul du code bi-caractère</h3>
 * Le premier caratère représante la "distance" entre le première et la deuxième mesure.
 * <br>Le deuxième caractère représente la "distance" entre la deuxième et la troisème mesure.
 * 
 * <p>NB: les mesures sont dans l'ordre Hp, Hm, N.
 * 
 * <p> On commance par "normaliser" les trois mesures, c-à-d les diviser par la valeur de la plus grande.
 * Ainsi on a que des valeurs entre 0 et 1.
 * 
 * <p>Pour le calcul d'un caractère, on considère que deux valeurs sont similaires (caractère '=')
 * si leur différence n'excède pas 0.3 (valeur paramètrable par l'utilisateur).
 * <br>Si la première mesure est plus grande que la seconde, on utilise le caractère '-'.
 * <br>Sinon on met un '+' (seconde mesure > première mesure).
 * 
 * <h3>Calcul des profils associés</h3>
 * On utilise un tableau pré-défini pour associer un ou plusieurs profils à chaque code bi-caractère.
 * <br>(flemme de recopier le tableau ici, faut aller lire le code, dsl)
 * 
 * <p></p><p></p>
 * <p>Pour chaque ligne en entrée, on transmet en sortie autant de copies de la ligne que de profils trouvés
 * (seul le profil change d'une ligne sur l'autre).
 * 
 * <p>NB: On peut se retrouver potentiellement avec beaucoup de redondance.
 * 
 * @author tnguyen
 */
public class ChainonRecalculeProfil extends ChainonSimple
{
	/**
	 * Le seuil de différenciation des valeurs.
	 * <p>Paramètrable par l'utilisateur.
	 */
	private final float seuilCodeBiChar;

	/**
	 * Le tableau de correspondance entre les codes bicaractères et les profils.
	 * <p>Codé en dur dans l'appli.
	 */
	private final Map correspondances;

	/**
	 * Stocke les mesures de la ligne en cours le temps du traitement.
	 */
	private ListeMesures2 ligneEnCours;

	/**
	 * La valeur de la première mesure.
	 */
	private float fluo_Hp;

	/**
	 * La valeur de la seconde mesure.
	 */
	private float fluo_Hm;

	/**
	 * Valeur de la troisième mesure.
	 */
	private float fluo_N;

	/**
	 * Constructeur.
	 * 
	 * @param seuilCodeBiChar seuil à partir duquel on considère que deux
	 *            valeurs sont différentes (par ex. 0.3)
	 */
	public ChainonRecalculeProfil(float seuilCodeBiChar)
	{
		this.seuilCodeBiChar = seuilCodeBiChar;

		this.correspondances = new HashMap();
		correspondances.put("--", Arrays.asList(new String[] { Profil.ACTIVE_PAR_HYPOXIE }));
		correspondances.put(
			"-=",
			Arrays.asList(
				new String[] {
					Profil.ACTIVE_PAR_HYPOXIE,
					Profil.PE_INHIBE_PAR_HM,
					Profil.PE_NON_MODIFIE_PAR_O2 }));
		correspondances.put("-+", Arrays.asList(new String[] { Profil.INHIBE_PAR_HM }));
		correspondances.put(
			"=-",
			Arrays.asList(
				new String[] {
					Profil.ACTIVE_PAR_HM,
					Profil.PE_NON_MODIFIE_PAR_O2,
					Profil.PE_ACTIVE_PAR_HYPOXIE }));
		correspondances.put("==", Arrays.asList(new String[] { Profil.NON_MODIFIE_PAR_O2 }));
		correspondances.put(
			"=+",
			Arrays.asList(
				new String[] {
					Profil.PE_INHIBE_PAR_HM,
					Profil.ACTIVE_PAR_NORMOXIE,
					Profil.PE_NON_MODIFIE_PAR_O2 }));
		correspondances.put("+-", Arrays.asList(new String[] { Profil.ACTIVE_PAR_HM }));
		correspondances.put(
			"+=",
			Arrays.asList(
				new String[] {
					Profil.PE_ACTIVE_PAR_NORMOXIE,
					Profil.PE_NON_MODIFIE_PAR_O2,
					Profil.ACTIVE_PAR_HM }));
		correspondances.put("++", Arrays.asList(new String[] { Profil.ACTIVE_PAR_NORMOXIE }));
	}

	public void startLigne(ListeMesures l)
	{
		assert l != null;

		this.ligneEnCours = new ListeMesures2(l);

		// bloque les évènements entrants en attendant d'avoir accumulé toutes
		// les mesures
		this.fluo_Hp = 0f;
		this.fluo_Hm = 0f;
		this.fluo_N = 0f;
	}

	public void mesure(Mesure m)
	{
		if ("Hp".equals(m.getConditions()))
			this.fluo_Hp = m.getFluo();
		else if ("Hm".equals(m.getConditions()))
			this.fluo_Hm = m.getFluo();
		else if ("N".equals(m.getConditions()))
			this.fluo_N = m.getFluo();

		//stocke les mesures
		this.ligneEnCours.add(m);
	}

	public void stopLigne()
	{
		//cherche la fluo max
		float max = Float.MIN_VALUE;
		if (this.fluo_Hp > max)
			max = this.fluo_Hp;
		if (this.fluo_Hm > max)
			max = this.fluo_Hm;
		if (this.fluo_N > max)
			max = this.fluo_N;

		//calcule le code bicaractère sur les valeurs nomalisées
		String codeBiChar =
			calculeCodeBichar(this.fluo_Hp / max, this.fluo_Hm / max, this.fluo_N / max);

		//cherche les profils compatibles
		List profilsCompatibles = (List) this.correspondances.get(codeBiChar);

		// pour chaque profil compatible, crée une copie de la ligne en cours
		Iterator ip = profilsCompatibles.iterator();
		while (ip.hasNext())
		{
			String profil = (String) ip.next();

			//modifie le profil
			ListeMesures lm = new ListeMesures(this.ligneEnCours);
			lm.setProfil(profil);
			lm.setCodeBiChar(codeBiChar);
			super.startLigne(lm);

			// "libère" toutes les mesures mémorisées
			Iterator im = this.ligneEnCours.iterator();
			while (im.hasNext())
			{
				super.mesure((Mesure) im.next());
			}

			super.stopLigne();
		}
		this.ligneEnCours = null;
	}

	/**
	 * Calcule le code bicaractère à partir des trois valeurs normalisées.
	 * 
	 * @param mesure1
	 * @param mesure2
	 * @param mesure3
	 * 
	 * @return le code bicaractère
	 */
	private String calculeCodeBichar(float mesure1, float mesure2, float mesure3)
	{
		assert 0f <= mesure1 && mesure1 <= 1f;
		assert 0f <= mesure2 && mesure2 <= 1f;
		assert 0f <= mesure3 && mesure3 <= 1f;

		//calcule premier caractère
		String code =
			(mesure1 >= mesure2 + this.seuilCodeBiChar)
				? "-"
				: (mesure1 + this.seuilCodeBiChar <= mesure2)
				? "+"
				: "=";

		// calcule le deuxième caractère
		code += (mesure2 >= mesure3 + this.seuilCodeBiChar)
			? "-"
			: (mesure2 + this.seuilCodeBiChar <= mesure3)
			? "+"
			: "=";

		return code;
	}

	public String toString()
	{
		return "{RecalculeProfil: seuil=" + this.seuilCodeBiChar + "}";
	}

}