#include<ODESolver.hpp>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_linalg.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_permutation.h>
#include <gsl/gsl_odeiv.h>
#include <gsl/gsl_multifit_nlin.h>
#include <gsl/gsl_randist.h>
#include <iostream>

using namespace std;

// functional response parameters
const double a0 = 10;                   // attack rate intercept if both consumer and resource are mobile
const double a0_sessile_res = 10;        // attack rate intercept if resource is sessile
const double a0_sessile_cons = 10;       // attack rate intercept if consumer is sessile
const double alpha_i = 0.47;            // exponent of consumer body mass scaling of attack rate
const double alpha_j = 0.15;            // exponent of resource body mass scaling of attack rate
const double h0 = 0.4;                  // handling time intercept
const double h_i = -0.48;		// exponent of consumer body mass scaling of handling time
const double h_j = -0.66;		// exponent of resource body mass scaling of handling time
double half_sat;			// half saturation density; actually not needed anymore (only for the next line)

// additional population dynamics parameters
const double x_plant = 0.138;		// respiration rate intercept for plants
const double x_invert = 0.314;          // respiration rate intercept for invertebrates
const double mu0 = 0.1;		        // strength of mortality relative to respiration
const double e_plant = 0.45;		// assimilation efficiency for plant resources
const double e_animal = 0.85;		// assimilation efficiency for animal resources

// parameters for basal species and nutrient dynamics
const double K0 = 1;			// intercept for carrying capacity
const double K_min = 0.1;		// minimum value for nutrient uptake half saturation density
const double K_max = 0.1;		// maximum value for nutrient uptake half saturation density
const double D_res = 0.2;			// nutrient turnover rate
const double S_res = 1;			// nutrient supply concentration

const double extinctthres = 1e-6; //1e-12;		// extinction threshold

int dynamics(double t, const double B[], double Bdot[], void *params);

ODESolver::ODESolver(unsigned int nbspecies)
{
  _nbspecies = nbspecies;
  _biomass= new double[_nbspecies+1];
}

ODESolver::~ODESolver ()
{
  if(_biomass) delete[](_biomass);
}

void ODESolver::initializeBiomasses(gsl_vector* biomass){
  if(biomass){
      for(unsigned int i=0; i<_nbspecies; i++)
	_biomass[i] = gsl_vector_get(biomass,i);
  } else{
      for(unsigned int i=0; i<_nbspecies; i++)
	//_biomass[i] = 1;
	_biomass[i] = 0.05 + (gsl_rng_uniform(RNGSingleton::get().rng()))*0.95;
  }
  _biomass[_nbspecies] = S_res;
}

void ODESolver::run(TrophicMatrix& TImat,
	       InterferenceMatrix& Imat,
	       PositiveMatrix& Pmat,
	       NegativeMatrix& Nmat,
	       RefugeMatrix& Rmat,
	       EstablishmentMatrix& Emat,
	       CompetitionMatrix& Cmat,
	       ParameterVector& basalvec,
	       ParameterVector& sessilevec,
	       ParameterVector& massvec,
	       int tend, int teval,
	       double extinctthres,
	       double hill,
	       bool islogistic,
	       bool verbose, bool isfirst){
  // mean biomass densities
  double *meanbiomass = new double[_nbspecies];
  for(unsigned int i=0; i<_nbspecies; i++) meanbiomass[i] = 0.;
  // mean biomass densities (for comparison)
  double *meanbiomass2 = new double[_nbspecies];
  for(unsigned int i=0; i<_nbspecies; i++) meanbiomass2[i] = 0.;
  // variances of biomasses
  double *varbiomass = new double[_nbspecies];
  for(unsigned int i=0; i<_nbspecies; i++) varbiomass[i] = 0.;
  // coefficients of variation
  double *varcoeffbiomass = new double[_nbspecies];
  for(unsigned int i=0; i<_nbspecies; i++) varcoeffbiomass[i] = 0.;

  //	*********** define parameters  ************
  gsl_vector *cs=gsl_vector_calloc(_nbspecies);							// competition for space
  gsl_vector *x_resp=gsl_vector_calloc(_nbspecies);						        // respiration rates
  gsl_vector *mort=gsl_vector_calloc(_nbspecies);						        // mortality rates
  gsl_vector *eff=gsl_vector_calloc(_nbspecies);    						        // assimilation efficiencies
  gsl_vector *r_plant=gsl_vector_calloc(_nbspecies);    					        // intrinsic growth rate of plants
  gsl_vector *k_plant=gsl_vector_calloc(_nbspecies);    					        // carrying capacities or nutrient uptake half saturation densities for plants

  //********** attack rates ***********
  gsl_matrix *A=gsl_matrix_calloc(_nbspecies,_nbspecies);							// attack rates
  for(unsigned int i=0; i<_nbspecies; i++){
      double m_i = gsl_vector_get(massvec.data(),i);
      double ses_i = gsl_vector_get(sessilevec.data(),i);
      for(unsigned int j=0; j<_nbspecies; j++){
	  double m_j = gsl_vector_get(massvec.data(),j);
	  double ses_j = gsl_vector_get(sessilevec.data(),j);
	  if(ses_i == 0 && ses_j == 1)                                // sessile resource, mobile consumer
	    gsl_matrix_set(A,i,j,a0_sessile_res*pow(m_i,alpha_i));
	  if(ses_i == 1 && ses_j == 0)                                // sessile consumer, mobile resource
	    gsl_matrix_set(A,i,j,a0_sessile_cons*pow(m_j,alpha_j));
	  if(ses_i == 0 && ses_j == 0)                                // mobile consumer and resource
	    gsl_matrix_set(A,i,j,a0*pow(m_i,alpha_i)*pow(m_j,alpha_j));
	  if(ses_i == 1 && ses_j == 1)
		gsl_matrix_set(A,i,j,a0_sessile_res);
      }
  }
  gsl_matrix_mul_elements(A,TImat.data());      // A <- A* A_p; set attack rates to 0 if there is no link

  for(unsigned int i=0; i<_nbspecies; i++){
      gsl_vector_view tempp1=gsl_matrix_row(A,i);  // Stores the row of all the attack rates of species i on its prey
      gsl_vector_view tempp2=gsl_matrix_row(TImat.data(),i); // Stores the row of all the prey of species i
      double temp1=gsl_blas_dasum(&tempp2.vector); // temp1 contains the number of prey of species i
      if(temp1)
	gsl_vector_scale(&tempp1.vector,1/temp1);    // attack rates = aij / no. of prey of sp i
  }
  if(verbose){
      cerr<<"A:\n";
      for(unsigned int i=0; i<_nbspecies; i++){
	  for(unsigned int j=0; j<_nbspecies; j++){
	      cerr<<gsl_matrix_get(A,i,j)<<" ";
	  }
	  cerr<<endl;
      }
  }

  //*********** handling times ***********
  gsl_matrix *H=gsl_matrix_calloc(_nbspecies,_nbspecies);
  for(unsigned int i=0; i<_nbspecies; i++){
      double m_i = gsl_vector_get(massvec.data(),i);
      for(unsigned int j=0; j<_nbspecies; j++){
	  double m_j = gsl_vector_get(massvec.data(),j);
	  gsl_matrix_set(H,i,j,h0*pow(m_i,h_i)*pow(m_j,h_j));
      }
  }
  //It may not be necessary because h is always multiplied by A in the equations, but we can add:
  gsl_matrix_mul_elements(H,TImat.data()); // no sense to define a handling time between two species i and j between which there is no trophic link
  if(verbose){
      cerr<<"H:\n";
      for(unsigned int i=0; i<_nbspecies; i++){
	  for(unsigned int j=0; j<_nbspecies; j++){
	      cerr<<gsl_matrix_get(H,i,j)<<" ";
	  }
	  cerr<<endl;
      }
  }

  //*********** competition for space, respiration rates, mortality rates, assimilation efficiencies, plant growth rates **********
  for(unsigned int i=0; i<_nbspecies; i++){
      double m_i = gsl_vector_get(massvec.data(),i);
      double bas_i = gsl_vector_get(basalvec.data(),i);

      gsl_vector_set(cs,i,pow(m_i,2./3.));

      if(bas_i){
	  if(islogistic)
	    gsl_vector_set(x_resp,i,0);
	  else
	    gsl_vector_set(x_resp,i,x_plant*pow(m_i,-0.25));
	  gsl_vector_set(mort,i,mu0*x_plant*pow(m_i,-0.25));
	  gsl_vector_set(eff,i,e_plant);
	  gsl_vector_set(r_plant,i,pow(m_i,-0.25));		        // intrinsic growth rate: only for plants; intercept = 1

	  if(islogistic)
	    gsl_vector_set(k_plant,i,K0*pow(m_i,0.25));			// carrying capacity
	  else
	    gsl_vector_set(k_plant,i,gsl_ran_flat(RNGSingleton::get().rng(),K_min,K_max));	// nutrient uptake half saturation density
      }
      else{
	  gsl_vector_set(x_resp,i,x_invert*pow(m_i,-0.25));
	  gsl_vector_set(mort,i,mu0*x_invert*pow(m_i,-0.25));
	  gsl_vector_set(eff,i,e_animal);
	  gsl_vector_set(k_plant,i,1);				        // dummy value for consumers to avoid division by 0
      }
  }
  if(verbose){
      cerr<<"k_plant:\n";
      for(unsigned int i=0; i<_nbspecies; i++) cerr<<gsl_vector_get(k_plant,i)<<" ";
      cerr<<endl;
      cerr<<"x_resp:\n";
      for(unsigned int i=0; i<_nbspecies; i++) cerr<<gsl_vector_get(x_resp,i)<<" ";
      cerr<<endl;
      cerr<<"mort:\n";
      for(unsigned int i=0; i<_nbspecies; i++) cerr<<gsl_vector_get(mort,i)<<" ";
      cerr<<endl;
      cerr<<"eff:\n";
      for(unsigned int i=0; i<_nbspecies; i++) cerr<<gsl_vector_get(eff,i)<<" ";
      cerr<<endl;
  }

  //*********** ODE parameters required for the dynamics function
  ODEParameters odeparams =
      {_nbspecies, A,H,
	  Imat.data(),Pmat.data(),Nmat.data(),Rmat.data(),Emat.data(),Cmat.data(),
	  x_resp,mort,eff,r_plant,k_plant,
	  basalvec.data(),sessilevec.data(),massvec.data(),
	  cs,hill,islogistic,false};

  //	*********** solve ODE ***********
  // DGL-Solver: Runge-Kutta-Fehlberg
  const gsl_odeiv_step_type *Solv=gsl_odeiv_step_rkf45;//rk8pd

  // absolute und relative error tolerated
  gsl_odeiv_control *c=gsl_odeiv_control_y_new((extinctthres*0.1),(extinctthres*0.01));
  gsl_odeiv_step *s=gsl_odeiv_step_alloc(Solv,_nbspecies+1);
  gsl_odeiv_evolve *e=gsl_odeiv_evolve_alloc(_nbspecies+1);

  // calls the function dynamics which solves the dynamical equations
  gsl_odeiv_system sys={dynamics,NULL,(size_t)_nbspecies+1,&odeparams};

  double temp1,temp2;
  double t=0.; // starting time
  double h=1e-5; // initial timestep length

  //TODO (later): eventually, add some kind of stability condition (when steady state is reached, i.e. when for all i the biomass does not change much any more through time, then stop running the simulation)
  if(isfirst){
      cout<<"Time ";
      for(unsigned int i=0;i<_nbspecies;i++) cout<<TImat.indexToLabel(i)<<" ";
      cout<<endl;
  }
  double tprev = -1.;
  while(t<tend){
      int status=gsl_odeiv_evolve_apply(e,c,s,&sys,&t,tend,&h,_biomass);
      if(status!= GSL_SUCCESS)
	break;
      for(unsigned int i=0;i<_nbspecies;i++){
	  if(_biomass[i]<extinctthres) // species whose biomass is too small are set extinct
	    _biomass[i]=0;
      }
      if(t-tprev>0.1){ // to avoid large output
	  cout<<t<<" "<<*this<<endl;
	  tprev = t;
      }
  }
  removeSpecies(TImat);

  // ****** computing means ******
  double tau=t,tau_old=0,delta_tau;
  while(t<tend+teval){
      int status=gsl_odeiv_evolve_apply(e,c,s,&sys,&t,tend+teval,&h,_biomass);
      if(status!= GSL_SUCCESS)
	break;
      for(unsigned int i=0;i<_nbspecies;i++){
	  if(_biomass[i]<extinctthres) // species whose biomass is too small are set extinct
	    _biomass[i]=0;
      }
      if(t-tprev>0.1){ // to avoid large output
	  cout<<t<<" "<<*this<<endl;
      }
      tau_old=tau;
      tau=t;
      delta_tau=tau-tau_old;

      for(unsigned int i=0;i<_nbspecies;i++)
	meanbiomass[i]+=_biomass[i]*delta_tau;
  }
  removeSpecies(TImat);
  // mean biomass of each species (through time)
  cout<<t+teval<<" ";
  for(unsigned int i=0;i<_nbspecies;i++){
    meanbiomass[i]=meanbiomass[i]/teval;
    cout<<meanbiomass[i]<<" ";
  }
  cout<<endl;

  // ****** computing variance ******
  tau=t;
  gsl_vector *tempvec=gsl_vector_calloc(_nbspecies);
  gsl_vector_set_all(tempvec,1);
  while(t<tend+2*teval){
      int status=gsl_odeiv_evolve_apply(e,c,s,&sys,&t,tend+2*teval,&h,_biomass);
      if(status!= GSL_SUCCESS)
	break;
      for(unsigned int i=0;i<_nbspecies;i++){
	  if(_biomass[i]<extinctthres) // species whose biomass is too small are set extinct
	    _biomass[i]=0;
      }
      tau_old=tau;
      tau=t;
      double Delta_tau=tau-tau_old;
      for(unsigned int i=0;i<_nbspecies;i++){
	  varbiomass[i]+=(_biomass[i]-meanbiomass[i])*(_biomass[i]-meanbiomass[i])*Delta_tau;
	  meanbiomass2[i]+=_biomass[i]*Delta_tau;
      }
  }
  removeSpecies(TImat);
  temp1=0;
  temp2=0;
  for(unsigned int i=0;i<_nbspecies;i++){
      meanbiomass2[i]=meanbiomass2[i]/teval;
      temp1+=meanbiomass[i];
      temp2+=meanbiomass2[i];
  }
  // indicates potentially pathological coefficients of variation
  // double cv_flag=temp1-temp2;
  for(unsigned int i=0;i<_nbspecies;i++){
      varbiomass[i]=varbiomass[i]/teval;
      if(meanbiomass[i]>0)
	varcoeffbiomass[i]=sqrt(varbiomass[i])/meanbiomass[i];
      // coefficient of variation of the biomass of each species (through time)
      else
	varcoeffbiomass[i]=0;
  }

  if(verbose){
      odeparams.verbose = true;
      double* dbiomassdt = new double[_nbspecies+1];
      dynamics(tend+2*teval, _biomass, dbiomassdt, &odeparams);
      delete [] dbiomassdt;
  }

  //	*********** Free memories ***********
  gsl_odeiv_step_free(s);
  gsl_odeiv_control_free(c);
  gsl_odeiv_evolve_free(e);
  gsl_vector_free(tempvec);
  delete[](meanbiomass);
  delete[](meanbiomass2);
  delete[](varbiomass);
  delete[](varcoeffbiomass);

  gsl_matrix_free(A);
  gsl_matrix_free(H);
  gsl_vector_free(cs);
  gsl_vector_free(x_resp);
  gsl_vector_free(mort);
  gsl_vector_free(r_plant);
  gsl_vector_free(k_plant);
  gsl_vector_free(eff);
}

void ODESolver::removeSpecies(TrophicMatrix& TImat)
{
  double *TLmin= new double[_nbspecies];
  gsl_matrix *A=gsl_matrix_calloc(_nbspecies,_nbspecies);
  gsl_matrix_memcpy(A,TImat.data());
  bool flag=true;
  while(flag){
      for(unsigned int i=0; i<_nbspecies; i++){
	  if(_biomass[i] <= extinctthres){
	      _biomass[i]=0;
	      for(unsigned int j=0;j<_nbspecies;j++)
		gsl_matrix_set(A,j,i,0);
	      // remove out-going links from extinct species
	  }
      }
      flag = false;
      // determine species that are not connected anymore (they have TLmin == 0)
      minTrophicLevel(A,TLmin);
      for(unsigned int i=0; i<_nbspecies; i++){
	  if(TLmin[i] == 0 && _biomass[i] > 0){
	      _biomass[i]=0;
	      flag=true;
	  }
      }
  }
  gsl_matrix_free(A);
  delete[](TLmin);
}

int dynamics(double t, const double B[], double Bdot[], void *params)
{
  ODEParameters odeparams = *( (ODEParameters*)params );
  unsigned int nbspecies = odeparams.nbspecies;
  // attack rates (incl. feeding preferences)
  gsl_matrix *Amat = odeparams.A;
  // handling times
  gsl_matrix *Hmat = odeparams.H;
  // interference competition
  gsl_matrix *Imat=odeparams.I;
  // positive effects on mortality
  gsl_matrix *Pmat=odeparams.P;
  // negative effects on mortality
  gsl_matrix *Nmat=odeparams.N;
  // refuge matrix
  gsl_matrix *Rmat=odeparams.R;
  // establishment matrix
  gsl_matrix *Emat=odeparams.E;
  // matrix of competition for space
  gsl_matrix *Cmat = odeparams.C;
  // respiration rates
  gsl_vector *xvec=odeparams.x_resp;
  // mortality rates
  gsl_vector *mortvec=odeparams.mort;
  // assimilation efficiencies
  gsl_vector *effvec=odeparams.eff;
  // max. plant growth rates
  gsl_vector *rvec=odeparams.r_plant;
  // carrying capacities or nutrient uptake half saturation densities
  gsl_vector *kvec=odeparams.k_plant;
  // identifies basal species
  gsl_vector *basvec=odeparams.basal;
  // identifies sessile species
  gsl_vector *sesvec=odeparams.sessile;
  // body masses
  gsl_vector *masvec=odeparams.mas;
  // competition for space
  gsl_vector *csvec=odeparams.cs;
  // hill coefficient
  double hill=odeparams.hill;
  // logistic growth
  bool islogistic=odeparams.islogistic;
  // verbose
  bool verbose=odeparams.verbose;


  //	************
  gsl_vector *Tvec=gsl_vector_calloc(nbspecies+1);
  // copy all variables (biomass and resource densities)
  gsl_vector_const_view T_vec1=gsl_vector_const_view_array(B,nbspecies+1); // copy operation from double* to gsl_vector
  gsl_vector_memcpy(Tvec,&T_vec1.vector);


  gsl_vector_view B_vec=gsl_vector_subvector(Tvec,0,nbspecies);
  // biomasses (plants + animals)
  gsl_vector *Bvec=&B_vec.vector;

  gsl_vector_view Bdot_vec=gsl_vector_view_array(Bdot,nbspecies); // Bdot of length nspecies+1 with Bdot[nbspecies] the derivative for the nutrients
  // time derivatives for plant and animal species
  gsl_vector *Bdotvec=&Bdot_vec.vector; // Bdotvec is a subset of Bdot that only contrains the species (i.e. the nspecies first elements)


  //	************ some temporary vectors and matrices (Note: free memory at the end of the function!) ***********
  gsl_vector *Prey=gsl_vector_calloc(nbspecies);				       // prey biomasses
  gsl_vector *Theta=gsl_vector_calloc(nbspecies);				       // heavyside function for net biomass production

  gsl_vector *temp1vec=gsl_vector_calloc(nbspecies);
  gsl_vector *temp2vec=gsl_vector_calloc(nbspecies);
  gsl_vector *temp3vec=gsl_vector_calloc(nbspecies);
  gsl_vector *temp4vec=gsl_vector_calloc(nbspecies);
  gsl_vector *temp5vec=gsl_vector_calloc(nbspecies);
  gsl_vector *temp6vec=gsl_vector_calloc(nbspecies);

  gsl_matrix *temp1mat=gsl_matrix_calloc(nbspecies,nbspecies);

  //	************ in case of Type 3 functional response ***********
  gsl_vector_memcpy(Prey,Bvec);

  for(unsigned int i=0;i<nbspecies;i++){
      if(gsl_vector_get(Prey,i)<0)
	gsl_vector_set(Prey,i,0); // paranoia (this doesn't mean that you can safely remove these lines!)

      gsl_vector_set(Prey,i,pow(gsl_vector_get(Prey,i),hill));               // pey_i = Bi^hill
  }

  //	************* refuge provisioning *************
  gsl_blas_dgemv(CblasTrans,1,Rmat,Bvec,0,temp1vec);		         // temp1vec_i = sum_j R_ji*B_j
  gsl_vector_add_constant(temp1vec,1);                                   // temp1vec = 1 + sum_j R_ji*B_j
  gsl_vector_set_all(temp2vec,1);                                        // temp2vec = 1
  gsl_vector_div(temp2vec,temp1vec);			                 // temp2vec_i = 1 / (1 + sum_j R_ji*B_j)
  gsl_vector_mul(Prey,temp2vec);					 // net effect of refuge provisioning: reduction of prey biomass density

  //	************* calculate functional response *************
  gsl_matrix_memcpy(temp1mat,Amat);                                          // temp1mat = A_ij (copies all the matrix)
  gsl_matrix_mul_elements(temp1mat,Hmat);				       // temp1mat_ij = A_ij * H_ij
  gsl_vector_set_all(temp1vec,1);					       // the '1' in the denominator of the functional response
  gsl_blas_dgemv(CblasNoTrans,1,temp1mat,Prey,1,temp1vec);// adding to temp1vec the handling time term of the functional response

  gsl_blas_dgemv(CblasTrans,1,Imat,Bvec,1,temp1vec);                       // adding to temp1vec the interference competition term of the function response
  gsl_vector_mul(temp1vec,masvec);				             // scale denominator with body mass

  gsl_vector_memcpy(temp2vec,effvec);                                        // temp2vec = effvec_i
  gsl_vector_mul(temp2vec,Prey);                                             // temp2vec_i = effvec_i * Bi^hill/(1+ sum_k R_ki Bk)
  // e is the efficiency of the prey, not of the predator
  gsl_blas_dgemv(CblasNoTrans,1,Amat,temp2vec,0,temp3vec);                   // temp3vec_i = sum_j A_ij * (e_j * Prey_j)

  gsl_vector_div(temp3vec,temp1vec);                                          // temp3vec_i = temp3vec_i/temp1vec_i
  gsl_vector_sub(temp3vec,xvec);					      // temp3vec_i: per unit biomass net production rate of i

  gsl_vector_memcpy(temp2vec,Bvec);					       // copy predator biomass density
  gsl_vector_div(temp2vec,temp1vec);					       // divide by denominator of the functional response
  gsl_blas_dgemv(CblasTrans,1,Amat,temp2vec,0,temp1vec);                       // temp1vec_i: per unit biomass predation mortality of species i
  gsl_vector_mul(temp1vec,Prey);					       // temp1vec_i: total predation mortality of species i

  //	************* calculate positive and negative effects on mortality *************
  gsl_vector_set_all(temp2vec,1);
  gsl_blas_dgemv(CblasTrans,1,Nmat,Bvec,1,temp2vec);		               // negative effects on mortality
  gsl_vector_set_all(temp4vec,1);
  gsl_blas_dgemv(CblasTrans,1,Pmat,Bvec,1,temp4vec);                           // positive effects on mortalityity
  gsl_vector_div(temp2vec,temp4vec);					       // total effect on mortality
  gsl_vector_mul(temp2vec,mortvec);					       // scale with baseline mortality
  gsl_vector_mul(temp2vec,Bvec);					       // total background mortality

  //	************* calculate the effect of facilitation on recruitment *************
  gsl_vector_set_all(temp6vec,1);
  // Version of the code below with linear dependency
  gsl_blas_dgemv(CblasTrans,1,Emat,Bvec,1,temp6vec);                          	// temp6vec_i = 1 + sum_j E_ji B_j
  gsl_vector_mul(temp6vec,rvec);    						// temp6vec_i = r_i*(1 + sum_j E_ji B_j)

  //	************* include terms for basal species *************
  if(islogistic){
      gsl_vector_memcpy(temp5vec,Bvec);
      gsl_vector_div(temp5vec,kvec);
      gsl_vector_add_constant(temp5vec,-1);
      gsl_vector_scale(temp5vec,-1);
      gsl_vector_mul(temp5vec,temp6vec);
      gsl_vector_mul(temp5vec,basvec);			       // per unit biomass growth rate in case of logistic growth
      Bdot[nbspecies] = 0;			                               // no resource dynamics in this case

      //		PrintVector(temp5vec);
  }
  else{
      double Res = gsl_vector_get(Tvec,nbspecies);

      gsl_vector_memcpy(temp4vec,kvec);
      gsl_vector_add_constant(temp4vec,Res);

      //incorrect: gsl_vector_memcpy(temp5vec,rvec)
      gsl_vector_memcpy(temp5vec,temp6vec);
      //gsl_vector_mul(temp5vec,Res);
      gsl_vector_scale(temp5vec,Res);
      gsl_vector_div(temp5vec,temp4vec);
      gsl_vector_mul(temp5vec,basvec);			       // per unit biomass growth rate in case of resource dynamics

      gsl_vector_memcpy(temp4vec,Bvec);
      gsl_vector_mul(temp4vec,temp5vec);
      Bdot[nbspecies] = D_res*(S_res - Res) - gsl_blas_dasum(temp4vec);      // resource dynamics
  }
  gsl_vector_add(temp3vec,temp5vec);

  //	************* determine if net production rate is positive *************
  for(unsigned int i=0; i<nbspecies; i++){
      if(gsl_vector_get(temp3vec,i) > 0)
	gsl_vector_set(Theta,i,1);			       // Theta identifies consumers and plants with positive net production rate
  }

  //	************* competition for space *************
  gsl_vector_memcpy(temp4vec,sesvec);						// only sessile species contribute to competition for space
  gsl_vector_mul(temp4vec,csvec);							// strength of competition for space
  gsl_vector_mul(temp4vec,Bvec);
  gsl_vector *temp7vec=gsl_vector_calloc(nbspecies);
  gsl_blas_dgemv(CblasTrans,1,Cmat,temp4vec,0,temp7vec);

  gsl_vector_mul(Theta,temp7vec);						// competition for space only affects species with positive net growth
  gsl_vector_free(temp7vec);
  gsl_vector_mul(Theta,sesvec);							// competition for space only affects sessile species
  gsl_vector_add_constant(Theta,-1);
  gsl_vector_scale(Theta,-1);

  gsl_vector_mul(temp3vec,Theta);						// decrease per unit biomass net production rate if competition for space is a thing
  gsl_vector_mul(temp3vec,Bvec);							// total net production rate

  // ************* output total net production rate *************
  if(verbose){
      cerr<<"production:\n";
      for(unsigned int i=0; i<nbspecies; i++)
	cerr<<gsl_vector_get(temp3vec,i)<<" ";
      cerr<<endl;
  }

  //	************* collect terms of the ODEs *************
  gsl_vector_memcpy(Bdotvec,temp3vec);					// net production (can be modified by competition for space)
  //std::cerr<<gsl_vector_get(temp3vec,17-1)<<std::endl;
  gsl_vector_sub(Bdotvec,temp1vec);						// predation mortality
  gsl_vector_sub(Bdotvec,temp2vec);

  //	********** free memory of vectors that allocated new memory in this function ************
  gsl_vector_free(Tvec);
  gsl_vector_free(Theta);
  gsl_vector_free(Prey);
  gsl_vector_free(temp1vec);
  gsl_vector_free(temp2vec);
  gsl_vector_free(temp3vec);
  gsl_vector_free(temp4vec);
  gsl_vector_free(temp5vec);
  gsl_vector_free(temp6vec);
  gsl_matrix_free(temp1mat);

  return GSL_SUCCESS;
}

