#include<TrophicMatrix.hpp>
#include <gsl/gsl_sort_vector.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_linalg.h>
#include <math.h>

void TrophicMatrix::simulateFromNicheModel(double initialconnectance,
					   double topoerror)
{
  unsigned int nbspecies = nbSpecies();
  gsl_vector *niche=gsl_vector_calloc(nbspecies);
  gsl_vector *range=gsl_vector_calloc(nbspecies);
  gsl_vector *center=gsl_vector_calloc(nbspecies);
  bool webOK = false;
  while (!webOK){
      //get the random niche values (uniform, 0, 1) and sort them
      for(unsigned int i = 0; i < nbspecies; i++){
	  gsl_vector_set(niche,i,gsl_rng_uniform(RNGSingleton::get().rng())); // sets niche values uniformly distributed in [0,1)
      }
      gsl_sort_vector(niche);
      //get the random feeding ranges (beta, 0, 1), first species is basal -> zero feeding range
      gsl_vector_set(range,0,0);
      double beta = (1 - 2 * initialconnectance) / (2 * initialconnectance);
      for(unsigned int i = 1; i < nbspecies; i++)
	gsl_vector_set(range,i,gsl_ran_beta(RNGSingleton::get().rng(),1,beta));
      gsl_vector_mul(range,niche);
      //get the random feeding centers (uniform, range/2, niche[i]), first species is basal -> no feeding center
      for(unsigned int i = 1; i < nbspecies; i++) {
	  double nichevalue = gsl_vector_get(niche,i);
	  double halfrange = gsl_vector_get(range,i) / 2;
	  gsl_vector_set(center,i,(gsl_ran_flat(RNGSingleton::get().rng(), halfrange, nichevalue)));		// This function returns a random variate from the flat (uniform) distribution from a to b. The distribution is, p(x) dx = {1 \over (b-a)} dx if a <= x < b and 0 otherwise.
      }
      //in case there are still links in the adjacency matrix
      gsl_matrix_set_zero(_container);
      //fill the matrix according to niches, ranges and centers
      for(unsigned int i = 1; i< nbspecies; i++) {// i=1 because first species is basal
	  double halfrange = gsl_vector_get(range,i) / 2;
	  double upperboundary = gsl_vector_get(center,i) + halfrange;
	  double lowerboundary = gsl_vector_get(center,i) - halfrange;
	  for(unsigned int j = 0; j < nbspecies; j++) {
	      double preyniche = gsl_vector_get(niche,j);
	      if((preyniche>lowerboundary) && (preyniche<upperboundary))
		gsl_matrix_set(_container,i,j,1);
	  }
      }
      if(!isDisconnected()) {
	  if(!containsMutualFeedingInteractions()) {
	      double conn = connectance();
	      if(fabs(100. * (conn - initialconnectance)/initialconnectance) <= topoerror)
		webOK = true;
	  }
      }
      if(webOK){
	int nbbasal = -1;
	getBasalSpecies(nbbasal);
	//std::cerr<<nbbasal<<std::endl;
	if(abs(nbbasal-20)>1e-5) webOK = false;
      }
  }
  gsl_vector_free(niche);
  gsl_vector_free(range);
  gsl_vector_free(center);
}

void minTrophicLevel(gsl_matrix *mat, double* TLmin)
{
  unsigned int nbspecies = mat->size1;
  for(unsigned int i=0;i<nbspecies;i++)
    {
      gsl_vector_view tempp=gsl_matrix_row(mat,i);
      if(gsl_vector_isnull(&tempp.vector)==1)
	TLmin[i]=1;											// identify basal species
      else
	TLmin[i]=0;
    }
  unsigned int k=2;
  while(k<=nbspecies)
    {
      int counter=0;
      for(unsigned int i=0;i<=nbspecies-1;i++)
	{
	  if(TLmin[i]==0)
	    {
	      for(unsigned int j=0;j<=nbspecies-1;j++)
		{
		  if(TLmin[j]==k-1&&gsl_matrix_get(mat,i,j)!=0)
		    {
		      TLmin[i]=k; // identify species at higher trophic level than basal
		      counter=counter+1;
		    }
		}
	    }
	}
      if(counter==0)
	k=nbspecies;
      k=k+1;
    }
}

ParameterVector TrophicMatrix::getTrophicLevel()
{
  unsigned int nbspecies = nbSpecies();
  ParameterVector TLvec(_labelindex);
  for(unsigned int i=0;i<nbspecies;i++){
      gsl_vector_set(TLvec.data(),i,1);
  }
  for(unsigned int rep=0;rep<10;rep++)
    {
      for(unsigned int i=0;i<nbspecies;i++)
	{
	  gsl_vector_view tempp2=gsl_matrix_row(_container,i); // Stores the row of all the prey of species i
	  double temp1=gsl_blas_dasum(&tempp2.vector);  // temp1 contains the number of prey of species i
	  double TLtemp = 0;
	  if(temp1) // If species i has at least one prey
	    {
	      for(unsigned int k=0;k<nbspecies;k++)
		{
		  TLtemp = TLtemp + gsl_matrix_get(_container,i,k)*gsl_vector_get(TLvec.data(),k);  // calculate the mean TL of the prey of species i
		}
	      TLtemp = TLtemp / temp1;  // TL = 1 + av TL of the prey
	    }
	  gsl_vector_set(TLvec.data(),i,1 + TLtemp);
	}
    }
  return TLvec;
}

ParameterVector TrophicMatrix::getTrophicLevel_Levins()
{
  unsigned int nbspecies = nbSpecies();
  ParameterVector TLvec(_labelindex);

  gsl_matrix * B = gsl_matrix_calloc (nbspecies, nbspecies);
  gsl_vector * b = gsl_vector_calloc (nbspecies);

  //fill basal sp with one:
  for (unsigned int i=0;i<nbspecies;i++){
      gsl_vector_view row = gsl_matrix_row(_container,i);
      double kin = gsl_blas_dasum(&row.vector);
      if (!kin){
	  gsl_matrix_set(B,i,i,1.);
	  gsl_vector_set(b,i,1.);
      }
      else{
	  for(unsigned int j=0;j<nbspecies;j++){
	      gsl_matrix_set(B,i,j,-(gsl_matrix_get(_container,i,j))/kin);
	  }
	  gsl_matrix_set(B,i,i,1.-(gsl_matrix_get(_container,i,i))/kin);
	  gsl_vector_set(b,i,1.);
      }
  }
  // T.Kouya's GSL sample program collection
  //          Solving Linear System of Equations
  //                            LU decomposition
  //                   Written by Tomonori Kouya/
  // LU decomposition and forward&backward substition
  gsl_set_error_handler_off(); // error manually managed
  int signum;
  gsl_permutation* perm = gsl_permutation_alloc(B->tda);
  int issuccess = false;
  if(gsl_linalg_LU_decomp(B, perm, &signum) == GSL_SUCCESS){
      if(gsl_linalg_LU_solve(B, perm, b, TLvec.data()) == GSL_SUCCESS){
	  issuccess = true;
      }
  }
  gsl_permutation_free(perm);
  gsl_vector_free(b);
  gsl_matrix_free(B);
  if(issuccess)
    return(TLvec);
  else
    return(ParameterVector()); // empty
}

bool TrophicMatrix::isDisconnected()
{
  unsigned int nbspecies = nbSpecies();
  bool disconnected = false;

  double *TestTL=(double *) calloc(nbspecies,sizeof(double));
  gsl_matrix *TestMat=gsl_matrix_calloc(nbspecies,nbspecies);
  gsl_matrix_transpose_memcpy(TestMat,_container);
  gsl_matrix_add(TestMat,_container);

  for(unsigned int i=0;i<nbspecies;i++)
    gsl_matrix_set(TestMat,0,i,0); // species 0 in web by definition

  minTrophicLevel(TestMat,TestTL); // species connected to species 0 should have TestTL > 0
  int flag=1;
  for(unsigned int i=0;i<nbspecies;i++)
    flag = flag * TestTL[i];
  if(flag == 0)
    disconnected = true;
  for(unsigned int i=1;i<nbspecies;i++){ // search for isolated basal species:
      if(TestTL[i] == 1)// they have TestTL = 1
	disconnected = true;
  }

  free(TestTL);
  gsl_matrix_free(TestMat);

  return(disconnected);
}

ParameterVector TrophicMatrix::numberOfPredators(){
  unsigned int nbspecies = nbSpecies();
  ParameterVector predvec(_labelindex);
  for(unsigned int i=0;i<nbspecies;i++){
      gsl_vector_view predcol=gsl_matrix_column(_container,i); // stores the column of all the predators of species i
      gsl_vector_set(predvec.data(),i,gsl_blas_dasum(&predcol.vector));  // contains the number of predators of species i
  }
  return(predvec);
}

bool TrophicMatrix::containsMutualFeedingInteractions()
{
  unsigned int nbspecies = nbSpecies();
  for(unsigned int i = 0; i < nbspecies; i++) {
      for(unsigned int j = 0; j < nbspecies; j++)  {
	  if(i != j) {
	      if((gsl_matrix_get(_container,i,j)>0) && (gsl_matrix_get(_container,j,i)>0))
		return true;
	  }
      }
  }
  return false;
}

double TrophicMatrix::connectance()
{
  unsigned int nbspecies = nbSpecies();
  unsigned int links = 0;
  for(unsigned int i=0; i< nbspecies; i++) {
      for(unsigned int j=0; j< nbspecies; j++) {
	  if(gsl_matrix_get(_container,i,j)>0)
	    links++;
      }
  }
  return((double) links / (double) (nbspecies*nbspecies));
}

ParameterVector TrophicMatrix::getBasalSpecies(int &nbbasal)
{
  unsigned int nbspecies = nbSpecies();
  ParameterVector basalvec(_labelindex);
  nbbasal = 0;
  for(unsigned int i=0;i<nbspecies;i++){
      gsl_vector_view gen_in =gsl_matrix_row(_container,i);
      double generality = gsl_blas_dasum(&gen_in.vector);
      if(generality == 0){
	  nbbasal += 1;
	  gsl_vector_set(basalvec.data(),i,1);
      }
  }
  return(basalvec);
}

ParameterVector TrophicMatrix::simulateBodyMasses(double expo)
{
  unsigned int nbspecies = nbSpecies();
  ParameterVector massvec(_labelindex);
  ParameterVector TLvec = getTrophicLevel();
  for(unsigned int i=0; i< nbspecies; i++) {
      double TL = gsl_vector_get(TLvec.data(),i);
      //gsl_vector_set(massvec.data(),i, pow(expo,(TL)-1));
      gsl_vector_set(massvec.data(),i, pow(expo,round(TL)-1));
      // TODO: round takes the nearest integer, i.e. round(2.5)=3 but round(2.1)=2
  }
  return(massvec);
}

ParameterVector TrophicMatrix::simulateSessility(ParameterVector& basalvec,
						 double psessileplant, double psessileanimal)
{
  unsigned int nbspecies = nbSpecies();
  ParameterVector sessilevec(_labelindex);
  bool isok = false;
  while(!isok){
    unsigned int nbses = 0;  
    for(unsigned int i=0; i<nbspecies; i++){
      if(gsl_vector_get(basalvec.data(),i) == 1 && gsl_rng_uniform(RNGSingleton::get().rng()) < psessileplant){
	gsl_vector_set(sessilevec.data(),i,1);
	nbses++;
      }
      if(gsl_vector_get(basalvec.data(),i) == 0 && gsl_rng_uniform(RNGSingleton::get().rng()) < psessileanimal){
	gsl_vector_set(sessilevec.data(),i,1);
	nbses++;
      }
    }
    if(nbses==33) isok = true;
    else for(unsigned int i=0; i<nbspecies; i++) gsl_vector_set(sessilevec.data(),i,0);
    //std::cerr<<nbses<<std::endl;
  }
  return(sessilevec);
}
