#include<ODESolver.hpp>
#include<CommandLineParser.hpp>
#include<string>
#include<iostream>
#include<iomanip>
#include<cstdlib>
#include <unistd.h>

using namespace std;

int main(int argc, char* argv[]){
  ////////////////////////////////////
  // PARSING ARGUMENTS ///////////////
  ////////////////////////////////////
  CommandLineParser parser(argc, argv);
  bool help = false;
  if(parser.cmdOptionExists("-h") || parser.cmdOptionExists("--help"))
    help = true;
  if(help) parser.printHelpCmdOption<bool>("-h","--help","","Print this help",false);
  bool verbose = false;
  if(help) parser.printHelpCmdOption<bool>("-v","--verbose","","Verbose mode",false);
  if(parser.cmdOptionExists("-v") || parser.cmdOptionExists("--verbose"))
    verbose = true;

  // seed for the pseudo-random number generator
  srand(time(NULL));
  int seed = rand()*getpid();
  if(help) parser.printHelpCmdOption<string>("","--seed","INT","Seed for the pseudo-random number generator","internal initialization");
  seed = parser.getCmdOption<int>("","--seed",seed);
  RNGSingleton::set(seed);

  // trophic interaction
  if(help) parser.printHelpCmdOption<string>("-T","--trophic-file","FILENAME","File containing trophic interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string trophicfile = parser.getCmdOption<string>("-T","--trophic-file");
  double ptrophic = 0.15;
  if(help) parser.printHelpCmdOption<double>("","--trophic-proba","FLOAT","Probability of trophic interactions (when simulated)",ptrophic);
  ptrophic = parser.getCmdOption<double>("","--trophic-proba",ptrophic);

  unsigned int nbspecies = 10;
  if(help) parser.printHelpCmdOption<unsigned int>("-x","--nb-species","INT","Number of species, in case trophic interactions are not provided",nbspecies);
  nbspecies = parser.getCmdOption<unsigned int>("-x","--nb-species",nbspecies);
  TrophicMatrix TImat;
  ParameterVector TLvec;
  if(!help){
      if(!trophicfile.empty()){
	  TImat = TrophicMatrix(trophicfile);
	  TLvec = TImat.getTrophicLevel_Levins();
      }
      else{
	  while(TLvec.isEmpty()){
	      TImat = TrophicMatrix(nbspecies,ptrophic,10);
	      TLvec = TImat.getTrophicLevel_Levins(); // TLvec is empty if TImat is circular
	  }
      }
      if(verbose) cerr<<"TImat:\n"<<TImat;
  }

  // trophic level, deduced from TImat
  if(!help){
      if(verbose) cerr<<"TLvec:\n"<<TLvec;
  }

  // number of predators
  ParameterVector predvec;
  if(!help){
      predvec = TImat.numberOfPredators();
      if(verbose) cerr<<"predvec:\n"<<predvec;
  }

  // basal species, deduced from TImat
  int nbbasal;
  ParameterVector basalvec;
  if(!help){
      basalvec = TImat.getBasalSpecies(nbbasal);
      if(verbose) cerr<<"basalvec:\n"<<basalvec;
  }

  // sessile species
  if(help) parser.printHelpCmdOption<string>("-S","--sessile-file","FILENAME","File containing sessile information in label-value format (e.g. \"LABEL VALUE\" where VALUE=1 if sessile or 0 if mobile)");
  string sessilefile = parser.getCmdOption<string>("-S","--sessile-file");
  ParameterVector sessilevec;
  if(!help){
      if(!sessilefile.empty())
	sessilevec = ParameterVector(sessilefile, TImat.labelIndex());
      else
	sessilevec = TImat.simulateSessility(basalvec, 0.8, 0.2);
      if(verbose) cerr<<"sessilevec:\n"<<sessilevec;
  }

  // body masses
  if(help) parser.printHelpCmdOption<string>("-M","--bodymass-file","FILENAME","File containing body mass in label-value format (e.g. \"LABEL VALUE\")");
  string massfile = parser.getCmdOption<string>("-M","--bodymass-file");
  ParameterVector massvec;
  double expo = 2.6;
  if(help) parser.printHelpCmdOption<double>("","--expo","FLOAT","Exponent linking trophic level and body mass (when body masses are drawn)",expo);
  expo = parser.getCmdOption<double>("","--expo",expo);
  if(!help){
      if(!massfile.empty())
	massvec = ParameterVector(massfile, TImat.labelIndex());
      else
	massvec = TImat.simulateBodyMasses(expo);
      if(verbose) cerr<<"massvec:\n"<<massvec;
  }

  // des/activate logistic growth
  bool islogistic=false;
  if(help) parser.printHelpCmdOption<bool>("","--logistic","","Activate logistic growth",false);
  islogistic = parser.cmdOptionExists("--logistic");

  // interference competition
  if(help) parser.printHelpCmdOption<string>("-I","--interference-file","FILENAME","File containing interference interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string interferencefile = parser.getCmdOption<string>("-I","--interference-file");
  double i0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-i","","FLOAT","Strength of interference interactions",i0);
  i0 = parser.getCmdOption<double>("-i","",i0);
  double pinterference = 0.2;
  if(help) parser.printHelpCmdOption<double>("","--interference-proba","FLOAT","Probability of interference interactions (when simulated)",pinterference);
  pinterference = parser.getCmdOption<double>("","--interference-proba",pinterference);
  InterferenceMatrix Imat;
  if(!help){
      if(!interferencefile.empty())
	Imat = InterferenceMatrix(interferencefile, TImat.labelIndex(), massvec, i0);
      else
	if(i0>0.){
	  Imat = InterferenceMatrix(TImat,
				    basalvec,sessilevec,massvec,
				    pinterference, i0);
	  //Imat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Imat = InterferenceMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Imat\n"<<Imat;
  }

  // positive effects on mortality
  if(help) parser.printHelpCmdOption<string>("-P","--positive-file","FILENAME","File containing positive interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string positivefile = parser.getCmdOption<string>("-P","--positive-file");
  double p0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-p","","FLOAT","Strength of positive interactions",p0);
  p0 = parser.getCmdOption<double>("-p","",p0);
  double ppositive = 0.5;
  if(help) parser.printHelpCmdOption<double>("","--positive-proba","FLOAT","Probability of positive interactions (when simulated)",ppositive);
  ppositive = parser.getCmdOption<double>("","--positive-proba",ppositive);
  PositiveMatrix Pmat;
  if(!help){
      if(!positivefile.empty())
	Pmat = PositiveMatrix(positivefile, TImat.labelIndex(), p0);
      else
	if(p0>0.){
	  Pmat = PositiveMatrix(TImat.labelIndex(), ppositive, p0);
	  //Pmat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Pmat = PositiveMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Pmat\n"<<Pmat;
  }

  // negative effects on mortality
  if(help) parser.printHelpCmdOption<string>("-N","--negative-file","FILENAME","File containing negative interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string negativefile = parser.getCmdOption<string>("-N","--negative-file");
  double n0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-n","","FLOAT","Strength of negative interactions",n0);
  n0 = parser.getCmdOption<double>("-n","",n0);
  double pnegative = 0.1;
  if(help) parser.printHelpCmdOption<double>("","--negative-proba","FLOAT","Probability of negative interactions (when simulated)",pnegative);
  pnegative = parser.getCmdOption<double>("","--negative-proba",pnegative);
  NegativeMatrix Nmat;
  if(!help){
      if(!negativefile.empty())
	Nmat = NegativeMatrix(negativefile, TImat.labelIndex(), n0);
      else
	if(n0>0.){
	  Nmat = NegativeMatrix(TImat.labelIndex(), pnegative, n0);
	  //Nmat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Nmat = NegativeMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Nmat\n"<<Nmat;
  }

  // refuge
  if(help) parser.printHelpCmdOption<string>("-R","--refuge-file","FILENAME","File containing refuge interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string refugefile = parser.getCmdOption<string>("-R","--refuge-file");
  double r0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-r","","FLOAT","Strength of refuge interactions",r0);
  r0 = parser.getCmdOption<double>("-r","",r0);
  double prefuge = 0.1;
  if(help) parser.printHelpCmdOption<double>("","--refuge-proba","FLOAT","Probability of refuge interactions (when simulated)",prefuge);
  prefuge = parser.getCmdOption<double>("","--refuge-proba",prefuge);
  RefugeMatrix Rmat;
  if(!help){
      if(!refugefile.empty())
	Rmat = RefugeMatrix(refugefile, TImat.labelIndex(), r0);
      else
	if(r0>0.){
	  Rmat = RefugeMatrix(TImat.labelIndex(), sessilevec, predvec, prefuge, r0);
	  //Rmat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Rmat = RefugeMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Rmat\n"<<Rmat;
  }

  // establishment facilitation / recruitment
  if(help) parser.printHelpCmdOption<string>("-E","--establishment-file","FILENAME","File containing establishment facilitation / recruitment interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string establishmentfile = parser.getCmdOption<string>("-E","--establishment-file");
  double e0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-e","","FLOAT","Strength of establishment facilitation / recruitment interactions",e0);
  e0 = parser.getCmdOption<double>("-e","",e0);
  double pestablishment = 0.1;
  if(help) parser.printHelpCmdOption<double>("","--establishment-proba","FLOAT","Probability of establishment interactions (when simulated)",pestablishment);
  pestablishment = parser.getCmdOption<double>("","--establishment-proba",pestablishment);
  EstablishmentMatrix Emat;
  if(!help){
      if(!establishmentfile.empty())
	Emat = EstablishmentMatrix(establishmentfile, TImat.labelIndex(), e0);
      else
	if(e0>0.){
	  Emat = EstablishmentMatrix(TImat.labelIndex(),
				     basalvec,sessilevec,
				     pestablishment, e0);
	  //Emat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Emat = EstablishmentMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Emat\n"<<Emat;
  }

  // competition for space
  if(help) parser.printHelpCmdOption<string>("-C","--competition-file","FILENAME","File containing competition interactions in edge-list format (e.g. \"LABEL1 LABEL2\")");
  string competitionfile = parser.getCmdOption<string>("-C","--competition-file");
  double c0 = 0.;
  if(help) parser.printHelpCmdOption<double>("-c","","FLOAT","Strength of competition interactions",c0);
  c0 = parser.getCmdOption<double>("-c","",c0);
  double pcompetitionsessile = 0.9;
  if(help) parser.printHelpCmdOption<double>("","--competitionsessile-proba","FLOAT","Probability of competition interactions between sessile species (when simulated)",pcompetitionsessile);
  pcompetitionsessile = parser.getCmdOption<double>("","--competitionsessile-proba",pcompetitionsessile);
  double pcompetitionmobile = 0.;
  //if(help) parser.printHelpCmdOption<double>("","--competitionmobile-proba","FLOAT","Probability of competition interactions between mobile species (when simulated)",pcompetitionmobile);
  //pcompetitionmobile = parser.getCmdOption<double>("","--competitionmobile-proba",pcompetitionmobile);
  CompetitionMatrix Cmat;
  if(!help){
      if (!competitionfile.empty())
	Cmat = CompetitionMatrix(competitionfile, TImat.labelIndex(), c0);
      else
	if(c0>0.){
	  Cmat = CompetitionMatrix(TImat.labelIndex(),
				   basalvec,sessilevec,
				   pcompetitionsessile, pcompetitionmobile, c0, islogistic);
	  //Cmat.rewireWithPreferentialAttachment("in", 0.5);
	}
	else
	  Cmat = CompetitionMatrix(TImat.labelIndex());
      if(verbose) cerr<<"Cmat\n"<<Cmat;
  }

  // initial biomasses
  if(help) parser.printHelpCmdOption<string>("-B","--biomass-file","FILENAME","File containing intial biomass in label-value format (e.g. \"LABEL VALUE\")");
  string biomassfile = parser.getCmdOption<string>("-B","--biomass-file");
  ParameterVector biomassvec;
  if(!help)
    if(!biomassfile.empty()){
	biomassvec = ParameterVector(biomassfile, TImat.labelIndex());
    }

  // Hill coefficient
  double hill = 1.3;
  if(help) parser.printHelpCmdOption<double>("","--hill","FLOAT","Hill coefficient",hill);
  hill = parser.getCmdOption<double>("","--hill",hill);

  // Number of repetition of random biomass initialization
  unsigned int nbrep = 1;
  if(help) parser.printHelpCmdOption<unsigned int>("","--nb-random","INT","Number of repetition of random biomass initialization (to use without -B/--biomassfile option)",nbrep);
  if ( (parser.cmdOptionExists("-B") || parser.cmdOptionExists("--biomass-file")) && (parser.cmdOptionExists("--nb-random")) ){
      cerr<<"Incoherent options -B/--biomassfile and --nb-random. Exiting."<<endl;
      exit(1);
  }
  nbrep = parser.getCmdOption<unsigned int>("","--nb-random",nbrep);

  ////////////////////////////////////
  // ODE SOLVER        ///////////////
  ////////////////////////////////////
  if(!help){
      RNGSingleton::reset(seed+1); // restarting generator to guarantee identical starting points
      ODESolver solver(TImat.nbSpecies());
      for (unsigned int rep=0; rep<nbrep; rep++){
	  if(!biomassfile.empty())
	    solver.initializeBiomasses(biomassvec.data());
	  else
	    solver.initializeBiomasses();
	  solver.run(TImat,Imat,Pmat,Nmat,Rmat,Emat,Cmat,
		     basalvec,sessilevec,massvec,
		     1000, 10, 1e-6,
		     hill, islogistic, verbose, rep==0);
      }
  }
}
