## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(AeroEvapR)

## -----------------------------------------------------------------------------
## Step 1: Load libraries and data

# Load in necessary libraries
library(dplyr)
library(ggplot2)
library(lubridate)
library(data.table)
library(AeroEvapR)

# Read in data
file_path <- system.file("extdata", "PadreBay_QAQC_2019_Sep_Oct.csv", package = "AeroEvapR")
df <- read.csv(file_path)

# View data and column names
str(df)

## -----------------------------------------------------------------------------
## Step 2: Clean up and format the data for the function

# Select the variables needed for the calculation
old_var_names = c('SkinT_C_Corr_Avg', 'Wind_Speed_ms_Avg', 
                            'BP_kPa_Avg', 'AirT_C_Avg',
                            'RH_Avg', 'date')
# List the new variables that correspond to the function requirements
new_var_names <- c('T_skin', 'WS', 'P', 'T_air', 'RH', 'date')

# Replace names column names with names required for package
setnames(df, old = old_var_names, new = new_var_names)
  
# Convert date to datetime objects
df$date <- as.POSIXct(df$date, format = "%m/%d/%Y %H:%M")

str(df)

## -----------------------------------------------------------------------------
# This data is 30-minute data. Therefore timestep = 1800 (1800 seconds in 30 minutes)
# Run aero_calc
evap = suppressWarnings(aero_calc(df=df, sensor_height=2, timestep=1800, out_file_format='csv',
          out_file_name = 'testing', 
          verbose = FALSE))

## -----------------------------------------------------------------------------
# View dataframe with new values 
str(evap)

## ----my-plot, dev = "png", fig.width=7, fig.height=5--------------------------
# Plot evaporation rate calculated from aero_calc function
plot(evap$date,evap$E,type='l',col='cadetblue3',
     ylab='Evaporation (mm/30-minute)',
     xlab = 'Date',axes=T)

## -----------------------------------------------------------------------------
# Hourly
evap_hourly <- evap %>%
  mutate(hour = floor_date(date, unit = "hour")) %>%
  group_by(hour) %>%
  summarise(across(where(is.numeric), \(x) sum(x, na.rm = TRUE)))

# Daily 
evap_daily <- evap %>%
  mutate(day = floor_date(date, unit = "day")) %>%
  group_by(day) %>%
  summarise(across(where(is.numeric), \(x) sum(x, na.rm = TRUE)))


## ----my-plot2, dev = "png", fig.width=7, fig.height=5-------------------------
old_par <- par(no.readonly = TRUE)
on.exit(par(old_par), add = TRUE)
par(mfrow = c(3, 1), mar = c(4, 4, 2, 1))
# Plot 30-minute
plot(evap$date,evap$E,type='l',col='cadetblue3',
     ylab='Evap (mm/30-min)',
     xlab = '',axes=T)
# Plot hourly
plot(evap_hourly$hour,evap_hourly$E,type='l',col='lightpink',
     ylab='Evap (mm/hr)',
     xlab = '',axes=T)
# Plot daily
plot(evap_daily$day,evap_daily$E,type='l',col='darkgreen',
     ylab='Evap (mm/day)',
     xlab = 'Date',axes=T)

