## -----------------------------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  eval = TRUE
)

vcr::setup_knitr(prefix = "introsp-")
meetupr:::mock_if_no_auth()
meetupr::local_meetupr_debug(0)

## -----------------------------------------------------------------------------
library(meetupr)
library(dplyr)

## -----------------------------------------------------------------------------
# Fetch once
schema <- meetupr_schema()

# Reuse for multiple queries (no additional API calls)
queries <- meetupr_schema_queries(schema = schema)
mutations <- meetupr_schema_mutations(schema = schema)
event_fields <- meetupr_schema_type("Event", schema = schema)

## -----------------------------------------------------------------------------
query_fields <- meetupr_schema_queries(schema = schema)
query_fields

## -----------------------------------------------------------------------------
# Find all group-related queries
query_fields |>
  filter(grepl("group", field_name, ignore.case = TRUE))

# Find queries that don't require arguments
query_fields |>
  filter(args_count == 0)

## -----------------------------------------------------------------------------
# Find all event-related types
event_types <- meetupr_schema_search("event", schema = schema)
event_types

## -----------------------------------------------------------------------------
# Find user/member related types
user_types <- meetupr_schema_search("user", schema = schema)
user_types

# Find location-related types
location_types <- meetupr_schema_search("location", schema = schema)
location_types

## -----------------------------------------------------------------------------
event_fields <- meetupr_schema_type("Event", schema = schema)
event_fields

## -----------------------------------------------------------------------------
# Use regular expression to choose one specific type
event <- meetupr_schema_type("^Event$", schema = schema)
event

## -----------------------------------------------------------------------------
# Look at specific field types
event |>
  select(field_name, type) |>
  head(10)

## -----------------------------------------------------------------------------
event |>
  filter(deprecated == TRUE)

## -----------------------------------------------------------------------------
# See what fields are on Group objects
group_fields <- meetupr_schema_type("^Group$", schema = schema)
group_fields

## -----------------------------------------------------------------------------
custom_query <- "
query GetEventDetails($eventId: ID!) {
  event(id: $eventId) {
    id
    title
    description
    dateTime
    duration

    # Nested group information
    group {
      name
      urlname
      city
    }

    venues {
      name
      address
      city
      state
      postalCode
      country
      lat
      lon
      venueType
    }
  }
}
"

## -----------------------------------------------------------------------------
result <- meetupr_query(custom_query, eventId = "103349942")
result

## -----------------------------------------------------------------------------
# Get just the event title
result$data$event$title

# Get group information
result$data$event$group

# Get venue coordinates
venue <- result$data$event$venues[[1]]
c(lat = venue$lat, lng = venue$lon)

## -----------------------------------------------------------------------------
mutations <- meetupr_schema_mutations(schema = schema)
mutations

## -----------------------------------------------------------------------------
mutation_query <- "
mutation OperationName($input: InputType!) {
  mutationField(input: $input) {
    # Always check for errors
    errors {
      code
      message
    }

    # The modified object (if successful)
    resultField {
      id
      # other fields you want back
    }
  }
}
"

## -----------------------------------------------------------------------------
# # First, explore the mutation
# mutations |>
#   filter(field_name == "createEventRsvp")
# 
# # Check what input fields are needed
# rsvp_input <- meetupr_schema_type("CreateEventRsvpInput", schema = schema)
# rsvp_input
# 
# # Build the mutation
# rsvp_mutation <- "
# mutation RSVPToEvent($input: CreateEventRsvpInput!) {
#   createEventRsvp(input: $input) {
#     errors {
#       code
#       message
#     }
#     rsvp {
#       id
#       response
#       created
#     }
#   }
# }
# "
# 
# # Execute (requires authentication and valid event)
# result <- meetupr_query(
#   rsvp_mutation,
#   input = list(
#     eventId = "123456",
#     response = "YES"
#   )
# )
# 
# # Check for errors
# if (!is.null(result$data$createEventRsvp$errors)) {
#   cli::cli_alert_danger("RSVP failed")
#   print(result$data$createEventRsvp$errors)
# } else {
#   cli::cli_alert_success("RSVP successful")
#   print(result$data$createEventRsvp$rsvp)
# }

## -----------------------------------------------------------------------------
# Find all enum types
enum_types <- schema$types[sapply(schema$types, function(x) {
  x$kind == "ENUM"
})]

# Example: Event status values
event_status <- enum_types[sapply(enum_types, function(x) {
  x$name == "EventStatus"
})][[1]]

# Get allowed values
sapply(event_status$enumValues, function(x) x$name)

## -----------------------------------------------------------------------------
# Find required fields on Event
event |>
  filter(grepl("!", type)) |>
  select(field_name, type)

## -----------------------------------------------------------------------------
# Find all input types
input_types <- meetupr_schema_search("Input", schema = schema) |>
  filter(kind == "INPUT_OBJECT")

input_types

## -----------------------------------------------------------------------------
# See what fields CreateEventInput requires
create_event_input <- meetupr_schema_type("CreateEventInput", schema = schema)
create_event_input

## -----------------------------------------------------------------------------
# Find pagination-related types
pagination_types <- meetupr_schema_search("Connection", schema = schema)
pagination_types

## -----------------------------------------------------------------------------
# # Export full schema
# schema_json <- meetupr_schema(asis = TRUE)
# writeLines(schema_json, "meetupr_schema.json")

## -----------------------------------------------------------------------------
# meetupr::local_meetupr_debug(TRUE)
# 
# # Your query here
# result <- meetupr_query(custom_query, eventId = "123")
# 
# meetupr::local_meetupr_debug(FALSE)

