## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  warning = FALSE,
  message = FALSE
)

## ----libraries, message = FALSE, warning = FALSE------------------------------
library(rbmi)
library(rbmiUtils)
library(dplyr)

## ----load-data----------------------------------------------------------------
data("ADMI", package = "rbmiUtils")

## ----factor-prep--------------------------------------------------------------
ADMI <- ADMI |>
  mutate(
    TRT = factor(TRT, levels = c("Placebo", "Drug A")),
    USUBJID = factor(USUBJID),
    AVISIT = factor(AVISIT),
    STRATA = factor(STRATA),
    REGION = factor(REGION)
  )

## ----define-vars--------------------------------------------------------------
vars_binary <- set_vars(
  subjid = "USUBJID",
  visit = "AVISIT",
  group = "TRT",
  outcome = "CRIT1FLN",
  covariates = c("BASE", "STRATA", "REGION")
)

## ----define-method------------------------------------------------------------
method <- method_bayes(
  n_samples = 100,
  control = control_bayes(warmup = 200, thin = 2)
)

## ----verify-crit1fln----------------------------------------------------------
# The responder criterion is pre-derived
ADMI |>
  distinct(CRIT) |>
  pull(CRIT)

## ----threshold-analyse, message = FALSE, warning = FALSE----------------------
ana_obj <- analyse_mi_data(
  data = ADMI,
  vars = vars_binary,
  method = method,
  fun = gcomp_responder_multi,
  reference_levels = "Placebo"
)

## ----threshold-pool-----------------------------------------------------------
pool_obj <- pool(ana_obj)

## ----threshold-tidy-----------------------------------------------------------
tidy_pool_obj(pool_obj)

## ----threshold-table, eval = requireNamespace("gt", quietly = TRUE)-----------
efficacy_table(
  pool_obj,
  title = "Responder Analysis: CHG > 3",
  subtitle = "G-computation with Marginal Effects (Ge et al.)",
  arm_labels = c(ref = "Placebo", alt = "Drug A")
)

## ----cutoff-derive------------------------------------------------------------
ADMI_cutoff <- ADMI |>
  mutate(RESP5 = as.numeric(CHG > 5))

## ----cutoff-rates-------------------------------------------------------------
ADMI_cutoff |>
  group_by(TRT, AVISIT) |>
  summarise(
    n = n(),
    responders = sum(RESP5),
    rate = mean(RESP5),
    .groups = "drop"
  )

## ----cutoff-vars--------------------------------------------------------------
vars_cutoff <- set_vars(
  subjid = "USUBJID",
  visit = "AVISIT",
  group = "TRT",
  outcome = "RESP5",
  covariates = c("BASE", "STRATA", "REGION")
)

## ----cutoff-analyse, message = FALSE, warning = FALSE-------------------------
ana_obj_cutoff <- analyse_mi_data(
  data = ADMI_cutoff,
  vars = vars_cutoff,
  method = method,
  fun = gcomp_responder_multi,
  reference_levels = "Placebo"
)

## ----cutoff-pool--------------------------------------------------------------
pool_obj_cutoff <- pool(ana_obj_cutoff)

## ----cutoff-tidy--------------------------------------------------------------
tidy_pool_obj(pool_obj_cutoff)

## ----cutoff-table, eval = requireNamespace("gt", quietly = TRUE)--------------
efficacy_table(
  pool_obj_cutoff,
  title = "Responder Analysis: CHG > 5",
  subtitle = "G-computation with Marginal Effects (Ge et al.)",
  arm_labels = c(ref = "Placebo", alt = "Drug A")
)

## ----ard-conversion, message = FALSE, warning = FALSE, eval = requireNamespace("cards", quietly = TRUE)----
ard <- pool_to_ard(pool_obj)
print(ard)

