% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linear_fits.R
\name{growth.gcFitLinear}
\alias{growth.gcFitLinear}
\title{Fit an exponential growth model with a heuristic linear method}
\usage{
growth.gcFitLinear(
  time,
  data,
  gcID = "undefined",
  quota = 0.95,
  control = growth.control(t0 = 0, tmax = NA, log.x.gc = FALSE, log.y.lin = TRUE,
    min.growth = NA, max.growth = NA, lin.h = NULL, lin.R2 = 0.97, lin.RSD = 0.1, lin.dY
    = 0.05, biphasic = FALSE)
)
}
\arguments{
\item{time}{Vector of the independent variable (usually: time).}

\item{data}{Vector of dependent variable (usually: growth values).}

\item{gcID}{(Character) The name of the analyzed sample.}

\item{quota}{(Numeric, between 0 an 1) Define what fraction of \eqn{mu_{max}} the slope of regression windows adjacent to the window with highest slope should have to be included in the overall linear fit.}

\item{control}{A \code{grofit.control} object created with \code{\link{growth.control}}, defining relevant fitting options.}

\item{log.x.gc}{(Logical) Indicates whether \emph{ln(x+1)} should be applied to the time data for \emph{linear} and \emph{spline} fits. Default: \code{FALSE}.}

\item{log.y.lin}{(Logical) Indicates whether \emph{ln(y/y0)} should be applied to the growth data for \emph{linear} fits. Default: \code{TRUE}}

\item{min.growth}{(Numeric) Indicate whether only growth values above a certain threshold should be considered for linear regressions.}

\item{max.growth}{(Numeric) Indicate whether only growth values below a certain threshold should be considered for linear regressions.}

\item{t0}{(Numeric) Minimum time value considered for linear and spline fits.}

\item{tmax}{(Numeric) Minimum time value considered for linear and spline fits.}

\item{lin.h}{(Numeric) Manually define the size of the sliding window . If \code{NULL}, h is calculated for each samples based on the number of measurements in the growth phase of the plot.}

\item{lin.R2}{(Numeric) \ifelse{html}{\out{R<sup>2</sup>}}{\eqn{R^2}} threshold for \code{\link{growth.gcFitLinear}}}

\item{lin.RSD}{(Numeric) Relative standard deviation (RSD) threshold for calculated slope in \code{\link{growth.gcFitLinear}}}

\item{lin.dY}{(Numeric) Enter the minimum percentage of growth increase that a linear regression should cover.}

\item{biphasic}{(Logical) Shall \code{\link{growth.gcFitLinear}} try to extract growth parameters for two different growth phases (as observed with, e.g., diauxic shifts) (\code{TRUE}) or not (\code{FALSE})?}
}
\value{
A \code{gcFitLinear} object with parameters of the fit. The lag time is
estimated as the intersection between the fit and the horizontal line with
\eqn{y=y_0}, where \code{y0} is the first value of the dependent variable.
Use \code{\link{plot.gcFitSpline}} to visualize the linear fit.
\item{raw.time}{Raw time values provided to the function as \code{time}.}
\item{raw.data}{Raw growth data provided to the function as \code{data}.}
\item{filt.time}{Filtered time values used for the heuristic linear method.}
\item{filt.data}{Filtered growth values.}
\item{log.data}{Log-transformed, filtered growth values used for the heuristic linear method.}
\item{gcID}{(Character) Identifies the tested sample.}
\item{FUN}{Linear \emph{function} used for plotting the tangent at mumax.}
\item{fit}{\code{lm} object; result of the final call of \code{\link{lm}} to perform the linear regression.}
\item{par}{List of determined growth parameters.}
\itemize{
\item \code{y0}: {Minimum growth value considered for the heuristic linear method.}
\item \code{dY}: {Difference in maximum growth and minimum growth.}
\item \code{A}: {Maximum growth.}
\item \code{y0_lm}: {Intersection of the linear fit with the abscissa.}
\item \code{mumax}: {Maximum growth rate (i.e., slope of the linear fit).}
\item \code{tD}: {Doubling time.}
\item \code{mu.se}: {Standard error of the maximum growth rate.}
\item \code{lag}: {Lag time.}
\item \code{tmax_start}: {Time value of the first data point within the window used for the linear regression.}
\item \code{tmax_end}: {Time value of the last data point within the window used for the linear regression.}
\item \code{t_turn}: {For biphasic growth: Time of the inflection point that separates two growth phases.}
\item \code{mumax2}: {For biphasic growth: Growth rate of the second growth phase.}
\item \code{tD2}: {Doubling time of the second growth phase.}
\item \code{y0_lm2}: {For biphasic growth: Intersection of the linear fit of the second growth phase with the abscissa.}
\item \code{lag2}: {For biphasic growth: Lag time determined for the second growth phase..}
\item \code{tmax2_start}: {For biphasic growth: Time value of the first data point within the window used for the linear regression of the second growth phase.}
\item \code{tmax2_end}: {For biphasic growth: Time value of the last data point within the window used for the linear regression of the second growth phase.}
}
\item{ndx}{Index of data points used for the linear regression.}
\item{ndx2}{Index of data points used for the linear regression for the second growth phase.}
\item{control}{Object of class \code{grofit.control} containing list of options passed to the function as \code{control}.}
\item{rsquared}{\ifelse{html}{\out{R<sup>2</sup>}}{\eqn{R^2}} of the linear regression.}
\item{rsquared2}{\ifelse{html}{\out{R<sup>2</sup>}}{\eqn{R^2}} of the linear regression for the second growth phase.}
\item{fitFlag}{(Logical) Indicates whether linear regression was successfully performed on the data.}
\item{fitFlag2}{(Logical) Indicates whether a second growth phase was identified.}
\item{reliable}{(Logical) Indicates whether the performed fit is reliable (to be set manually).}
}
\description{
Determine maximum growth rates from the log-linear part of a growth curve using
a heuristic approach similar to the ``growth rates made easy''-method of
Hall et al. (2013).
}
\details{
The algorithm works as follows:
\enumerate{
\item Fit linear regressions (Theil-Sen estimator) to all subsets of \code{h} consecutive, log-transformed data
points (sliding window of size \code{h}). If for example \eqn{h=5}, fit a linear regression to points
1 \dots 5, 2 \dots 6, 3 \dots 7 and so on.
\item Find the subset with the highest slope \eqn{mu_{max}}. Do the \ifelse{html}{\out{R<sup>2</sup>}}{\eqn{R^2}} and relative standard deviation (RSD) values of the regression meet the in \code{lin.R2} and \code{lin.RSD} defined thresholds and do the data points within the regression window account for a fraction of at least \code{lin.dY} of the total growth increase? If not, evaluate the subset with the second highest slope, and so on.
\item Include also the data points of adjacent subsets that have a slope of at least \eqn{quota \cdot mu{max}}, e.g., all regression windows that have at least 95\% of the maximum slope.
\item Fit a new linear model to the extended data window identified in step 3.
}
If \code{biphasic = TRUE}, the following steps are performed to define a second growth phase:
\enumerate{
\item Perform a smooth spline fit on the data with a smoothing factor of 0.5.
\item Calculate the second derivative of the spline fit and perform a smooth spline fit of the derivative with a smoothing factor of 0.4.
\item Determine local maxima and minima in the second derivative.
\item Find the local minimum following \eqn{mu_{max}} and repeat the heuristic linear method for later time values.
\item Find the local maximum before \eqn{mu_{max}} and repeat the heuristic linear method for earlier time values.
\item Choose the greater of the two independently determined slopes as \eqn{mu_{max}2}.
}
}
\examples{
# Create random growth dataset
rnd.dataset <- rdm.data(d = 35, mu = 0.8, A = 5, label = "Test1")

# Extract time and growth data for single sample
time <- rnd.dataset$time[1,]
data <- rnd.dataset$data[1,-(1:3)] # Remove identifier columns

# Perform linear fit
TestFit <- growth.gcFitLinear(time, data, gcID = "TestFit",
                 control = growth.control(fit.opt = "l"))

plot(TestFit)

}
\references{
Hall, BG., Acar, H, Nandipati, A and Barlow, M (2014) Growth Rates Made Easy. \emph{Mol. Biol. Evol.} 31: 232-38, DOI: 10.1093/molbev/mst187

Petzoldt T (2022). growthrates: Estimate Growth Rates from Experimental Data. R package version 0.8.3, \url{https://CRAN.R-project.org/package=growthrates}.

Theil, H.(1992). A rank-invariant method of linear and polynomial regression analysis. In: Henri Theil’s contributions to economics and econometrics. Springer, pp. 345–381. DOI: 10.1007/978-94-011-2546-8_20
}
\seealso{
Other growth fitting functions: 
\code{\link{growth.drFit}()},
\code{\link{growth.gcBootSpline}()},
\code{\link{growth.gcFitModel}()},
\code{\link{growth.gcFitSpline}()},
\code{\link{growth.gcFit}()},
\code{\link{growth.workflow}()}
}
\concept{growth fitting functions}
