// Copyright (C) 1997-1999  Adrian Trapletti
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

//
// Linear algebra package 
//


#include <iostream.h>
extern "C" {
#include "nrutil.h"
#include "nrlinal.h"
#include "nrext.h"
}
#include "utils.hh"
#include "str.hh"
#include "linal.hh"


// utilities for working with class vec and mat

void vec2Rvec (const vec& v_src, double* v_dest)  // copy vec to already allocated R vector
{
  long i;
  
  for (i=1; i<=v_src.rows(); i++)
    v_dest[i-1] = v_src(i);
}

void mat2Rmat (const mat& m_src, double* m_dest)  // copy mat to already allocated R matrix
{
  long i, j;
  long n = m_src.rows();
  long m = m_src.columns();
  
  for (i=1; i<=n; i++)
    for (j=1; j<=m; j++)
      m_dest[(i-1)+n*(j-1)] = m_src(i,j);
}


// column vector of double

istream& operator>> (istream& s, vec& v)  // deallocate v and read v from istream
{
  str buf1, buf2;
  long i, n;
  
  if (!(s >> buf1 >> buf2 >> n) || (buf1 != "#") || (buf2 != "n:"))
    RTerror ("linal.cc", "istream& operator>> (istream& s, vec& v)", "1");
  if (n != v.n)  // v has not the right dimension
  {
    free_dvector (v.p, 1, v.n);
    v.n = n;
    v.p = dvector (1, v.n);
  }
  for (i=1; i<=v.n; i++)
    if (!(s >> v.p[i]))
      RTerror ("linal.cc", "istream& operator>> (istream& s, vec& v)", "2");
  return s;
}

ostream& operator<< (ostream& s, const vec& v) 
{
  long i;
  
  s << "# n: " << v.n << endl;
  for (i=1; i<=v.n; i++)
    s << v.p[i] << endl;
  return s;
}

vec::vec (const vec& v)  // copy constructor
{
  n = v.n;
  p = dvector (1, n);
  copy_dvector (v.p, p, 1, n);
}

vec::vec (long nn)
{
#ifdef CHECK
  if (nn < 0) RTerror ("linal.cc", "vec::vec (long nn)");
#endif  
  n = nn;
  p = dvector (1, n);
}

vec::vec (double* x, long nn)  // copy constructor with an R vector as input
{
  long i;

#ifdef CHECK
  if (nn < 0) RTerror ("linal.cc", "vec::vec (double* x, long nn)");
#endif  
  n = nn;
  p = dvector (1, n);
  for (i=1; i<=n; i++)
    p[i] = x[i-1];
}

vec::~vec ()
{ 
  free_dvector (p, 1, n);
}

#ifdef CHECK
double& vec::operator() (long i)  // read and write elements
{
  if ((i < 1) || (i > n))
    RTerror ("linal.cc", "double& vec::operator() (long i)");
  return p[i];
}
#endif

#ifdef CHECK
double vec::operator() (long i) const  // read elements
{
  if ((i < 1) || (i > n))
    RTerror ("linal.cc", "double vec::operator() (long i) const");
  return p[i];
}
#endif

vec& vec::operator= (const vec& v)  // assignment 
{
  if (this != &v)  // beware of v = v, i.e., this is not equal to v
  {
    if (n != v.n)  // this has wrong dimension 
    {
      free_dvector (p, 1, n);
      n = v.n; 
      p = dvector (1, n);
    }
    copy_dvector (v.p, p, 1, n);
  }
  return *this;
}

vec& vec::operator= (const vPLv& v)  // v3 = v1+v2;
{
#ifdef CHECK
  if (v.v1.n != v.v2.n) RTerror ("linal.cc", "vec& vec::operator= (const vPLv& v)");
#endif
  if (n != v.v1.n)  // v3 has wrong dimension and therefore is not equal to v1 or v2
  { 
    free_dvector (p, 1, n);
    n = v.v1.n; 
    p = dvector (1, n);
  }
  vEQvPLv (p, v.v1.p, v.v2.p, n);
  return *this;
}

vec& vec::operator= (const vMIv& v)  // v3 = v1-v2;
{
#ifdef CHECK
  if (v.v1.n != v.v2.n) RTerror ("linal.cc", "vec& vec::operator= (const vMIv& v)");
#endif
  if (n != v.v1.n)  // v3 has wrong dimension and therefore is not equal to v1 or v2
  { 
    free_dvector (p, 1, n);
    n = v.v1.n; 
    p = dvector (1, n);
  }
  vEQvMIv (p, v.v1.p, v.v2.p, n);
  return *this;
}

vec& vec::operator= (const vTIv& v)  // v3 = v1*v2;
{
#ifdef CHECK
  if (v.v1.n != v.v2.n) RTerror ("linal.cc", "vec& vec::operator= (const vTIv& v)");
#endif
  if (n != v.v1.n)  // v3 has wrong dimension and therefore is not equal to v1 or v2
  { 
    free_dvector (p, 1, n);
    n = v.v1.n; 
    p = dvector (1, n);
  }
  vEQvTIv (p, v.v1.p, v.v2.p, n);
  return *this;
}

vec& vec::operator= (const vDIv& v)  // v3 = v1/v2;
{
#ifdef CHECK
  if (v.v1.n != v.v2.n) RTerror ("linal.cc", "vec& vec::operator= (const vDIv& v)");
#endif
  if (n != v.v1.n)  // v3 has wrong dimension and therefore is not equal to v1 or v2
  { 
    free_dvector (p, 1, n);
    n = v.v1.n; 
    p = dvector (1, n);
  }
  vEQvDIv (p, v.v1.p, v.v2.p, n);
  return *this;
}

vec& vec::operator+= (const vec& v)  // v2 += v1;
{
#ifdef CHECK
  if (n != v.n) RTerror ("linal.cc", "vec& vec::operator+= (const vec& v)");
#endif
  vPLEQv (p, v.p, n);
  return *this;
}

vec& vec::operator-= (const vec& v)  // v2 -= v1;
{
#ifdef CHECK
  if (n != v.n) RTerror ("linal.cc", "vec& vec::operator-= (const vec& v)");
#endif
  vMIEQv (p, v.p, n);
  return *this;
}

vec& vec::operator*= (const vec& v)  // v2 *= v1;
{
#ifdef CHECK
  if (n != v.n) RTerror ("linal.cc", "vec& vec::operator*= (const vec& v)");
#endif
  vTIEQv (p, v.p, n);
  return *this;
}

vec& vec::operator/= (const vec& v)  // v2 /= v1;
{
#ifdef CHECK
  if (n != v.n) RTerror ("linal.cc", "vec& vec::operator/= (const vec& v)");
#endif
  vDIEQv (p, v.p, n);
  return *this;
}

vec& vec::operator+= (double d)  // v += d;
{
  vPLEQd (p, d, n);
  return *this;
}

vec& vec::operator-= (double d)  // v -= d;
{
  vMIEQd (p, d, n);
  return *this;
}

vec& vec::operator*= (double d)  // v *= d;
{
  vTIEQd (p, d, n);
  return *this;
}

vec& vec::operator/= (double d)  // v /= d;
{
  vDIEQd (p, d, n);
  return *this;
}

vec& vec::operator= (const vPLd& v)  // v2 = v1+d;
{
  if (n != v.v.n)  // v2 has wrong dimension and therefore is not equal to v1
  { 
    free_dvector (p, 1, n);
    n = v.v.n; 
    p = dvector (1, n);
  }
  vEQvPLd (p, v.v.p, v.d, n);
  return *this;
}

vec& vec::operator= (const vMId& v)  // v2 = v1-d;
{
  if (n != v.v.n)  // v2 has wrong dimension and therefore is not equal to v1
  { 
    free_dvector (p, 1, n);
    n = v.v.n; 
    p = dvector (1, n);
  }
  vEQvMId (p, v.v.p, v.d, n);
  return *this;
}

vec& vec::operator= (const vTId& v)  // v2 = v1*d;
{
  if (n != v.v.n)  // v2 has wrong dimension and therefore is not equal to v1
  { 
    free_dvector (p, 1, n);
    n = v.v.n; 
    p = dvector (1, n);
  }
  vEQvTId (p, v.v.p, v.d, n);
  return *this;
}

vec& vec::operator= (const vDId& v)  // v2 = v1/d;
{
  if (n != v.v.n)  // v2 has wrong dimension and therefore is not equal to v1
  { 
    free_dvector (p, 1, n);
    n = v.v.n; 
    p = dvector (1, n);
  }
  vEQvDId (p, v.v.p, v.d, n);
  return *this;
}

vec& vec::operator= (const SQRv& v)  // v2 = sqr(v1);  elementwise square of vector
{
  if (n != v.v.n)  // v2 has wrong dimension and therefore is not equal to v1
  { 
    free_dvector (p, 1, n);
    n = v.v.n; 
    p = dvector (1, n);
  }
  vEQvTIv (p, v.v.p, v.v.p, n);
  return *this;
}

double inn (const vec& v1, const vec& v2)  // inner product 
{
  double sum;

#ifdef CHECK
  if (v1.n != v2.n) RTerror ("linal.cc", "double inn (const vec& v1, const vec& v2)");
#endif
  sum = INNER (v1.p, v2.p, v1.n);
  return sum;
}

double sum (const vec& v)  // sum over all elements
{
  double s;

  s = SUMv (v.p, v.n);
  return s;
}

double mean (const vec& v)  // arithmetic mean over all elements
{
  double s;

  s = MEANv (v.p, v.n);
  return s;
}

vec& vec::operator= (const mTIv& ma)  // v2 = m*v1;
{
  double *p_temp;

#ifdef CHECK
  if (ma.ma.m != ma.v.n) RTerror ("linal.cc", "vec& vec::operator= (const mTIv& ma)");
#endif
  if ((this != &(ma.v)) && (n == ma.ma.n))
    // this is different from v1 and this has already the right dimension
  {
    vEQmTIv (p, ma.ma.p, ma.v.p, ma.ma.n, ma.ma.m);
  }
  else
  {
    p_temp = dvector (1, ma.ma.n);
    vEQmTIv (p_temp, ma.ma.p, ma.v.p, ma.ma.n, ma.ma.m);  // write result to temporary
    free_dvector (p, 1, n);
    n = ma.ma.n;
    p = p_temp;
  }
  return *this;
}

vec& vec::rand (int type, double mean, double var)  // initialize vector elements randomly
{
  RANDv (p, n, type, mean, var);
  return *this;
}

vec& vec::zero ()  // set vector elements to zero
{
  ZEROv (p, n);
  return *this;
}

vec& vec::one ()  // set vector elements to one
{
  ONEv (p, n);
  return *this;
}

void vec::resize (long nn)
  /* (*this) is resized to a new size of nn elements
     the element values are preserved as far as this is possible, i.e.,
     el(1) = el.old(1), ..., el(min(n,n.old)) = el.old(min(n,n.old)) 
     the other element values are undefined */
{
  long i;
  double* pnew;

  if (n != nn)  // (*this) has not the right dimension
  {
    pnew = dvector (1, nn);
    for (i=1; i<=LMIN(n,nn); i++)
      pnew[i] = p[i];
    free_dvector (p, 1, n);
    n = nn;
    p = pnew;
  }
}


// matrix of double

istream& operator>> (istream& s, mat& ma)  // deallocate ma and read ma from istream
{
  str buf1, buf2;
  long i, j, n, m;
  
  if (!(s >> buf1 >> buf2 >> n) || (buf1 != "#") || (buf2 != "n:"))
    RTerror ("linal.cc", "istream& operator>> (istream& s, mat& ma)", "1");
  if (!(s >> buf1 >> buf2 >> m) || (buf1 != "#") || (buf2 != "m:"))
    RTerror ("linal.cc", "istream& operator>> (istream& s, mat& ma)", "2");
  if ((n != ma.n) || (m != ma.m))  // ma has not the right dimension
  {
    free_dmatrix (ma.p, 1, ma.n, 1, ma.m);
    ma.n = n; ma.m = m;
    ma.p = dmatrix (1, ma.n, 1, ma.m);
  }
  for (i=1; i<=ma.n; i++)
    for (j=1; j<=ma.m; j++)
      if (!(s >> ma.p[i][j]))
	RTerror ("linal.cc", "istream& operator>> (istream& s, mat& ma)", "3");
  return s;
}

ostream& operator<< (ostream& s, const mat& ma) 
{
  long i, j;

  s << "# n: " << ma.n << endl;
  s << "# m: " << ma.m << endl;
  for (i=1; i<=ma.n; i++)
  {
    for (j=1; j<=ma.m; j++)
      s << ma.p[i][j] << " ";
    s << endl;
  }
  return s;
}

mat::mat (const mat& ma)  // copy constructor
{
  n = ma.n; m = ma.m;
  p = dmatrix (1, n, 1, m);
  copy_dmatrix (ma.p, p, 1, n, 1, m);
}

mat::mat (long nn, long mm)
{
#ifdef CHECK
  if ((nn < 0) || (mm < 0)) RTerror ("linal.cc", "mat::mat (long nn, long mm)");
#endif    
  n = nn; m = mm;
  p = dmatrix (1, n, 1, m);
}

mat::mat (double* x, long nn, long mm)  // copy constructor with an R matrix as input
{
  long i, j;

#ifdef CHECK
  if ((nn < 0) || (mm < 0)) RTerror ("linal.cc", "mat::mat (double* x, long nn, long mm)");
#endif    
  n = nn; m = mm;
  p = dmatrix (1, n, 1, m);
  for (i=1; i<=n; i++)
    for (j=1; j<=m; j++)
      p[i][j] = x[(i-1)+n*(j-1)];
}

mat::~mat ()
{ 
  free_dmatrix (p, 1, n, 1, m);
}

#ifdef CHECK
double& mat::operator() (long i, long j)  // read and write elements
{
  if ((i < 1) || (i > n) || (j < 1) || (j > m))
    RTerror ("linal.cc", "double& mat::operator() (long i, long j)");
  return p[i][j];
}
#endif

#ifdef CHECK
double mat::operator() (long i, long j) const  // read elements
{
  if ((i < 1) || (i > n) || (j < 1) || (j > m))
    RTerror ("linal.cc", "double mat::operator() (long i, long j) const");
  return p[i][j];
}
#endif

mat& mat::operator= (const mat& ma)  // assignment 
{
  if (this != &ma)  // beware of ma = ma, i.e., this is not equal to ma
  {
    if ((n != ma.n) || (m != ma.m))  // this has wrong dimension
    {
      free_dmatrix (p, 1, n, 1, m);
      n = ma.n; m = ma.m;
      p = dmatrix (1, n, 1, m);
    }
    copy_dmatrix (ma.p, p, 1, n, 1, m);
  }
  return *this;
}

mat& mat::operator= (const mPLm& ma)  // m3 = m1+m2;
{
#ifdef CHECK
  if ((ma.ma1.n != ma.ma2.n) || (ma.ma1.m != ma.ma2.m)) 
    RTerror ("linal.cc", "mat& mat::operator= (const mPLm& ma)");
#endif
  if ((n != ma.ma1.n) || (m != ma.ma1.m))  
    // m3 has wrong dimension and therefore is not equal to m1 or m2
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma1.n; m = ma.ma1.m; 
    p = dmatrix (1, n, 1, m);
  }
  mEQmPLm (p, ma.ma1.p, ma.ma2.p, n, m);
  return *this;
}

mat& mat::operator= (const mMIm& ma)  // m3 = m1-m2;
{
#ifdef CHECK
  if ((ma.ma1.n != ma.ma2.n) || (ma.ma1.m != ma.ma2.m)) 
    RTerror ("linal.cc", "mat& mat::operator= (const mMIm& ma)");
#endif
  if ((n != ma.ma1.n) || (m != ma.ma1.m))  
    // m3 has wrong dimension and therefore is not equal to m1 or m2
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma1.n; m = ma.ma1.m; 
    p = dmatrix (1, n, 1, m);
  }
  mEQmMIm (p, ma.ma1.p, ma.ma2.p, n, m);
  return *this;
}

mat& mat::operator= (const mTIm& ma)  // m3 = m1*m2;
{
#ifdef CHECK
  if ((ma.ma1.n != ma.ma2.n) || (ma.ma1.m != ma.ma2.m)) 
    RTerror ("linal.cc", "mat& mat::operator= (const mTIm& ma)");
#endif
  if ((n != ma.ma1.n) || (m != ma.ma1.m))  
    // m3 has wrong dimension and therefore is not equal to m1 or m2
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma1.n; m = ma.ma1.m; 
    p = dmatrix (1, n, 1, m);
  }
  mEQmTIm (p, ma.ma1.p, ma.ma2.p, n, m);
  return *this;
}

mat& mat::operator= (const mDIm& ma)  // m3 = m1/m2;
{
#ifdef CHECK
  if ((ma.ma1.n != ma.ma2.n) || (ma.ma1.m != ma.ma2.m)) 
    RTerror ("linal.cc", "mat& mat::operator= (const mDIm& ma)");
#endif
  if ((n != ma.ma1.n) || (m != ma.ma1.m))  
    // m3 has wrong dimension and therefore is not equal to m1 or m2
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma1.n; m = ma.ma1.m; 
    p = dmatrix (1, n, 1, m);
  }
  mEQmDIm (p, ma.ma1.p, ma.ma2.p, n, m);
  return *this;
}

mat& mat::operator+= (const mat& ma)  // m2 += m1;
{
#ifdef CHECK
  if ((n != ma.n) || (m != ma.m)) RTerror ("linal.cc", "mat& mat::operator+= (const mat& ma)");
#endif
  mPLEQm (p, ma.p, n, m);
  return *this;
}

mat& mat::operator-= (const mat& ma)  // m2 -= m1;
{
#ifdef CHECK
  if ((n != ma.n) || (m != ma.m)) RTerror ("linal.cc", "mat& mat::operator-= (const mat& ma)");
#endif
  mMIEQm (p, ma.p, n, m);
  return *this;
}

mat& mat::operator*= (const mat& ma)  // m2 *= m1;
{
#ifdef CHECK
  if ((n != ma.n) || (m != ma.m)) RTerror ("linal.cc", "mat& mat::operator*= (const mat& ma)");
#endif
  mTIEQm (p, ma.p, n, m);
  return *this;
}

mat& mat::operator/= (const mat& ma)  // m2 /= m1;
{
#ifdef CHECK
  if ((n != ma.n) || (m != ma.m)) RTerror ("linal.cc", "mat& mat::operator/= (const mat& ma)");
#endif
  mDIEQm (p, ma.p, n, m);
  return *this;
}

mat& mat::operator+= (double d)  // m += d;
{
  mPLEQd (p, d, n, m);
  return *this;
}

mat& mat::operator-= (double d)  // m -= d;
{
  mMIEQd (p, d, n, m);
  return *this;
}

mat& mat::operator*= (double d)  // m *= d;
{
  mTIEQd (p, d, n, m);
  return *this;
}

mat& mat::operator/= (double d)  // m /= d;
{
  mDIEQd (p, d, n, m);
  return *this;
}

mat& mat::operator= (const mPLd& ma)  // m2 = m1+d;
{
  if ((n != ma.ma.n) || (m != ma.ma.m))  // m2 has wrong dimension and therefore is not equal to m1
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma.n; m = ma.ma.m;
    p = dmatrix (1, n, 1, m);
  }
  mEQmPLd (p, ma.ma.p, ma.d, n, m);
  return *this;
}

mat& mat::operator= (const mMId& ma)  // m2 = m1-d;
{
  if ((n != ma.ma.n) || (m != ma.ma.m))  // m2 has wrong dimension and therefore is not equal to m1
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma.n; m = ma.ma.m;
    p = dmatrix (1, n, 1, m);
  }
  mEQmMId (p, ma.ma.p, ma.d, n, m);
  return *this;
}

mat& mat::operator= (const mTId& ma)  // m2 = m1*d;
{
  if ((n != ma.ma.n) || (m != ma.ma.m))  // m2 has wrong dimension and therefore is not equal to m1
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma.n; m = ma.ma.m;
    p = dmatrix (1, n, 1, m);
  }
  mEQmTId (p, ma.ma.p, ma.d, n, m);
  return *this;
}

mat& mat::operator= (const mDId& ma)  // m2 = m1/d;
{
  if ((n != ma.ma.n) || (m != ma.ma.m))  // m2 has wrong dimension and therefore is not equal to m1
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma.n; m = ma.ma.m;
    p = dmatrix (1, n, 1, m);
  }
  mEQmDId (p, ma.ma.p, ma.d, n, m);
  return *this;
}

mat& mat::operator= (const mMTIm& ma)  // m3 = mul(m1,m2);  matrix multiplication
{
  double **p_temp;

#ifdef CHECK
  if (ma.ma1.m != ma.ma2.n) RTerror ("linal.cc", "mat& mat::operator= (const mMTIm& ma)");
#endif
  if ((this != &(ma.ma1)) && (this != &(ma.ma2)) && (n == ma.ma1.n) && (m == ma.ma2.m))
    // this is different from m1 and m2 and this has already the right dimensions
  {
    mEQmMTIm (p, ma.ma1.p, ma.ma2.p, ma.ma1.n, ma.ma1.m, ma.ma2.m); 
  }
  else
  {
    p_temp = dmatrix (1, ma.ma1.n, 1, ma.ma2.m);  
    mEQmMTIm (p_temp, ma.ma1.p, ma.ma2.p, ma.ma1.n, ma.ma1.m, ma.ma2.m);  // write result to temporary
    free_dmatrix (p, 1, n, 1, m);
    n = ma.ma1.n; m = ma.ma2.m; 
    p = p_temp;
  }
  return *this;
}

mat& mat::operator= (const vOUTv& v)  // m = out(v1,v2);  outer product
{
#ifdef CHECK
  if (v.v1.n != v.v2.n) RTerror ("linal.cc", "mat& mat::operator= (const vOUTv& v)");
#endif
  if ((n != v.v1.n) || (m != v.v1.n))  // m has wrong dimension
  { 
    free_dmatrix (p, 1, n, 1, m);
    n = m = v.v1.n; 
    p = dmatrix (1, n, 1, m);
  }
  OUTER (p, v.v1.p, v.v2.p, n);
  return *this;
}

mat& mat::rand (int type, double mean, double var)  // initialize matrix elements randomly
{
  RANDm (p, n, m, type, mean, var);
  return *this;
}

mat& mat::zero ()  // set matrix elements to zero
{
  ZEROm (p, n, m);
  return *this;
}

mat& mat::one ()  // set matrix elements to one
{
  ONEm (p, n, m);
  return *this;
}

mat& mat::eye ()  // set matrix to identity matrix, only for a square matrix
{
#ifdef CHECK
  if (n != m) RTerror ("linal.cc", "mat& mat::eye ()");
#endif  
  EYEm (p, n);
  return *this;
}

void mat::resize (long nn, long mm)
  /* (*this) is resized to a new size of nn x mm elements
     the element values are preserved as far as this is possible, i.e.,
     el(1,1) = el.old(1,1), ..., el(min(n,n.old),min(m,m.old)) = el.old(min(n,n.old),min(m,m.old)) 
     the other element values are undefined */
{
  long i, j;
  double** pnew;

  if ((n != nn) || (m != mm))  // (*this) has not the right dimension
  {
    pnew = dmatrix (1, nn, 1, mm);
    for (i=1; i<=LMIN(n,nn); i++)
      for (j=1; j<=LMIN(m,mm); j++)
	pnew[i][j] = p[i][j];
    free_dmatrix (p, 1, n, 1, m);
    n = nn; m = mm;
    p = pnew;
  }
}


// vector with elements of type integer

istream& operator>> (istream& s, ivec& v)  // deallocate v and read v from istream
{
  str buf1, buf2;
  long i, n;
  
  if (!(s >> buf1 >> buf2 >> n) || (buf1 != "#") || (buf2 != "n:"))
    RTerror ("linal.cc", "istream& operator>> (istream& s, ivec& v)", "1");
  if (n != v.n)  // v has not the right dimension
  {
    free_ivector (v.p, 1, v.n);
    v.n = n;
    v.p = ivector (1, v.n);
  }
  for (i=1; i<=v.n; i++)
    if (!(s >> v.p[i]))
      RTerror ("linal.cc", "istream& operator>> (istream& s, ivec& v)", "2");
  return s;
}

ostream& operator<< (ostream& s, const ivec& v) 
{
  long i;
  
  s << "# n: " << v.n << endl;
  for (i=1; i<=v.n; i++)
    s << v.p[i] << endl;
  return s;
}

ivec::ivec (const ivec& v)  // copy constructor
{
  n = v.n;
  p = ivector (1, n);
  copy_ivector (v.p, p, 1, n);
}

ivec::ivec (long nn)
{
#ifdef CHECK
  if (nn < 0) RTerror ("linal.cc", "ivec::ivec (long nn)");
#endif  
  n = nn;
  p = ivector (1, n);
}

ivec::~ivec ()
{ 
  free_ivector (p, 1, n);
}

#ifdef CHECK
int& ivec::operator() (long i)  // read and write elements
{
  if ((i < 1) || (i > n))
    RTerror ("linal.cc", "int& ivec::operator() (long i)");
  return p[i];
}
#endif

#ifdef CHECK
int ivec::operator() (long i) const  // read elements
{
  if ((i < 1) || (i > n))
    RTerror ("linal.cc", "int ivec::operator() (long i) const");
  return p[i];
}
#endif

ivec& ivec::operator= (const ivec& v)  // assignment 
{
  if (this != &v)  // beware of v = v, i.e., this is not equal to v
  {
    if (n != v.n)  // this has wrong dimension 
    {
      free_ivector (p, 1, n);
      n = v.n; 
      p = ivector (1, n);
    }
    copy_ivector (v.p, p, 1, n);
  }
  return *this;
}

ivec rperm (long n, long k) 
/* Returns the ivec I(k) with the k elements I(1),..,I(k) sampled
   randomly from {1,..,n} without replacement, n >= k */
{
  long i;
  int *p_temp;
  ivec I(k);
  
#ifdef CHECK
  if (k > n) RTerror ("linal.cc", "ivec rperm (long n, long k)");
#endif  
  p_temp = ivector (1, n);
  RPERM (n, k, p_temp);
  for (i=1; i<=k; i++)
    I.p[i] = p_temp[i];
  return I;  // inefficient, for an assignment N = rperm(), I is copied twice!
}


