/* CAUTION: This is the ANSI C (only) version of the Numerical Recipes
   utility file nrutil.c.  Do not confuse this file with the same-named
   file nrutil.c that is supplied in the 'misc' subdirectory.
   *That* file is the one from the book, and contains both ANSI and
   traditional K&R versions, along with #ifdef macros to select the
   correct version.  *This* file contains only ANSI C.               */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include "nrutil.h"
#ifdef RFLAG
#include "R.h"
#endif
#define NR_END 1
#define FREE_ARG char*

void nrerror(char error_text[])
     /* Numerical Recipes standard error handler */
{
  fprintf(stderr,"Numerical Recipes run-time error...\n");
  fprintf(stderr,"%s\n",error_text);
  fprintf(stderr,"...now exiting...\n");
#ifdef RFLAG
  error("Returning to R\n");
#else
  exit(1);
#endif
}

float *vector(long nl, long nh)
     /* allocate a float vector with subscript range v[nl..nh] */
{
  float *v;

#ifdef RFLAG
  v=(float *)R_alloc(nh-nl+1+NR_END, sizeof(float));
#else
  v=(float *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(float)));
#endif
  if (!v) nrerror("allocation failure in vector()");
  return v-nl+NR_END;
}

int *ivector(long nl, long nh)
     /* allocate an int vector with subscript range v[nl..nh] */
{
  int *v;

#ifdef RFLAG
  v=(int *)R_alloc(nh-nl+1+NR_END, sizeof(int));
#else
  v=(int *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(int)));
#endif
  if (!v) nrerror("allocation failure in ivector()");
  return v-nl+NR_END;
}

unsigned char *cvector(long nl, long nh)
     /* allocate an unsigned char vector with subscript range v[nl..nh] */
{
  unsigned char *v;

#ifdef RFLAG
  v=(unsigned char *)R_alloc(nh-nl+1+NR_END, sizeof(unsigned char));
#else
  v=(unsigned char *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(unsigned char)));
#endif
  if (!v) nrerror("allocation failure in cvector()");
  return v-nl+NR_END;
}

unsigned long *lvector(long nl, long nh)
     /* allocate an unsigned long vector with subscript range v[nl..nh] */
{
  unsigned long *v;

#ifdef RFLAG
  v=(unsigned long *)R_alloc(nh-nl+1+NR_END, sizeof(unsigned long));
#else
  v=(unsigned long *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(unsigned long)));
#endif
  if (!v) nrerror("allocation failure in lvector()");
  return v-nl+NR_END;
}

double *dvector(long nl, long nh)
     /* allocate a double vector with subscript range v[nl..nh] */
{
  double *v;

#ifdef RFLAG
  v=(double *)R_alloc(nh-nl+1+NR_END, sizeof(double));
#else
  v=(double *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(double)));
#endif
  if (!v) nrerror("allocation failure in dvector()");
  return v-nl+NR_END;
}

float **matrix(long nrl, long nrh, long ncl, long nch)
     /* allocate a float matrix with subscript range m[nrl..nrh][ncl..nch] */
{
  long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
  float **m;

  /* allocate pointers to rows */
#ifdef RFLAG
  m=(float **) R_alloc(nrow+NR_END, sizeof(float*));
#else
  m=(float **) malloc((size_t)((nrow+NR_END)*sizeof(float*)));
#endif
  if (!m) nrerror("allocation failure 1 in matrix()");
  m += NR_END;
  m -= nrl;

  /* allocate rows and set pointers to them */
#ifdef RFLAG
  m[nrl]=(float *) R_alloc(nrow*ncol+NR_END, sizeof(float));
#else
  m[nrl]=(float *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(float)));
#endif
  if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

  /* return pointer to array of pointers to rows */
  return m;
}

double **dmatrix(long nrl, long nrh, long ncl, long nch)
     /* allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] */
{
  long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
  double **m;

  /* allocate pointers to rows */
#ifdef RFLAG
  m=(double **) R_alloc(nrow+NR_END, sizeof(double*));
#else
  m=(double **) malloc((size_t)((nrow+NR_END)*sizeof(double*)));
#endif
  if (!m) nrerror("allocation failure 1 in matrix()");
  m += NR_END;
  m -= nrl;

  /* allocate rows and set pointers to them */
#ifdef RFLAG
  m[nrl]=(double *) R_alloc(nrow*ncol+NR_END, sizeof(double));
#else
  m[nrl]=(double *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(double)));
#endif
  if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

  /* return pointer to array of pointers to rows */
  return m;
}

int **imatrix(long nrl, long nrh, long ncl, long nch)
     /* allocate a int matrix with subscript range m[nrl..nrh][ncl..nch] */
{
  long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
  int **m;

  /* allocate pointers to rows */
#ifdef RFLAG
  m=(int **) R_alloc(nrow+NR_END, sizeof(int*));
#else
  m=(int **) malloc((size_t)((nrow+NR_END)*sizeof(int*)));
#endif
  if (!m) nrerror("allocation failure 1 in matrix()");
  m += NR_END;
  m -= nrl;

  /* allocate rows and set pointers to them */
#ifdef RFLAG
  m[nrl]=(int *) R_alloc(nrow*ncol+NR_END, sizeof(int));
#else
  m[nrl]=(int *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(int)));
#endif
  if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
  m[nrl] += NR_END;
  m[nrl] -= ncl;

  for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

  /* return pointer to array of pointers to rows */
  return m;
}

float **submatrix(float **a, long oldrl, long oldrh, long oldcl, long oldch,
		  long newrl, long newcl)
     /* point a submatrix [newrl..][newcl..] to a[oldrl..oldrh][oldcl..oldch] */
{
  long i,j,nrow=oldrh-oldrl+1,ncol=oldcl-newcl;
  float **m;

  /* allocate array of pointers to rows */
#ifdef RFLAG  
  m=(float **) R_alloc(nrow+NR_END, sizeof(float*));
#else  
  m=(float **) malloc((size_t) ((nrow+NR_END)*sizeof(float*)));
#endif
  if (!m) nrerror("allocation failure in submatrix()");
  m += NR_END;
  m -= newrl;

  /* set pointers to rows */
  for(i=oldrl,j=newrl;i<=oldrh;i++,j++) m[j]=a[i]+ncol;

  /* return pointer to array of pointers to rows */
  return m;
}

float **convert_matrix(float *a, long nrl, long nrh, long ncl, long nch)
     /* allocate a float matrix m[nrl..nrh][ncl..nch] that points to the matrix
	declared in the standard C manner as a[nrow][ncol], where nrow=nrh-nrl+1
	and ncol=nch-ncl+1. The routine should be called with the address
	&a[0][0] as the first argument. */
{
  long i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1;
  float **m;

  /* allocate pointers to rows */
#ifdef RFLAG  
  m=(float **) R_alloc(nrow+NR_END, sizeof(float*));
#else  
  m=(float **) malloc((size_t) ((nrow+NR_END)*sizeof(float*)));
#endif
  if (!m) nrerror("allocation failure in convert_matrix()");
  m += NR_END;
  m -= nrl;

  /* set pointers to rows */
  m[nrl]=a-ncl;
  for(i=1,j=nrl+1;i<nrow;i++,j++) m[j]=m[j-1]+ncol;
  /* return pointer to array of pointers to rows */
  return m;
}

float ***f3tensor(long nrl, long nrh, long ncl, long nch, long ndl, long ndh)
     /* allocate a float 3tensor with range t[nrl..nrh][ncl..nch][ndl..ndh] */
{
  long i,j,nrow=nrh-nrl+1,ncol=nch-ncl+1,ndep=ndh-ndl+1;
  float ***t;

  /* allocate pointers to pointers to rows */
#ifdef RFLAG  
  t=(float ***) R_alloc(nrow+NR_END, sizeof(float**));
#else 
  t=(float ***) malloc((size_t)((nrow+NR_END)*sizeof(float**)));
#endif
  if (!t) nrerror("allocation failure 1 in f3tensor()");
  t += NR_END;
  t -= nrl;

  /* allocate pointers to rows and set pointers to them */
#ifdef RFLAG  
  t[nrl]=(float **) R_alloc(nrow*ncol+NR_END, sizeof(float*));
#else 
  t[nrl]=(float **) malloc((size_t)((nrow*ncol+NR_END)*sizeof(float*)));
#endif
  if (!t[nrl]) nrerror("allocation failure 2 in f3tensor()");
  t[nrl] += NR_END;
  t[nrl] -= ncl;

  /* allocate rows and set pointers to them */
#ifdef RFLAG  
  t[nrl][ncl]=(float *) R_alloc(nrow*ncol*ndep+NR_END, sizeof(float));
#else 
  t[nrl][ncl]=(float *) malloc((size_t)((nrow*ncol*ndep+NR_END)*sizeof(float)));
#endif
  if (!t[nrl][ncl]) nrerror("allocation failure 3 in f3tensor()");
  t[nrl][ncl] += NR_END;
  t[nrl][ncl] -= ndl;

  for(j=ncl+1;j<=nch;j++) t[nrl][j]=t[nrl][j-1]+ndep;
  for(i=nrl+1;i<=nrh;i++) {
    t[i]=t[i-1]+ncol;
    t[i][ncl]=t[i-1][ncl]+ncol*ndep;
    for(j=ncl+1;j<=nch;j++) t[i][j]=t[i][j-1]+ndep;
  }

  /* return pointer to array of pointers to rows */
  return t;
}

void free_vector(float *v, long nl, long nh)
     /* free a float vector allocated with vector() */
{
#ifndef RFLAG 
  free((FREE_ARG) (v+nl-NR_END));
#endif
}

void free_ivector(int *v, long nl, long nh)
     /* free an int vector allocated with ivector() */
{
#ifndef RFLAG  
  free((FREE_ARG) (v+nl-NR_END));
#endif
}

void free_cvector(unsigned char *v, long nl, long nh)
     /* free an unsigned char vector allocated with cvector() */
{
#ifndef RFLAG  
  free((FREE_ARG) (v+nl-NR_END));
#endif  
}

void free_lvector(unsigned long *v, long nl, long nh)
     /* free an unsigned long vector allocated with lvector() */
{
#ifndef RFLAG  
  free((FREE_ARG) (v+nl-NR_END));
#endif  
}

void free_dvector(double *v, long nl, long nh)
     /* free a double vector allocated with dvector() */
{
#ifndef RFLAG  
  free((FREE_ARG) (v+nl-NR_END));
#endif  
}

void free_matrix(float **m, long nrl, long nrh, long ncl, long nch)
     /* free a float matrix allocated by matrix() */
{
#ifndef RFLAG  
  free((FREE_ARG) (m[nrl]+ncl-NR_END));
  free((FREE_ARG) (m+nrl-NR_END));
#endif 
}

void free_dmatrix(double **m, long nrl, long nrh, long ncl, long nch)
     /* free a double matrix allocated by dmatrix() */
{
#ifndef RFLAG  
  free((FREE_ARG) (m[nrl]+ncl-NR_END));
  free((FREE_ARG) (m+nrl-NR_END));
#endif   
}

void free_imatrix(int **m, long nrl, long nrh, long ncl, long nch)
     /* free an int matrix allocated by imatrix() */
{
#ifndef RFLAG  
  free((FREE_ARG) (m[nrl]+ncl-NR_END));
  free((FREE_ARG) (m+nrl-NR_END));
#endif   
}

void free_submatrix(float **b, long nrl, long nrh, long ncl, long nch)
     /* free a submatrix allocated by submatrix() */
{
#ifndef RFLAG  
  free((FREE_ARG) (b+nrl-NR_END));
#endif 
}

void free_convert_matrix(float **b, long nrl, long nrh, long ncl, long nch)
     /* free a matrix allocated by convert_matrix() */
{
#ifndef RFLAG  
  free((FREE_ARG) (b+nrl-NR_END));
#endif   
}

void free_f3tensor(float ***t, long nrl, long nrh, long ncl, long nch,
		   long ndl, long ndh)
     /* free a float f3tensor allocated by f3tensor() */
{
#ifndef RFLAG  
  free((FREE_ARG) (t[nrl][ncl]+ndl-NR_END));
  free((FREE_ARG) (t[nrl]+ncl-NR_END));
  free((FREE_ARG) (t+nrl-NR_END));
#endif 
}

/* extensions, A.Trapletti */

void copy_ivector (int *v_src, int *v_dest, long nl, long nh)  
     /* copy integer vector v_src[nl..nh] to v_dest[nl..nh] */
{
  long i;
  
  for (i=nl; i<=nh; i++) v_dest[i] = v_src[i];
}

void copy_vector (float *v_src, float *v_dest, long nl, long nh)  
     /* copy float vector v_src[nl..nh] to v_dest[nl..nh] */
{
  long i;

  for (i=nl; i<=nh; i++) v_dest[i] = v_src[i];
}

void copy_dvector (double *v_src, double *v_dest, long nl, long nh)
     /* copy double vector v_src[nl..nh] to v_dest[nl..nh] */
{
  long i;

  for (i=nl; i<=nh; i++) v_dest[i] = v_src[i];
}
 
void copy_matrix (float **m_src, float **m_dest, long nrl, long nrh, long ncl, long nch)
     /* copy float matrix m_src[nrl..nrh][ncl..nch] to m_dest[nrl..nrh][ncl..nch] */
{
  long i, j;

  for (i=nrl; i<=nrh; i++)
    for (j=ncl; j<=nch; j++)
      m_dest[i][j] = m_src[i][j];
}

void copy_dmatrix (double **m_src, double **m_dest, long nrl, long nrh, long ncl, long nch)
     /* copy double matrix m_src[nrl..nrh][ncl..nch] to m_dest[nrl..nrh][ncl..nch] */
{
  long i, j;

  for (i=nrl; i<=nrh; i++)
    for (j=ncl; j<=nch; j++)
      m_dest[i][j] = m_src[i][j];
}

