/*
 * Portable Exception Handling for ANSI C.
 * Copyright (C) 1999 Kaz Kylheku <kaz@ashi.footprints.net>
 *
 * ---
 * Modified by Federico Spinazzi
 * Copyright (C) 2000 Federico Spinazzi <fspinazzi@hotmail.com>
 * The modifications are related to adaptation of the library to
 * support threads other than POSIX, using ad additional parameter
 * for each function. Global and thread-locally stored variables
 * have been removed.
 * The original license follows.
 * This code can be used under that license.
 * ---
 *
 * Free Software License:
 *
 * All rights are reserved by the author, with the following exceptions:
 * Permission is granted to freely reproduce and distribute this software,
 * possibly in exchange for a fee, provided that this copyright notice appears
 * intact. Permission is also granted to adapt this software to produce
 * derivative works, as long as the modified versions carry this copyright
 * notice and additional notices stating that the work has been modified.
 * This source code may be translated into executable form and incorporated
 * into proprietary software; there is no requirement for such software to
 * contain a copyright notice related to this source.
 *
 * $Id: except.c 1.5 2000/02/27 12:26:40 federico Exp $
 * $Name:  $
 * $Log: except.c $
 * Revision 1.5  2000/02/27 12:26:40  federico
 * Added additional void * parameter to memory management functions.
 *
 * Revision 1.4  2000/02/09 23:36:30  federico
 * Modified Win32 test to handle different allocators.
 * Tested 2 indipendent out of memory catchers, one with KazLib
 * exception, the other one with Win32 exception.
 * They seems to work.
 * LCC can't compile this
 *
 * Revision 1.3  2000/02/08 23:24:25  federico
 * Corrected the test.
 * Added a little test for Win32 Threads.
 * Not tested different allocators.
 * Not checked against memory leaks.
 *
 */

/*
 * TODO: check exstensively cleanup handlers for multiple thread
 */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <limits.h>
#include "except.h"

#define XCEPT_BUFFER_SIZE	1024

#ifdef WITH_RCSID
static const char rcsid[] = "$Id: except.c 1.5 2000/02/27 12:26:40 federico Exp $";
#endif

#define group except_group
#define code except_code
#define id except_id
#define message except_message
#define func except_func
#define context except_context
#define id except_id
#define size except_size
#define obj except_obj
#define jmp except_jmp
#define down except_down
#define type except_type
#define catcher except_catcher
#define cleanup except_cleanup
#define info except_info


static void
  (*get_catcher (struct except_params *ex)) (except_t *)
{
	assert (ex != NULL);
  return ex->uh_catcher_ptr;
}

static struct except_stacknode *
get_top (struct except_params *ex)
{
	assert (ex != NULL);
  return ex->top;
}

static void
set_top (struct except_params *ex, struct except_stacknode *newtop)
{
	assert (ex != NULL);
	/*
	 * assert (newtop != NULL);
	 */
  ex->top = newtop;
}



static void unhandled_catcher (except_t * except);

int
except_init (struct except_params *ex)
{
  int retval = 1;

  assert (ex != NULL);

  ex->uh_catcher_ptr = unhandled_catcher;
  ex->top = NULL;

  return retval;
}

void
except_deinit (struct except_params *ex)
{
  ex->uh_catcher_ptr = NULL;
  ex->top = NULL;
}



static int
match (const except_id_t * thrown, const except_id_t * caught)
{
  int group_match;
  int code_match;

  group_match =
    (caught->group == XCEPT_GROUP_ANY || caught->group == thrown->group);

  code_match =
    (caught->code == XCEPT_CODE_ANY || caught->code == thrown->code);

  return group_match && code_match;
}

static void
do_throw (struct except_params *ex, except_t * except)
{
  struct except_stacknode *top;

  assert (except->id.group != 0 && except->id.code != 0);

  /*
   * Walk down the stack:
   * - when a cleanup handler is found, esecute it;
   * - when a catch handler is found for the exception being
   *   thrown, (long)jump to it.
   */
  for (top = get_top (ex); top != 0; top = top->down)
    {
      if (top->type == XCEPT_CLEANUP)
        {
          top->info.cleanup->func (top->info.cleanup->context);
        }
      else
        {
          struct except_catch *catcher = top->info.catcher;
          const except_id_t *pi = catcher->id;
          size_t i;

          assert (top->type == XCEPT_CATCHER);

          for (i = 0; i < catcher->size; pi++, i++)
            {
              if (match (&except->id, pi))
                {
                  catcher->obj = *except;
                  set_top (ex, top);
                  longjmp (catcher->jmp, 1);
                }
            }
        }
    }

  set_top (ex, top);
  get_catcher (ex) (except);    /* unhandled exception */

  /* TODO: accept to abort? */
  abort ();
}

static void
unhandled_catcher (except_t * except)
{
  fprintf (stderr, "Unhandled exception (\"%s\", group=%ld, code=%ld)\n",
           except->message, except->id.group, except->id.code);
  abort ();
}

static void
stack_push (struct except_params *ex,
            struct except_stacknode *node)
{
  node->down = get_top (ex);
  set_top (ex, node);
}

void
except_setup_clean (struct except_params *ex,
                    struct except_stacknode *esn,
                    struct except_cleanup *ecl,
                    void (*cleanf) (void *),
                    void *context)
{
  esn->type = XCEPT_CLEANUP;
  ecl->func = cleanf;
  ecl->context = context;
  esn->info.cleanup = ecl;
  stack_push (ex, esn);
}

void
except_setup_try (struct except_params *ex,
                  struct except_stacknode *esn,
                  struct except_catch *ech,
                  const except_id_t id[],
                  size_t size)
{
  ech->id = id;
  ech->size = size;
  esn->type = XCEPT_CATCHER;
  esn->info.catcher = ech;
  stack_push (ex, esn);
}

struct except_stacknode *
except_pop (struct except_params *ex)
{
  struct except_stacknode *top = get_top (ex);
  set_top (ex, top->down);
  return top;
}

void
except_rethrow (struct except_params *ex, except_t * except)
{
  struct except_stacknode *top = get_top (ex);
  assert (top != 0);
  assert (top->type == XCEPT_CATCHER);
  assert (&top->info.catcher->obj == except);
  set_top (ex, top->down);
  do_throw (ex, except);
}

void
except_throw (struct except_params *ex,
              long group,
              long code,
              const char *msg)
{
  except_t except;

  except.id.group = group;
  except.id.code = code;
  except.message = msg;

  do_throw (ex, &except);
}

void
except_throwd (struct except_params *ex,
               long group,
               long code,
               const char *msg,
               void *data)
{
  except_t except;

  except.id.group = group;
  except.id.code = code;
  except.message = msg;

  do_throw (ex, &except);
}

void (*except_unhandled_catcher (struct except_params *ex,
                             void (*new_catcher) (except_t *))) (except_t *)
{
  void (*old_catcher) (except_t *) = get_catcher (ex);
  ex->uh_catcher_ptr = new_catcher;
  return old_catcher;
}

#undef except_code
#undef except_group
#undef except_message
#undef except_data

unsigned long
except_code (except_t * ex)
{
  return ex->id.code;
}

unsigned long
except_group (except_t * ex)
{
  return ex->id.group;
}

const char *
except_message (except_t * ex)
{
  return ex->message;
}

#ifdef TEST_EXCEPT

#include <stdio.h>
#include <ctype.h>

static void
cleanup (void *arg)
{
  printf ("cleanup(\"%s\") called\n", (char *) arg);
}

static void
bottom_level (struct except_params *ex_pars)
{
  char buf[256];
  printf ("throw exception? ");
  fflush (stdout);
  fgets (buf, sizeof (buf), stdin);

  if (toupper (buf[0]) == 'Y')
    except_throw (ex_pars, 1, 1, "nasty exception");
}

static void
top_level (struct except_params *ex_pars)
{
  except_cleanup_push (cleanup, "argument");
  bottom_level (ex_pars);
  except_cleanup_pop (0);
}

void
test_a (struct except_params *ex_pars, size_t num)
{
  static const except_id_t catch[] =
  {
    {1, 1},
    {1, 2}};
  except_t *exc;
  char *ptr_a = NULL;

  fprintf (stderr, "Trying to allocate %ld bytes ...\n", num);
  ptr_a = except_alloc (ex_pars, num);
  fprintf (stderr, "Ok, at %p ...\n", ptr_a);

  except_try_push (catch, 1, &exc);

  if (exc == 0)
    {
      int i;
      char *ptr_b;
      const int SIZE = 100000;

      for (i = 1; i < num; i++)
        {
          fprintf (stderr, "\rTrying to allocate %d bytes ...", i * SIZE);
          ptr_b = except_alloc (ex_pars, i * SIZE);
          ptr_b[i * SIZE / 2] = 'f';
          except_free (ex_pars, ptr_b);
        }
    }

  fprintf (stderr, "Deallocating at %p ...\n", ptr_a);
  ptr_a = except_alloc (ex_pars, num);

  if (exc != 0)
    {
      except_rethrow (ex_pars, exc);
    }

  except_try_pop ();
}

#ifdef WIN32

#include <windows.h>
#include <process.h>
/*
 Create a chBEGINTHREADEX macro that calls the C run-time's
 _beginthreadex function. The C run-time library doesn't
 want to have any reliance on Win32 data types such as
 HANDLE. This means that a Win32 programmer needs to cast
 the return value to a HANDLE. This is terribly inconvenient,
 so I have created this macro to perform the casting.
*/
typedef unsigned (__stdcall * PTHREAD_START) (void *);

#define chBEGINTHREADEX(lpsa, cbStack, lpStartAddr, \
   lpvThreadParm, fdwCreate, lpIDThread)            \
      ((HANDLE)_beginthreadex(                      \
         (void *) (lpsa),                           \
         (unsigned) (cbStack),                      \
         (PTHREAD_START) (lpStartAddr),             \
         (void *) (lpvThreadParm),                  \
         (unsigned) (fdwCreate),                    \
         (unsigned *) (lpIDThread)))

enum
  {
    THREAD_NUMBER = 2
  };


HANDLE gHeap = NULL;

void *
my_malloc (size_t size)
{
  LPVOID *p;
  if (gHeap == NULL)
    {
      fprintf (stderr, "Creating a new Heap\n");
      gHeap = HeapCreate (HEAP_NO_SERIALIZE | HEAP_GENERATE_EXCEPTIONS, 4026, 0);
    }


  p = HeapAlloc (gHeap, HEAP_NO_SERIALIZE | HEAP_GENERATE_EXCEPTIONS, size);
  //p = HeapAlloc (gHeap, HEAP_NO_SERIALIZE, size);

  return p;
}

void
my_free (void *ptr)
{
  HeapFree (gHeap, HEAP_NO_SERIALIZE, ptr);
}

DWORD WINAPI
FirstThread (LPVOID lpvThreadParam)
{
  char *to_free;
  DWORD dwResult = 0;
  static const except_id_t catch[] =
  {
    {1, 1},
    {1, 2}};
  except_t *exc;
  struct except_params pars;
  struct except_params *ex_pars = &pars;
  int num = (int) lpvThreadParam;

  __try
  {
    fprintf (stderr, "First Thread started\n");

    except_init (ex_pars, my_malloc, my_free);

    to_free = except_alloc (ex_pars, 30000);

    except_cleanup_push (ex_pars->dealloc, to_free);
    except_cleanup_push (cleanup, "argument");

    except_try_push (catch, 2, &exc);

    if (!exc)
      {

        int i;
        char *ptr_b;
        const int SIZE = 900000;

        for (i = 1; i < num; i++)
          {
            fprintf (stderr, "\rFirst Thread: "
                     "trying to allocate %d bytes ...", i * SIZE);
            ptr_b = ex_pars->alloc (i * SIZE);
            ptr_b[i * SIZE / 2] = 'f';
            ex_pars->dealloc (ptr_b);
						//except_free (ex_pars, ptr_b);
          }
      }
    else
      {
        /* outer catch */
        printf ("caught exception (%s): \"%s\", s=%ld, c=%ld\n", __func__,
                except_message (exc), except_group (exc), except_code (exc));
      }

    except_try_pop ();
    except_cleanup_pop (except_call);
    except_cleanup_pop (0);
  }

  __except (EXCEPTION_EXECUTE_HANDLER)
  {
    fprintf (stderr, "\rFirst Thread: Win32 exception caught ...\n");
    HeapDestroy (gHeap);
  }

  return dwResult;
}

DWORD WINAPI
SecondThread (LPVOID lpvThreadParam)
{
  DWORD dwResult = 0;
  static const except_id_t catch[] =
  {
    {1, 1},
    {1, 2}};
  except_t *exc;
  struct except_params pars;
  struct except_params *ex_pars = &pars;
  int num = (int) lpvThreadParam;

  fprintf (stderr, "Second Thread started\n");

  except_init (ex_pars, malloc, free);

  except_try_push (catch, 2, &exc);

  if (!exc)
    {

      int i;
      char *ptr_b;
      const int SIZE = 100000;

      for (i = 1; i < num; i++)
        {
          fprintf (stderr, "\rSecond Thread: "
                   "trying to allocate %d bytes ...", i * SIZE);
          ptr_b = except_alloc (ex_pars, i * SIZE);
          ptr_b[i * SIZE / 2] = 'f';
          except_free (ex_pars, ptr_b);
        }
    }
  else
    {
      /* outer catch */
      printf ("caught exception (%s): \"%s\", s=%ld, c=%ld\n", __func__,
              except_message (exc), except_group (exc), except_code (exc));
    }
  except_try_pop ();

	fprintf (stderr, "Second thread exiting ...\n");

  return dwResult;
}

void
test_threads (void)
{
  int x = 4000;
  HANDLE hThread[THREAD_NUMBER];
  DWORD dwThreadId[THREAD_NUMBER];

  fprintf (stderr, "Testing Win32 threads ...\n");


  hThread[0] = chBEGINTHREADEX (NULL, 0,
                              FirstThread, (LPVOID) & x, 0, &dwThreadId[0]);

  hThread[1] = chBEGINTHREADEX (NULL, 0,
                             SecondThread, (LPVOID) & x, 0, &dwThreadId[1]);

  WaitForMultipleObjects (THREAD_NUMBER, hThread, TRUE, INFINITE);

  CloseHandle (hThread[0]);
  CloseHandle (hThread[1]);

  fprintf (stderr, "Win32 threads tested ...\n");
}

#endif


int
main (int argc, char **argv)
{
  static const except_id_t catch[] =
  {
    {1, 1},
    {1, 2}};
  except_t *ex;
  struct except_params pars;
  struct except_params *ex_pars = &pars;

  /*
   * Nested exception ``try blocks''
   */

  /* outer */


  except_init (ex_pars, malloc, free);

  except_try_push (catch, 2, &ex);

  if (!ex)
    {
      /* inner */
      test_a (ex_pars, argc > 1 ? atoi (argv[1]) : 300);

#ifdef WIN32
      test_threads ();
#endif

      except_try_push (catch, 2, &ex);
      if (!ex)
        {
          top_level (ex_pars);
        }
      else
        {
          /* inner catch */
          fprintf (stderr, "caught exception (inner):"
                   " \"%s\", s=%ld, c=%ld\n",
                   except_message (ex), except_group (ex), except_code (ex));
          except_rethrow (ex_pars, ex);
        }
      except_try_pop ();
    }
  else
    {
      /* outer catch */
      printf ("caught exception (outer): \"%s\", s=%ld, c=%ld\n",
              except_message (ex), except_group (ex), except_code (ex));
    }
  except_try_pop ();

  //except_throw (ex_pars, 99, 99, "exception in main");

  return 0;
}


#endif


