% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EValue.R
\name{evalue}
\alias{evalue}
\title{Compute an E-value for unmeasured confounding}
\usage{
evalue(est, lo = NA, hi = NA, se = NA, delta = 1, true = c(0, 1), ...)
}
\arguments{
\item{est}{The effect estimate that was observed but which is suspected to be
biased. A number of class "estimate" (constructed with \code{\link[=RR]{RR()}}, \code{\link[=OR]{OR()}},
\code{\link[=HR]{HR()}}, \code{\link[=OLS]{OLS()}}, or \code{\link[=MD]{MD()}}; for E-values for risk differences,
see \code{\link[=evalues.RD]{evalues.RD()}}).}

\item{lo}{Optional. Lower bound of the confidence interval. If not an object
of class "estimate", assumed to be on the same scale as \code{est}.}

\item{hi}{Optional. Upper bound of the confidence interval. If not an object
of class "estimate", assumed to be on the same scale as \code{est}.}

\item{se}{The standard error of the point estimate, for \code{est} of class "OLS"}

\item{delta}{The contrast of interest in the exposure, for \code{est} of class "OLS"}

\item{true}{A number to which to shift the observed estimate to. Defaults to
1 for ratio measures (\code{\link[=RR]{RR()}}, \code{\link[=OR]{OR()}}, \code{\link[=HR]{HR()}}) and 0 for additive measures
(\code{\link[=OLS]{OLS()}}, \code{\link[=MD]{MD()}}).}

\item{...}{Arguments passed to other methods.}
}
\description{
Returns a data frame containing point estimates, the lower confidence limit,
and the upper confidence limit on the risk ratio scale (possibly through an
approximate conversion) as well as E-values for the point estimate and the
confidence interval limit closer to the null.
}
\details{
An E-value for unmeasured confounding is minimum strength of
association, on the risk ratio scale, that unmeasured confounder(s) would
need to have with both the treatment and the outcome to fully explain away
a specific treatment–outcome association, conditional on the measured
covariates.

The estimate is converted appropriately before the E-value is calculated.
See \link[=convert_measures]{conversion functions} for more details. The point
estimate and confidence limits after conversion are returned, as is the
E-value for the point estimate and the confidence limit closest to the
proposed "true" value (by default, the null value.)

For an \code{\link[=OLS]{OLS()}} estimate, the E-value is for linear regression with a
continuous exposure and outcome. Regarding the continuous exposure, the
choice of \code{delta} defines essentially a dichotomization in the
exposure between hypothetical groups of subjects with exposures equal to an
arbitrary value \emph{c} versus to another hypothetical group with
exposures equal to \emph{c} + \code{delta}.

For example, if resulting E-value is 2, this means that unmeasured
confounder(s) would need to double the probability of a subject's having
exposure equal to \emph{c} + \code{delta} instead of \emph{c}, and would
also need to double the probability of being high versus low on the
outcome, in which the cutoff for "high" versus "low" is arbitrary subject
to some distributional assumptions (Chinn, 2000).
}
\examples{
# compute E-value for leukemia example in VanderWeele and Ding (2017)
evalue(RR(0.80), 0.71, 0.91)

# you can also pass just the point estimate
# and return just the E-value for the point estimate with summary()
summary(evalue(RR(0.80)))

# demonstrate symmetry of E-value
# this apparently causative association has same E-value as the above
summary(evalue(RR(1 / 0.80)))

# E-value for a non-null true value
summary(evalue(RR(2), true = 1.5))

## Hsu and Small (2013 Biometrics) Data
## sensitivity analysis after log-linear or logistic regression
head(lead)

## log linear model -- obtain the conditional risk ratio
lead.loglinear = glm(lead ~ ., family = binomial(link = "log"),
                         data = lead[,-1])
est_se = summary(lead.loglinear)$coef["smoking", c(1, 2)]

est      = RR(exp(est_se[1]))
lowerRR  = exp(est_se[1] - 1.96*est_se[2])
upperRR  = exp(est_se[1] + 1.96*est_se[2])
evalue(est, lowerRR, upperRR)

## logistic regression -- obtain the conditional odds ratio
lead.logistic = glm(lead ~ ., family = binomial(link = "logit"),
                        data = lead[,-1])
est_se = summary(lead.logistic)$coef["smoking", c(1, 2)]

est      = OR(exp(est_se[1]), rare = FALSE)
lowerOR  = exp(est_se[1] - 1.96*est_se[2])
upperOR  = exp(est_se[1] + 1.96*est_se[2])
evalue(est, lowerOR, upperOR)

## linear regression
# standardizing conservatively by SD(Y)
ols = lm(age ~ income, data = lead)
est = OLS(ols$coefficients[2], sd = sd(lead$age))

# for a 1-unit increase in income 
evalue(est = est, 
       se = summary(ols)$coefficients['income', 'Std. Error'])

# for a 0.5-unit increase in income
evalue(est = est,
       se = summary(ols)$coefficients['income', 'Std. Error'],
       delta = 0.5)

# E-value for Cohen's d = 0.5 with SE = 0.25
evalue(est = MD(.5), se = .25)

# compute E-value for HR = 0.56 with CI: [0.46, 0.69]
# for a common outcome
evalue(HR(0.56, rare = FALSE), lo = 0.46, hi = 0.69)
# for a rare outcome
evalue(HR(0.56, rare = TRUE), lo = 0.46, hi = 0.69)
}
\references{
\enumerate{
\item Ding & VanderWeele (2016). Sensitivity analysis without assumptions. \emph{Epidemiology.} 27(3), 368.
\item VanderWeele & Ding (2017). Sensitivity analysis in observational research: Introducing the E-value. \emph{Annals of Internal Medicine.} 27(3), 368.
}
}
\keyword{e-value}
