### rtweet: Collecting Twitter data

# Retrieving Tweets via Twitter's Search API

This vignette demonstrates the `search_tweets()` function found in 
the `rtweet` package.

To get started, make sure you've installed rtweets and then load the package.

``` r
#install from CRAN
#install.packages("rtweet")

#install the github version
#devtools::install_github("mkearney/rtweet")

library(rtweet)
```

## Search tweets

In order to use [Twitter's search API](https://dev.twitter.com/rest/public/search), 
you'll need to [create and store your personal access token](https://github.com/mkearney/rtweet/blob/master/README.md).
Fortunately, once you've successfully configured your access token, 
`rtweet` will call your token for you, so you won't have to deal with
it again.

One of the most useful functions in `rtweet` is `search_twitter()`,
which returns detailed results for custom Twitter searches. In the
example below, I've searched (q for query) for the term "election" 
`q = "election"`, asking for 1000 tweets `count = 1000` using the
english language `lang = "en"`. As you can see from the output below,
`search_twitter()` returns a properly formatted and organized data
frame for each tweet (1,000 rows) that contains information on 
26 different variables (26 columns)!

``` r
srch <- search_tweets(q = "election", n = 1000, lang = "en")
srch$tweets
srch$users
```


## Analyze text of Tweets

Returning a data set of search results makes it easy to perform a variety 
of approaches to analyze the Twitter data. If, for example, you wanted to
analyze the text of the tweets (variable name of `text`), the `rtweet`
package includes a `clean_tweets()` function, which allows you to identify
non-trivial words used in each Tweet. (Note: because this is a convenience 
function early in its development, it may not always follow the best 
practices of text wrangling.)

``` r
wrds <- clean_tweets(srch$tweets, min = 2, exclude_words = "election")
wrds[1:6]
srch$tweets[1:6]
```

Also included in the return object generated by `search_tweet()` are `user_mentions` and
`hashtags`. In the example below, I've estimated the number of Hillary Clinton references 
by summing `user_mentions` (+1 for mentioning Hillary's `user_id`) and textual mentions
(+1 for each instance of hillary or clinton in the text of each tweet). I did the same for Donald Trump.

``` r
srch_df$hillary <- sapply(srch_df$user_mentions, function(x) sum(grepl("1339835893", unlist(x)), na.rm = TRUE)) + 
  sapply(wrds, function(x) sum(grepl("hillary|clinton", unlist(x)), na.rm = TRUE))

srch_df$donald <- sapply(srch_df$user_mentions, function(x) sum(grepl("25073877", unlist(x)), na.rm = TRUE)) + 
  sapply(wrds, function(x) sum(grepl("donald|trump", unlist(x)), na.rm = TRUE))

srch_df$withher <- as.numeric(srch_df$hillary > srch_df$donald)
srch_df$withher <- srch_df$withher - as.numeric(srch_df$hillary < srch_df$donald)
```

## Plotting

I then plotted these new variables using `ggplot2` (a cheat code package for 
generating pretty plots).

``` r
library(ggplot2)
theme_set(theme_minimal())
ggplot(srch_df, aes(x = hillary, y = donald, fill = factor(withher), alpha = .9)) + 
  geom_jitter(size = 3, shape = 21) + 
  theme(legend.position = "none")
```
<p align="center">
<img src="files/search_tweets_plot1.png" alt="plot">
</p>
