% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcSeasCal.R
\name{calcSeasCal}
\alias{calcSeasCal}
\title{Wet Season Calendar}
\usage{
calcSeasCal(data, onsetWind.start, onsetWind.end, cessaWind.end, soilWHC)
}
\arguments{
\item{data}{An R dataframe returned by \code{\link{calcWatBal}} or
a dataframe with similar parameters.}

\item{onsetWind.start}{The earliest possible start date for the onset window
in "MM-DD" format.}

\item{onsetWind.end}{The latest possible end date for the onset window
in "MM-DD" format.}

\item{cessaWind.end}{The latest possible end date for the cessation window
in "MM-DD" format.}

\item{soilWHC}{The available soil water holding capacity at
root zone depth (in mm).}
}
\value{
A dataframe containing the following columns:
\describe{
\item{Year}{The year of the season (YYYY).}
\item{OnsetDate}{The onset date, formatted as "YYYY-MM-DD".}
\item{OnsetDOY}{The Julian day (DOY) of the onset.}
\item{OnsetValue}{The number of days since \code{onsetWind.start}.}
\item{CessationDate}{The cessation date, formatted as "YYYY-MM-DD".}
\item{CessationDOY}{The Julian day (DOY) of the cessation.}
\item{CessationValue}{The number of days since \code{onsetWind.start}.}
\item{Duration}{The duration of the wet season (in days).}
}
}
\description{
Estimates the wet season calendar, including the onset date,
cessation date, and duration, based on an agroclimatic approach. The
function relies on daily soil water balance parameters.
}
\details{
The agroclimatic approach defines the wet season based on the balance
between precipitation and potential evapotranspiration (PET). The wet season
begins when the moisture available to crops exceeds their
evapotranspiration demands, ensuring optimal growth conditions.

\strong{Onset:}
The wet season onset is defined as the first day after \code{onsetWind.start}
when the ratio of actual evapotranspiration (AET) to potential
evapotranspiration (PET) exceeds 0.5 for at least 5 consecutive days, and the
soil moisture remains above 25\% of the available soil water holding capacity
(\code{soilWHC}) for a minimum of 20 consecutive days, ensuring sufficient
moisture availability for plant growth.

\strong{Cessation:}
The wet season ends on the first day after \code{onsetWind.end} when the
AET/PET ratio falls below 0.5 for at least 5 consecutive days, and the
soil moisture remains below 25\% of the available soil water holding capacity
(\code{soilWHC}) for a minimum of 12 consecutive days.

\strong{Duration:}
The total duration of the wet season is the number of days between
onset and cessation.
}
\examples{
\donttest{
## Load example data:
data(AgroClimateData)

## Estimate daily PET:
PET <- calcEto(AgroClimateData, method = "PM", Zh = 10)

## Add the estimated PET 'ET.Daily' to a new column in AgroClimateData:
AgroClimateData$Eto <- PET$ET.Daily

## Estimate daily water balance for the soil having 100mm of WHC:
watBal.list <- calcWatBal(data = AgroClimateData, soilWHC = 100)
watBal <- watBal.list$data

## seasonal calendar is estimated for the onset window ranges from
## 01 September to 31 January having a soil with 100mm of WHC:

soilWHC <- 100
onsetWind.start <- "09-01"
onsetWind.end <- "01-31"
cessaWind.end <- "06-30"

seasCal.dF <- calcSeasCal(
  data = watBal, onsetWind.start, onsetWind.end,
  cessaWind.end, soilWHC
)

str(seasCal.dF)
}
}
\references{
FAO, 1977. Crop water requirements. FAO Irrigation and Drainage Paper No. 24,
by Doorenbos J. and W.O. Pruitt. FAO, Rome, Italy.

FAO, 1978. Forestry for Local Community Development. FAO Forestry Paper
No. 7, FAO, Rome.

FAO, 1986. Early Agrometeorological Crop Yield Forecasting. FAO Plant
Production and Protection Paper No. 73, by M. Frère and G.F.
Popov. FAO, Rome.
}
\seealso{
\code{\link{calcEto}, \link{calcWatBal}}
}
