% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/proLikelihood.R
\name{proLikelihood}
\alias{proLikelihood}
\alias{averageBE}
\alias{totalVarianceBE}
\alias{withinVarianceBE}
\title{Calculate profile likelihood for bioequivalence data}
\usage{
proLikelihood(
  dat,
  colSpec = list(),
  theta = NULL,
  xlow,
  xup,
  xlength = 100,
  method
)

averageBE(dat, colSpec = list(), theta = NULL, xlow, xup, xlength)

totalVarianceBE(dat, colSpec = list(), theta = NULL, xlow, xup, xlength)

withinVarianceBE(dat, colSpec = list(), theta = NULL, xlow, xup, xlength)
}
\arguments{
\item{dat}{data frame contains BE data (AUC and Cmax) with missing data allowed.}

\item{colSpec}{a named list that should specify columns in \sQuote{dat}; \sQuote{subject} (subject ID),
\sQuote{formula} (must be coded as T or R, where T for test drug and R for reference drug), and \sQuote{y} (either AUC or Cmax) are
required. \sQuote{period} and \sQuote{seq} may also be provided.
The \sQuote{formula} column should identify a test or a reference drug with R and T.}

\item{theta}{An optional numeric vector contains initial values of the parameters for use in optimization.
For example, in a 2x4 fully replicated design, the vector is [mu, p2, p3, p4, S, phi,log(sbt2), log(sbr2), log(swt2), log(sbr2), rho], where  
\sQuote{mu} is the population mean for the reference drug when there are no period or sequence effects; \sQuote{p2} to \sQuote{p4} are fixed 
period effects with period 1 as the reference period; \sQuote{S} the fixed sequence effect with seq 1 as the reference sequence; \sQuote{phi} 
is the mean difference between the two drugs; \sQuote{sbt2} and \sQuote{sbr2} are between-subject variances for the test and reference drugs, 
respectively; \sQuote{swt2} and \sQuote{swr2} are within-subject variances for the test and reference drugs, respectively; \sQuote{rho} is 
the correlation coefficient within a subject. When \sQuote{theta} (default is null) is not provided, the function 
will choose the starting values automatically based on a linear mixed-effects model. If users want to provide these values, for method 
\sQuote{average} (mean difference), user may put any value for \sQuote{phi}. Similarly, for method \sQuote{total}, user can put any value 
for \sQuote{log(sbt2)}, and for method \sQuote{within}, user can put any value for \sQuote{log(swt2)}.}

\item{xlow}{numeric value, the lower limit of x-axis for the profile likelihood plot, at which the profile likelihood is calculated. It is 
optional and can be automatically generated using the maximum likelihood estimate (MLE) depending on the \sQuote{method}. We strongly 
recommend users trying a better value that would better fit for purpose.}

\item{xup}{numeric value, the upper limit of x-axis for the profile likelihood plot, at which the profile likelihood is calculated. It is 
optional and can be automatically generated using the MLE depending on the \sQuote{method}. We strongly recommend users trying 
a better value that would better fit for purpose.}

\item{xlength}{numeric value. Defaults to 100. It is the number of grids between the lower and upper limits, which controls smoothness of 
the curve. It will take longer time to run for larger number of grids, but we strongly recommend users using a larger number than the default 
value.}

\item{method}{character value. Should be one of \sQuote{average}, \sQuote{total}, or \sQuote{within}. 
\sQuote{average} will provide the profile likelihood for the mean difference between test and reference drugs. 
\sQuote{total} will provide the profile likelihood for the total standard deviation ratio of test to reference drug. \sQuote{within} 
will provide the profile likelihood for the within-subject standard deviation ratio of test to reference drug when appropriate.}
}
\value{
A \sQuote{proLikelihood} object, with elements \sQuote{poi}, \sQuote{maxLik}, \sQuote{MAX}, \sQuote{LI}, and \sQuote{method}. 
\sQuote{poi} and \sQuote{maxLik} are the interested parameter (mean difference, total standard deviation ratio 
or within-subject standard deviation ratio) values and the corresponding profile likelihood values, respectively. \sQuote{MAX} is the MLE 
estimate for that parameter. \sQuote{LI} is the likelihood intervals with the 1/4.5, 1/8 and 1/32 intervals. 
\sQuote{method} is one of \sQuote{average},\sQuote{total}, and \sQuote{within}.
}
\description{
This is a general function to calculate the profile likelihoods for the mean difference, total standard deviation ratio, 
and within-subject standard deviation ratio of the test drug to the reference drug from bioequivalence (BE) study data. 
Standardized profile likelihood plots with the 1/8 and 1/32 likelihood intervals can be generated using the plot method. 
The within-subject standard deviation ratio can be obtained only for a fully replicated 2x4 or a partially replicated 2x3 design.
}
\details{
This function implements a likelihood method for evaluating BE for pharmacokinetic parameters (AUC and Cmax) (see reference below). It accepts a dataframe collected with various crossover designs commonly used in BE studies such as 
a fully replicated crossover design (e.g., 2x4 two-sequence, four-period, RTRT/TRTR), a partially replicated crossover design 
(e.g., 2x3, two-sequence, three-period, RTR/TRT), and a two-sequence, two-period, crossover design design (2x2, RT/TR), 
where "R" stands for a reference drug and "T" stands for a test drug. 
It allows missing data (for example, a subject may miss the period 2 data) and utilizes all available data. It will 
calculate the profile likelihoods for the mean difference, total standard deviation ratio, and within-subject standard deviation ratio. 
Plots of standardized profile likelihood can be generated and provide evidence for various quantities of interest for evaluating 
BE in a unified framework.
}
\examples{
\donttest{
data(dat)
cols <- list(subject = 'subject', formula = 'formula', y = 'AUC')
p4a <- proLikelihood(dat, colSpec = cols, xlength = 300, method = 'average')
p4t <- proLikelihood(dat, colSpec = cols, xlength = 300, method = 'total')
p4w <- proLikelihood(dat, colSpec = cols, xlength = 300, method = 'within')
# three period case
dd3 <- dat[dat$period < 4,]
p3a <- averageBE(dd3, colSpec = cols, xlength = 300)
p3t <- totalVarianceBE(dd3, colSpec = cols, xlength = 300)
p3w <- withinVarianceBE(dd3, colSpec = cols, xlength = 300)
# two period case
dd2 <- dat[dat$period < 3,]
p2a <- averageBE(dd2, colSpec = cols, xlength = 300)
p2t <- totalVarianceBE(dd2, colSpec = cols, xlength = 300)
}

}
\references{
Liping Du and Leena Choi, Likelihood approach for evaluating bioequivalence of highly variable drugs, Pharmaceutical Statistics, 14(2): 82-94, 2015
}
