#' CLIC function based on LIC with cosine_random distributed errors
#'
#'The CLIC function builds on the LIC function by introducing the assumption that the error term follows a cosine_random distribution, thereby enhancing the length and information optimisation criterion.
#' @param X is a design matrix
#' @param Y is a random response vector of observed values
#' @param alpha is the significance level
#' @param K is the number of subsets
#' @param nk is the sample size of subsets
#' @param dist_type is the type where the error term obeys a cosine_random distribution
#'
#' @return MUopt, Bopt, MAEMUopt, MSEMUopt, opt, Yopt
#' @export
#'
#' @references
#' Guo, G., Song, H. & Zhu, L. The COR criterion for optimal subset selection in distributed estimation. \emph{Statistics and Computing}, 34, 163 (2024). \doi{10.1007/s11222-024-10471-z}
#'
#' Guo, G., Sun, Y., Qian, G., & Wang, Q. (2022). LIC criterion for optimal subset selection in distributed interval estimation. \emph{Journal of Applied Statistics}, 50(9), 1900-1920. \doi{10.1080/02664763.2022.2053949}.
#'
#' Chang, D., Guo, G. (2024). LIC: An R package for optimal subset selection for distributed data. \emph{SoftwareX}, 28, 101909.
#'
#' Jing, G., & Guo, G. (2025). TLIC: An R package for the LIC for T distribution regression analysis. \emph{SoftwareX}, 30, 102132.
#'
#' Chang, D., & Guo, G. (2025). Research on Distributed Redundant Data Estimation Based on LIC. \emph{IAENG International Journal of Applied Mathematics}, 55(1), 1-6.
#'
#' Gao, H., & Guo, G. (2025). LIC for Distributed Skewed Regression. \emph{IAENG International Journal of Applied Mathematics}, 55(9), 2925-2930.
#'
#' Zhang, C., & Guo, G. (2025). The optimal subset estimation of distributed redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 270–277.
#'
#' Jing, G., & Guo, G. (2025). Student LIC for distributed estimation. \emph{IAENG International Journal of Applied Mathematics}, 55(3), 575–581.
#'
#' Liu, Q., & Guo, G. (2025). Distributed estimation of redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 332–337.
#'
#' @examples
#' set.seed(12)
#' n <- 1200
#' nr <- 200
#' p <- 5
#' data <- cerr(n, nr, p, dist_type = "cosine_random")
#' CLIC(data$X, data$Y, alpha = 0.05, K = 10, nk = n / 10, dist_type = "cosine_random")
#'
#' @importFrom stats qt
#' @importFrom stats runif
#' @importFrom stats rt
#' @importFrom LaplacesDemon rst
#' @importFrom LaplacesDemon rstp
#' @importFrom fBasics rght
#' @importFrom fBasics rsght
CLIC <- function(X, Y, alpha = 0.05, K = 10, nk = NULL, dist_type = "cosine_random") {
  if (is.null(nk)) {
    nk <- nrow(X) / K
  }

  n <- nrow(X)
  p <- ncol(X)

  N <- L1 <- rep(NA, K)
  Rm <- matrix(rep(0, nk * K), ncol = K)
  mr <- matrix(rep(0, K * nk), ncol = nk)

  for (i in 1:K) {
    mr[i, ] <- sample(1:n, nk, replace = FALSE)
    r <- matrix(c(1:nk, mr[i, ]), ncol = nk, byrow = TRUE)
    Rm[, i] <- r[2,]
    R <- matrix(rep(0, nk * n), ncol = n)
    R[t(r)] <- 1

    X1 <- R %*% X
    Y1 <- R %*% Y

    Hr <- X1 %*% solve(crossprod(X1)) %*% t(X1)
    I1 <- diag(rep(1, nk))

    SX <- (t(Y1) %*% (I1 - Hr) %*% Y1) / (nk - p)
    SY <- sqrt(t(Y1) %*% (I1 - Hr) %*% Y1) / (nk - p)
    C1 <- sum(diag(X1 %*% solve(crossprod(X1)) %*% t(X1))) / nk

    if (dist_type == "cosine_random") {
      L1[i] <- 2 * SY * C1 * qt(1 - alpha / 2, nk - p)
    } else if (dist_type == "cosine_rejection_sampling") {
      L1[i] <- 2 * SY * C1 * qt(1 - alpha / 2, nk - p)
    } else if (dist_type == "cosine_metropolis_hastings") {
      L1[i] <- 2 * SY * C1 * qt(1 - alpha / 2, nk - p)
    }

    N[i] <- det(t(X1) %*% X1)
  }

  opt1 <- Rm[, which.min(L1)]
  opt2 <- Rm[, which.max(N)]
  opt <- intersect(opt1, opt2)
  Yopt <- Y[opt]
  Xopt <- X[opt, ]
  Bopt <- solve(crossprod(Xopt)) %*% t(Xopt) %*% Yopt
  MUopt <- Xopt %*% Bopt
  Nopt <- length(Yopt)
  E5 <- (t(Yopt - MUopt) %*% (Yopt - MUopt)) / Nopt
  A5 <- sum(abs(Yopt - MUopt)) / Nopt

  return(list(MUopt = MUopt, Bopt = Bopt, MAEMUopt = A5, MSEMUopt = E5, opt = opt, Yopt = Yopt))
}
