#' plotSig
#'
#' This function is used to create a plot for the specified signature to look at
#' the contribution of each of the components to the signatures
#'
#' @param sigs The dataset of component contribution to each signature
#' @param sigNum The signature number to plot
#' @keywords plot signature
#' @return displays a plot of the signature
#' @import ggplot2
#' @export
#' @examples
#' plotSig(referenceExp$sigs, 1) #Plots first signature
plotSig = function(sigs, sigNum) {
   
   if (!is.numeric(sigNum) || sigNum <= 0 || sigNum > ncol(sigs)) {
      msg = paste0("Invalid signature number given. Must be in range of 1 to ",
                   as.character(ncol(sigs)),".")
      stop(msg)
   }
   
   # Normalizes the values for plotting
   sigs = t(apply(sigs,1,function(x){x/sum(x)}))
   sigs = as.data.frame(sigs[,sigNum])

   #Creates uppercase feature names and combines that to make the labels to
   #the signature plots
   sigs[grep("segsize",rownames(sigs)),2]="Segsize"
   sigs[grep("bp10MB",rownames(sigs)),2]="Bp10MB"
   sigs[grep("osCN",rownames(sigs)),2]="OsCN"
   sigs[grep("change",rownames(sigs)),2]="Changepoint"
   sigs[grep("copy",rownames(sigs)),2]="CopyNumber"
   sigs[grep("chrarm",rownames(sigs)),2]="Bpchrarm"
   sigs[grep("ploidy",rownames(sigs)),2]="Ploidy"
   sigs[,2] = as.factor(sigs[,2])
   sigs[,3] = factor(rownames(sigs),levels=rownames(sigs))

   colnames(sigs) = c("value","Distribution","Feature")

   #Plots the signature as a column plot
   ggplot(sigs,aes_string(x="Feature",y='value',fill="Distribution",group="Distribution"))+
      geom_col(position="dodge")+
      scale_x_discrete(labels=levels(sigs$Feature))+
      theme(legend.position="none",
          axis.text.x=element_text(size=5,angle=45,hjust=1),
          axis.title=element_text(size=5), 
          panel.grid.major = element_blank(),
          panel.grid.minor = element_blank(), 
          panel.background = element_blank(),
          axis.line = element_line(colour = "black"))+
      ylab("")+
      ggtitle(paste0("Signature ",sigNum))+
      xlab("Component number")
}
