% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Intbc.R
\name{CST_Intbc}
\alias{CST_Intbc}
\title{Downscaling using interpolation and bias adjustment.}
\usage{
CST_Intbc(
  exp,
  obs,
  exp_cor = NULL,
  target_grid,
  bc_method,
  int_method = NULL,
  points = NULL,
  method_point_interp = NULL,
  lat_dim = "lat",
  lon_dim = "lon",
  sdate_dim = "sdate",
  member_dim = "member",
  time_dim = "time",
  region = NULL,
  ncores = NULL,
  loocv = TRUE,
  ...
)
}
\arguments{
\item{exp}{an 's2dv object' containing the experimental field on the
coarse scale for which the downscaling is aimed. The object must have, at least,
the dimensions latitude, longitude, start date and member. The object is expected to be
already subset for the desired region. Data can be in one or two integrated regions, e.g.,
crossing the Greenwich meridian. To get the correct results in the latter case,
the borders of the region should be specified in the parameter 'region'. See parameter
'region'.}

\item{obs}{an 's2dv object' containing the observational field. The object
must have, at least, the dimensions latitude, longitude and start date. The object is
expected to be already subset for the desired region.}

\item{exp_cor}{an optional 's2dv_cube' object with named dimensions containing the seasonal
forecast experiment data. If the forecast is provided, it will be downscaled using the
hindcast and observations; if not provided, the hindcast will be downscaled instead. The
default value is NULL.}

\item{target_grid}{a character vector indicating the target grid to be passed to CDO.
It must be a grid recognised by CDO or a NetCDF file.}

\item{bc_method}{a character vector indicating the bias adjustment method to be applied after
the interpolation. Accepted methods are 'quantile_mapping', 'bias', 'evmos',
'mse_min', 'crps_min', 'rpc-based'. The abbreviations 'qm' can also be used.}

\item{int_method}{a character vector indicating the regridding method to be passed to CDORemap.
Accepted methods are "con", "bil", "bic", "nn", "con2". If "nn" method is to be used, CDO_1.9.8
or newer version is required. For method "con2", CDO_2.2.2 or older version is required.}

\item{points}{a list of two elements containing the point latitudes and longitudes
of the locations to downscale the model data. The list must contain the two elements
named as indicated in the parameters 'lat_dim' and 'lon_dim'. If the downscaling is
to a point location, only regular grids are allowed for exp and obs. Only needed if the
downscaling is to a point location.}

\item{method_point_interp}{a character vector indicating the interpolation method to interpolate
model gridded data into the point locations. Accepted methods are "nearest", "bilinear", "9point",
"invdist4nn", "NE", "NW", "SE", "SW". Only needed if the downscaling is to a point location.}

\item{lat_dim}{a character vector indicating the latitude dimension name in the element 'data'
in exp and obs. Default set to "lat".}

\item{lon_dim}{a character vector indicating the longitude dimension name in the element 'data'
in exp and obs. Default set to "lon".}

\item{sdate_dim}{a character vector indicating the start date dimension name in the element
'data' in exp and obs. Default set to "sdate".}

\item{member_dim}{a character vector indicating the member dimension name in the element
'data' in exp and obs. Default set to "member".}

\item{time_dim}{a character vector indicating the time dimension name in the element
'data' in exp and obs. Default set to "time".}

\item{region}{a numeric vector indicating the borders of the downscaling region.
It consists of four elements in this order: lonmin, lonmax, latmin, latmax. lonmin refers
to the left border, while lonmax refers to the right border. latmin indicates the lower
border, whereas latmax indicates the upper border. If set to NULL (default), the function
takes the first and last elements of the latitudes and longitudes in obs.}

\item{ncores}{an integer indicating the number of cores to use in parallel computation.
The default value is NULL.}

\item{loocv}{a logical indicating whether to apply leave-one-out cross-validation when
applying the bias correction. In this procedure, all values from the corresponding
year are excluded, so that when building the correction function for a given year, no data
from that year are used. Default to TRUE.}

\item{...}{additional arguments passed to internal methods}
}
\value{
An 's2dv' object. The element 'data' contains the dowscaled field, 'lat' the
downscaled latitudes, and 'lon' the downscaled longitudes.
}
\description{
This function performs a downscaling using an interpolation and a later bias
adjustment. It is recommended that the observations are passed already in the target grid.
Otherwise, the function will also perform an interpolation of the observed field into the
target grid. The coarse scale and observation data can be either global or regional. In the
latter case, the region is defined by the user. In principle, the coarse and observation data
are intended to be of the same variable, although different variables can also be admitted.
}
\examples{
exp <- rnorm(500)
dim(exp) <- c(member = 5, lat = 4, lon = 5, sdate = 5)
exp_lons <- 1:5
exp_lats <- 1:4
obs <- rnorm(900)
dim(obs) <- c(lat = 12, lon = 15, sdate = 5)
obs_lons <- seq(1,5, 4/14)
obs_lats <- seq(1,4, 3/11)
exp <- CSTools::s2dv_cube(data = exp, coords = list(lat = exp_lats, lon = exp_lons))
obs <- CSTools::s2dv_cube(data = obs, coords = list(lat = obs_lats, lon = obs_lons))
if (Sys.which("cdo") != "") {
res <- CST_Intbc(exp = exp, obs = obs, target_grid = 'r1280x640', 
                bc_method = 'bias', int_method = 'conservative')
}
}
\author{
J. Ramon, \email{jaumeramong@gmail.com}

E. Duzenli, \email{eren.duzenli@bsc.es}
}
