## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  warning = FALSE, 
  message = FALSE,
  comment = "#>"
)

## ----load packages, message=FALSE, warning=FALSE------------------------------
# Load packages
library(ConSciR)
library(dplyr)
library(ggplot2)

## ----table of calculated values, message=FALSE, warning=FALSE-----------------
filepath <- data_file_path("mydata.xlsx")
mydata <- readxl::read_excel(filepath, sheet = "mydata")
mydata <- mydata |> filter(Sensor == "Room 1")


# Add calculated values using mutate
head(mydata) |> 
  mutate(
    
    # Humidity functions
    Absolute_Humidity = calcAH(Temp, RH), 
    Dew_Point = calcDP(Temp, RH), 
    Mixing_Ratio = calcMR(Temp, RH), 
    Humidity_Ratio = calcHR(Temp, RH),
    Enthalpy = calcEnthalpy(Temp, RH), 
    Saturation_Vapour_Pressure = calcPws(Temp), 
    Actual_Vapour_Pressure = calcPw(Temp, RH), 
    Air_Density = calcAD(Temp, RH),
    
    # Conservation risks
    Mould_Growth_Rate_mm_day = calcMould_Zeng(Temp, RH, label = TRUE),
    Mould_Growth_Limit = calcMould_Zeng(Temp, RH), 
    Mould_Growth_Risk = ifelse(RH > Mould_Growth_Limit, "Mould risk", "No risk"),
    Mould_Growth_Index = calcMould_VTT(Temp, RH), 
    Lifetime = calcLM(Temp, RH), 
    Preservation_Index = calcPI(Temp, RH), 
    EMC_Wood = calcEMC_wood(Temp, RH),
    
    # Sustainability calculations 
    Temp_from_DP = calcTemp(RH, Dew_Point),
    RH_from_DP = calcRH_DP(Temp, Dew_Point),
    RH_from_AH = calcRH_AH(Temp, Absolute_Humidity),
    RH_if_2C_higher = calcRH_AH(Temp + 2, Absolute_Humidity)
    
  ) |>
  glimpse()


head(mydata) |> 
  tidy_TRHdata() |> # tidy
  add_time_vars() |> # add time factors 
  add_humidity_calcs() |> # add humidity values 
  add_conservation_calcs() |> # add conservation risks
  add_humidity_adjustments() # add environmental zones and RH adjustments


## ----calculations to visualise the data, message=FALSE, warning=FALSE, fig.alt="visualisations"----

mydata |>
  # Calculate Absolute Humidity and Dew Point
  mutate(
    AbsHum = calcAH(Temp, RH),
    DewPoint = calcDP(Temp, RH)
  ) |>
  # Create base plot using graph_TRH function
  graph_TRH() +
  # Add Absolute Humidity line
  geom_line(aes(Date, AbsHum), color = "cyan4", alpha = 0.7) +
  # Add Dew Point line
  geom_line(aes(Date, DewPoint), color = "mediumvioletred", alpha = 0.7) +
  # Apply a theme
  theme_bw()


## ----mould_risk, message=FALSE, warning=FALSE, fig.alt="mould"----------------
mydata |>
  mutate(Mould = calcMould_VTT(Temp, RH)) |>
  ggplot() +
  geom_area(aes(Date, Mould), fill = "lightseagreen") +
  labs(title = "Mould Growth Index", y = "Mould Index", x = NULL) + 
  theme_bw()

## ----psychrometric custom, message=FALSE, warning=FALSE, fig.alt="psychrometric_chart_custom"----

head(mydata, 100) |>
  graph_psychrometric(
    LowT = 12, 
    HighT = 28,
    LowRH = 40, 
    HighRH = 70,
    data_alpha = 0.3,
    y_func = calcAH
  ) + 
  theme_classic()

