% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/block_bootstrap.R
\name{block_bootstrap}
\alias{block_bootstrap}
\title{Block Bootstrap}
\usage{
block_bootstrap(
  X,
  maxLag,
  x = 1:length(X),
  n_bootstrap = 100,
  l = ceiling(length(X)^(1/3)),
  estimator = standard_est,
  type = "autocovariance",
  alpha = 0.05,
  boot_type = "moving",
  plot = FALSE,
  boot_mat = FALSE,
  ylim = c(-1, 1),
  ...
)
}
\arguments{
\item{X}{A vector representing observed values of the time series.}

\item{maxLag}{The maximum lag to compute the estimated autocovariance function at.}

\item{x}{A vector of indices. Defaults to the sequence \code{1:length(X)}.}

\item{n_bootstrap}{The number of times to run moving block bootstrap. Defaults to 100.}

\item{l}{The block length considered for bootstrap. Defaults to \eqn{\lceil N \rceil^{1/3}}, where \eqn{N} is the length of the observation window.}

\item{estimator}{The function name of the estimator to use. Defaults to \code{standard\_est}.}

\item{type}{Compute either the 'autocovariance' or 'autocorrelation'. Defaults to 'autocovariance'.}

\item{alpha}{The quantile used to compute the \eqn{(1 - \alpha)\%} confidence interval. Defaults to \eqn{0.05.}}

\item{boot_type}{What type of block bootstrap should be used, either 'moving' for moving block bootstrap or 'circular' for circular block bootstrap.}

\item{plot}{A boolean determining whether a plot should be created. By default, no plot is created.}

\item{boot_mat}{A boolean determining whether a bootstrap matrix should be returned or not. By default, no matrix is returned.}

\item{ylim}{A vector of length two denoting the limits of the y-axis for the plot. Defaults to \code{c(-1, 1)}.}

\item{...}{Optional named arguments to the chosen estimator. See the examples.}
}
\value{
A list containing three items. The first
A list consisting of three items. The first is the average estimated autocovariance/autocorrelation function for the bootstrap samples, the second is a matrix of the estimated autocovariance/autocorrelation functions from the bootstrapped samples, and the third is a matrix of confidence intervals for each lag. If the option \code{boot_mat = TRUE}, an addition value is returned, a matrix where each row is a bootstrap estimated autocovariance function. If the option \code{plot = TRUE} is used, the plot shows the esitmated autocovariance function in black, the average bootstrap estimated autocovariance function in red and the \eqn{(1 - \alpha)\%} confidence region is the grey shaded area.
}
\description{
This function performs block bootstrap (moving or circular) to obtain a \eqn{(1-\alpha)\%} confidence-interval for the autocovariance function. It will also compute average autocovariance function across all bootstrapped estimates.
}
\details{
This function performs block bootstrap to obtain a \eqn{(1-\alpha)\%} confidence-interval for the autocovariance function. It will also compute average autocovariance function across all bootstrapped estimates.

Moving block bootstrap can be described as follows.
For some times series \eqn{X(1), X(2), \dots, X(n),} construct \eqn{k \in N} overlapping blocks of the form
\eqn{B_{i} = (X(i), \dots, X(i + \ell - 1)),} where \eqn{\ell \in \{1, \dots , n\}} is the block length.
Randomly sample, with replacement, from the discrete uniform distribution with on \eqn{\{1, \dots, n - \ell + 1\}} to obtain a set of random starting locations \eqn{I_{1}, \dots, I_{k}.}
Construct a bootstrapped time series \eqn{B_{1}^{\ast}, B_{2}^{\ast}, \dots, B_{k}^{\ast},} where \eqn{B_{i}^{\ast} = B_{I_{i}}.}
The bootstrapped time series is truncated to have length \eqn{n,} and will be of the form \eqn{X^{\ast}(1), \dots , X^{\ast}(n).}
The autocovariance function is then computed for the bootstrapped time series.

An alternative to moving block bootstrap is circular block bootstrap.
Circular block bootstrap uses the time series like a circle, that is, the observation at \eqn{n + i} is the same as the observation at location \eqn{i.}
For example, for the block \eqn{B_{n - \ell + 2}}, we obtain \eqn{(X(n - \ell + 2) , \dots , X(n), X(n + 1))} is the same as \eqn{(X(n - \ell + 2) , \dots , X(n), X(1)).}
When performing random sampling to obtain starting locations, the set \eqn{\{1, \dots, n\}} is now considered.
The procedure for constructing the bootstrap time series after constructing the blocks and selecting the starting indices is the same as moving block bootstrap.

This process is repeated \code{n_bootstrap} times to obtain \code{n_boostrap} estimates of the autocovariance function using the bootstrapped time series, for which the average autocovariance function
and the \eqn{(1 - \alpha)\%} confidence intervals are constructed pointwise for each lag.

The choice of the block length, \eqn{\ell ,} depends on the degree of dependence present in the time series. If the time series has a high degree of dependence, a larger block size should be chosen to ensure the dependency structure is maintained within the block.

Any estimator of the autocovariance function can be used in this function, including a custom estimator not in the package, see the examples.
}
\examples{
X <- c(1, 2, 3, 3, 2, 1)
block_bootstrap(X, 4, n_bootstrap = 3, l = 2, type = 'autocorrelation')
block_bootstrap(X, 4, n_bootstrap = 3, l = 2, plot =TRUE, type = 'autocovariance')
block_bootstrap(X, 4, n_bootstrap = 3, l = 2, estimator=tapered_est,
    rho = 0.5, window_name = 'blackman', window_params = c(0.16),
    type='autocorrelation')
my_cov_est <- function(X, maxLag) {
  n <- length(X)
  covVals <- c()
  for(h in 0:maxLag) {
    covVals_t <- (X[1:(n-h)] - mean(X)) * (X[(1+h):n] - mean(X))
    covVals <- c(covVals, sum(covVals_t) / (n - h))
  }
  return(covVals)
}
block_bootstrap(X, 4, n_bootstrap = 3, l = 2, estimator=my_cov_est)

plot(LakeHuron, main="Lake Huron Levels", ylab="Feet")
X <- as.vector(LakeHuron)
block_bootstrap(X, 20, n_bootstrap = 100, l = 40, type = 'autocorrelation')
block_bootstrap(X, 20, n_bootstrap = 100, l = 40, plot = TRUE, type = 'autocorrelation')
block_bootstrap(X, 20, n_bootstrap = 100, l = 40, estimator=tapered_est,
    rho = 0.5, window_name = 'blackman', window_params = c(0.16),
    type='autocorrelation', plot =TRUE)

my_cov_est <- function(X, maxLag) {
  n <- length(X)
  covVals <- c()
  for(h in 0:maxLag) {
    covVals_t <- (X[1:(n-h)] - mean(X)) * (X[(1+h):n] - mean(X))
    covVals <- c(covVals, sum(covVals_t) / (n - h))
  }
  return(covVals)
}
block_bootstrap(X, 20, n_bootstrap = 100, l = 40, estimator = my_cov_est,
    plot = TRUE, type = 'autocorrelation')
}
\references{
Chapters 2.5 and 2.7 in Lahiri, S. N. (2003). Resampling Methods for Dependent Data. Springer. https://doi.org/10.1007/978-1-4757-3803-2

Künsch, H. R. (1989). The Jackknife and the Bootstrap for General Stationary Observations. The Annals of Statistics 17(3), 1217-1241. https://doi.org/10.1214/aos/1176347265

Politis, D. N. & Romano, J. P. (1991). A Circular Block-Resampling Procedure for Stationary Data. In R. LePage & L. Billard, eds, Exploring the Limits of Bootstrap, Wiley, 263-270.
}
