# if (!requireNamespace("shiny", quietly = TRUE)) install.packages("shiny")
# if (!requireNamespace("MASS", quietly = TRUE)) install.packages("MASS")
# if (!requireNamespace("Matrix", quietly = TRUE)) install.packages("Matrix")
# if (!requireNamespace("htmltools", quietly = TRUE)) install.packages("htmltools")

library(shiny)
library(MASS)
library(Matrix)
library(htmltools)

# ----------------- Utility Functions -----------------
autogenerate_design = function(n_factors, factor_levels, block_size, replications)
{
  n <- n_factors
  # levels <- suppressWarnings(as.numeric(unlist(strsplit(factor_levels,","))))
  levels = factor_levels
  if (length(levels) != n || any(is.na(levels))) {
    stop("Factor levels must be numeric and match n_factors")
  }
  k <- block_size; r <- replications
  total_treatments <- prod(levels)
  if (total_treatments %% k != 0) {
    stop("Block size k must divide total number of treatments")
  }
  base_blocks <- total_treatments / k
  grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s-1)))
  grids <- grids[, rev(seq_len(ncol(grids)))]
  treatments <- apply(grids, 1, paste0, collapse = "")
  ncol_mat <- base_blocks * r
  mat <- matrix("", nrow = k, ncol = ncol_mat)
  for (rep_i in 0:(r-1)) {
    for (b in 1:base_blocks) {
      col_idx <- rep_i*base_blocks + b
      start <- (b-1)*k + 1
      mat[, col_idx] <- treatments[start:(start + k - 1)]
    }
  }
  header <- paste0("B", 1:ncol_mat)
  table_text <- paste(c(paste(header, collapse = "\t"), apply(mat, 1, paste, collapse = "\t")), collapse = "\n")
  # updateTextAreaInput(session, "blocks", value = table_text)
  # return(table_text)
  colnames(mat) = paste0("B", 1:ncol_mat)
  rownames(mat) = rep("",nrow(mat))
  mat = noquote(mat)
  return(mat)
}

analyze_design <- function(factor_levels, blocks) {
  levels <- factor_levels
  mpinv <- function(A, tol = 1e-8) {
    sv <- svd(A)
    d <- ifelse(sv$d > tol, 1/sv$d, 0)
    sv$v %*% diag(d, nrow = length(d)) %*% t(sv$u)
  }
  
  is_zero_vec <- function(x, tol = 1e-10) all(abs(x) < tol)
  
  # Orthonormal contrast for given number of levels si
  orthonormal_contrast_full <- function(si) {
    if (si < 2) stop("si must be >= 2")
    P <- matrix(0, nrow = si - 1, ncol = si)
    for (r in 1:(si - 1)) {
      P[r, 1:r] <- 1 / sqrt(r * (r + 1))
      P[r, r + 1] <- -r / sqrt(r * (r + 1))
    }
    P
  }
  
  # Generate all tuples (y) of m factors, excluding all-zero
  generate_y_tuples <- function(m) {
    tuples <- as.matrix(expand.grid(replicate(m, 0:1, simplify = FALSE)))
    tuples <- tuples[rowSums(tuples) != 0, , drop = FALSE]
    tuples
  }
  
  # Generate labels like F1, F2, F1F2, F1F2F3, etc.
  generate_labels_from_tuples <- function(tuples) {
    apply(tuples, 1, function(row) {
      idx <- which(row == 1)
      paste0(paste(paste0("F", idx), collapse = ""))
    })
  }
  
  #  Compute p^(y)
  compute_p_y <- function(y, s_levels) {
    m <- length(y)
    p_list <- list()
    for (i in 1:m) {
      if (y[i] == 1) {
        p_list[[i]] <- orthonormal_contrast_full(s_levels[i])
      } else {
        p_list[[i]] <- matrix(rep(1 / sqrt(s_levels[i]), s_levels[i]), nrow = 1)
      }
    }
    Reduce(kronecker, p_list)
  }
  
  # ----------------- Design Components -----------------
  # returns N, treatments labels, and grid (matrix of level values 0:(s-1), columns = factors)
  build_incidence <- function(levels, blocks) {
    grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s - 1)))
    grids <- grids[, rev(seq_len(ncol(grids)))]
    trt_labels <- apply(grids, 1, paste0, collapse = "")
    v <- nrow(grids); b <- length(blocks)
    
    N <- matrix(0, nrow = v, ncol = b)
    rownames(N) <- trt_labels
    colnames(N) <- paste0("B", seq_len(b))
    
    for (h in seq_len(b)) {
      for (trt in blocks[[h]]) {
        pos <- match(trt, trt_labels)
        if (is.na(pos)) stop(sprintf("Block %d contains treatment '%s' not in treatment list", h, trt))
        N[pos, h] <- N[pos, h] + 1
      }
    }
    list(N = N, treatments = trt_labels, grids = as.matrix(grids))
  }
  
  compute_Cmatrix <- function(N) {
    N <- as.matrix(N)
    r <- as.vector(rowSums(N))
    k <- as.vector(colSums(N))
    
    # handle single block or single treatment edge cases
    if (length(k) == 1) {
      NkN <- (N %*% t(N)) / k
    } else {
      NkN <- N %*% diag(1 / k, nrow = length(k)) %*% t(N)
    }
    
    C <- diag(r, nrow = length(r)) - NkN
    list(C = C, r = r, k = k)
  }
  
  
  # check proper: equal block sizes
  check_proper <- function(N, tol = 1e-8) {
    k <- colSums(N)
    proper <- (max(k) - min(k)) < tol
    list(proper = proper, block_sizes = k)
  }
  # check equireplicate: equal replications r
  check_equireplicate <- function(N, tol = 1e-8) {
    r <- rowSums(N)
    equirep <- (max(r) - min(r)) < tol
    list(equireplicate = equirep, replicates = r)
  }
  
  # ==========================================================
  # Function: Check balance and OFS of a factorial block design (Robust Version)
  # (modified to return structured output so it can be called programmatically)
  # ==========================================================
  check_design_balance_OFS <- function(levels, blocks, tol = 1e-8, verbose = TRUE) {
    decode_treatment <- function(trt) {
      # splits each character; assumes each level is coded as a single digit 0..(s-1)
      as.numeric(strsplit(trt, "")[[1]])
    }
    
    # Function to generate coded levels
    code_levels <- function(s) {
      if (s %% 2 == 1) {
        seq(-(s - 1) / 2, (s - 1) / 2, by = 1)   
      } else {
        seq(-(s - 1), (s - 1), by = 2)           
      }
    }
    
    # Extract treatments and parameters
    all_treatments <- unique(unlist(blocks))
    if (length(all_treatments) == 0) stop("No treatments found in blocks")
    m <- nchar(all_treatments[1])
    
    # Build design data frame
    design_data <- data.frame()
    for (b in seq_along(blocks)) {
      block <- blocks[[b]]
      for (t in block) {
        factors <- decode_treatment(t)
        if (length(factors) != m) stop("Inconsistent treatment label length")
        row <- data.frame(Block = paste0("B", b), t(factors))
        design_data <- rbind(design_data, row)
      }
    }
    colnames(design_data)[-1] <- paste0("F", 1:m)
    
    # if (verbose) {
    #   cat("\n================= DESIGN MATRIX =================\n")
    #   print(design_data)
    # }
    
    # ---------- (1) BALANCE CHECK ----------
    if (verbose) cat("\n---------------- BALANCE CHECK ----------------\n")
    balance_ok <- TRUE
    for (f in paste0("F", 1:m)) {
      tbl <- table(design_data[[f]])
      if (verbose) {
        cat("\nFactor:", f, "\n")
        print(tbl)
      }
      if (length(unique(as.vector(tbl))) == 1) {
        if (verbose) cat(" Balanced for", f, "\n")
      } else {
        if (verbose) cat(" NOT balanced for", f, "\n")
        balance_ok <- FALSE
      }
    }
    
    # ---------- (2) OFS CHECK ----------
    # if (verbose) cat("\n---------------- OFS CHECK ----------------\n")
    
    # Apply coding
    coded_data <- design_data
    for (i in 1:m) {
      s <- levels[i]
      code_map <- code_levels(s)
      map_df <- data.frame(orig = 0:(s - 1), coded = code_map)
      coded_data[[paste0("F", i)]] <- sapply(design_data[[paste0("F", i)]],
                                             function(x) map_df$coded[map_df$orig == x])
    }
    
    # if (verbose) {
    #   cat("\nCoded Design Matrix (for OFS check):\n")
    #   print(coded_data)
    # }
    
    # Build factorial model formula
    form_str <- paste("~ (", paste(colnames(coded_data)[-1], collapse = " + "), ")^", m, sep = "")
    form <- as.formula(form_str)
    
    # Model matrix for factorial effects (no intercept)
    X <- tryCatch({
      mm <- model.matrix(form, data = coded_data[,-1])
      # remove intercept column if present
      if ("(Intercept)" %in% colnames(mm)) mm <- mm[, setdiff(colnames(mm), "(Intercept)"), drop = FALSE]
      mm
    }, error = function(e) {
      if (verbose) cat("Model matrix could not be fully generated due to collinearity.\n")
      return(NULL)
    })
    
    if (is.null(X)) {
      if (verbose) cat("Cannot proceed with OFS check - insufficient unique treatments.\n")
      return(invisible(list(ok = FALSE, reason = "insufficient_unique_treatments")))
    }
    
    effect_names <- colnames(X)
    
    # Block matrix (built explicitly to avoid contrasts() on single-level factors)
    blk_levels <- unique(as.character(coded_data$Block))
    blk_factor <- factor(as.character(coded_data$Block), levels = blk_levels)
    B <- matrix(0, nrow = nrow(coded_data), ncol = length(blk_levels))
    for (i in seq_len(nrow(coded_data))) {
      B[i, as.integer(blk_factor[i])] <- 1
    }
    colnames(B) <- paste0("Block", seq_len(ncol(B)))
    rownames(B) <- rownames(coded_data)
    
    
    # Projection to remove block effects
    Q <- diag(nrow(X)) - B %*% solve(t(B) %*% B) %*% t(B)
    X_adj <- Q %*% X
    
    # Cross-product matrix (add small tolerance to avoid singularity)
    Cmat <- t(X_adj) %*% X_adj + diag(1e-12, ncol(X_adj))
    
    # Safe correlation conversion
    safe_cov2cor <- function(M) {
      d <- sqrt(diag(M))
      d[d == 0 | is.na(d)] <- 1
      corM <- M / (d %o% d)
      corM[is.na(corM)] <- 0
      return(corM)
    }
    
    corC <- safe_cov2cor(Cmat)
    
    # if (verbose) {
    #   cat("\nCorrelation matrix among factorial effects:\n")
    #   print(round(corC, 3))
    # }
    # 
    # ---------- (3) EFFECT-WISE ORTHOGONALITY REPORT ----------
    #if (verbose) cat("\n---------------- EFFECT GROUP ANALYSIS ----------------\n")
    effect_order <- sapply(strsplit(effect_names, ":"), length)
    
    effect_group_report <- list()
    for (ord in sort(unique(effect_order))) {
      idx <- which(effect_order == ord)
      subC <- corC[idx, idx, drop = FALSE]
      
      if (length(idx) <= 1) {
        if (verbose) cat(paste0("\n", ord, "-FACTOR INTERACTIONS: Only one effect, skipping correlation check.\n"))
        effect_group_report[[as.character(ord)]] <- list(mean_offdiag = 0, orthogonal = TRUE)
        next
      }
      
      offdiag_vals <- abs(subC[upper.tri(subC)])
      offdiag_mean <- ifelse(length(offdiag_vals) == 0, 0, mean(offdiag_vals, na.rm = TRUE))
      
      label <- switch(as.character(ord),
                      "1" = "MAIN EFFECTS",
                      "2" = "TWO-FACTOR INTERACTIONS",
                      paste0(ord, "-FACTOR INTERACTIONS"))
      
      if (verbose) {
        cat("\n", label, ":\n", sep = "")
        cat("Average off-diagonal correlation =", round(offdiag_mean, 3), "\n")
        if (!is.na(offdiag_mean) && offdiag_mean < tol) {
          cat("Orthogonal among themselves\n")
        } else {
          cat("Some correlation exists among same-order effects\n")
        }
      }
      effect_group_report[[as.character(ord)]] <- list(mean_offdiag = offdiag_mean, orthogonal = (!is.na(offdiag_mean) && offdiag_mean < tol))
    }
    
    # ---------- (4) OVERALL OFS DECISION ----------
    offdiag_all <- mean(abs(corC[upper.tri(corC)]), na.rm = TRUE)
    if (is.na(offdiag_all)) offdiag_all <- 0
    
    if (verbose) {
      cat("\n=========================================================\n")
      if (offdiag_all < tol) {
        cat("Overall Design has Orthogonal Factorial Structure (OFS)\n")
      } else {
        cat("Design deviates from perfect OFS (mean off-diagonal =", round(offdiag_all, 3), ")\n")
      }
      cat("=========================================================\n")
    }
    
    # Return structured info for programmatic use:
    invisible(list(ok = (offdiag_all < tol),
                   offdiag_all = offdiag_all,
                   corC = corC,
                   balance_ok = balance_ok,
                   effect_report = effect_group_report,
                   design_data = design_data))
  }
  
  # ----------------- general methodPROCEDURE (when OFS fails but design proper & equirep) -----------------
  # Steps as you described: divisors D(Fi), C(Fi) sequences f_ij, build L(Fi), build interactions L, compute efficiencies.
  
  # Step: compute divisors D(Fi)
  compute_divisors <- function(levels) {
    v <- prod(levels)
    sprod <- v
    sapply(seq_along(levels), function(i) {
      si <- levels[i]
      (sprod / si) * choose(si, 2)
    })
  }
  
  # Step: compute f_ij sequences for each factor i: j = 0..si-1 => f_{i,j+1} = (si - (2*j+1)) / D(Fi)
  compute_f_sequences <- function(levels) {
    D <- compute_divisors(levels)
    seqs <- vector("list", length(levels))
    for (i in seq_along(levels)) {
      si <- levels[i]
      j <- 0:(si - 1)
      seqs[[i]] <- (si - (2 * j + 1)) / D[i]   # vector length si, indices j->1..si
    }
    seqs
  }
  
  # Step: Build L for each main factor
  # grids: matrix with columns = factors, entries 0:(s_i-1)
  build_L_mains <- function(levels, grids, f_seqs) {
    v <- nrow(grids)
    m <- length(levels)
    L_list <- vector("list", m)
    for (i in 1:m) {
      si <- levels[i]
      
      # --- SAFETY CHECK ---
      if (si < 2) {
        warning(sprintf("Factor %d has only one level; skipping contrast generation.", i))
        L_list[[i]] <- rep(0, v)
        next
      }
      # --------------------
      
      # f_seqs[[i]] corresponds to j=0..si-1 -> positions 1..si
      # For each treatment (row in grids), get level value (0..si-1) and map to f_seq element
      vals <- sapply(seq_len(v), function(t) {
        lvl <- grids[t, i]   # value 0..si-1
        f_seqs[[i]][lvl + 1]
      })
      L_list[[i]] <- vals
    }
    L_list
  }
  
  build_L_interaction <- function(L_list, idxs) {
    # If any of the involved factors has all zeros (i.e., constant), skip
    if (any(sapply(idxs, function(ii) all(abs(L_list[[ii]]) < 1e-12)))) {
      return(rep(0, length(L_list[[1]])))
    }
    
    v <- length(L_list[[1]])
    L <- numeric(v)
    for (t in 1:v) {
      vals <- sapply(idxs, function(ii) L_list[[ii]][t])
      if (any(abs(vals) < 1e-12)) {
        L[t] <- 0
      } else {
        sgn <- prod(sign(vals))
        mag <- min(abs(vals))
        L[t] <- sgn * mag
      }
    }
    L
  }
  
  # Step: compute efficiency scalar from L vector: (L C L') / (r * (L L'))
  compute_efficiency_from_L <- function(L, C, r_scalar) {
    num <- as.numeric(L %*% C %*% L)
    den <- as.numeric(r_scalar * (L %*% L))
    if (abs(den) < 1e-12) return(NA_real_)
    num / den
  }
  
  # ----------------- Main Analyzer (integrates both OFS route and fallback) -----------------
  # Unified treatment label generator (lexicographic order)
  make_treatments <- function(levels) {
    grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s - 1)))
    grids <- grids[, rev(seq_len(ncol(grids))), drop = FALSE]
    apply(grids, 1, paste0, collapse = "")
  }
  
  analyze_design_efficiency <- function(levels, blocks, verbose = TRUE, tol = 1e-6) {
    built <- build_incidence(levels, blocks)
    N <- built$N
    grids <- built$grids   # matrix
    v <- nrow(grids); m <- length(levels)
    comp <- compute_Cmatrix(N)
    C <- comp$C
    r_vec <- comp$r
    k_vec <- comp$k
    
    if (verbose) {
      # cat("Lexicographic Treatment Order (first 50 shown if large):\n")
      # print(head(rownames(N), 50))
      # cat("\nIncidence Matrix (N) [showing dims]: ", dim(N), "\n")
      # cat("\nNN' (showing top-left 8x8 if large):\n")
      # if (v <= 8) print(N %*% t(N)) else print((N %*% t(N))[1:8, 1:8])
      # cat("\nC Matrix (showing top-left 8x8 if large):\n")
      # if (v <= 8) print(round(C, 6)) else print(round(C[1:8, 1:8], 6))
      cat("\nBlock sizes (k):", paste(round(k_vec, 6), collapse = ", "), "\n")
      cat("Replications (r) [per treatment]:", paste(round(r_vec, 6), collapse = ", "), "\n\n")
    }
    
    # ----------------- REPLACED: Use robust OFS/balance checker -----------------
    v_ofs <- check_design_balance_OFS(levels, blocks, tol = tol, verbose = verbose)
    if (!is.null(v_ofs) && isTRUE(v_ofs$ok)) {
      cat("Design verified: OFS OK - using standard p^(y) method\n\n")
      # compute standard p^(y) efficiencies
      y_tuples <- generate_y_tuples(m)
      labels <- generate_labels_from_tuples(y_tuples)
      results <- data.frame(label = character(0), rho = numeric(0), efficiency = numeric(0), stringsAsFactors = FALSE)
      for (i in seq_len(nrow(y_tuples))) {
        y <- y_tuples[i, ]
        P_y <- compute_p_y(y, levels)   # rows x v
        trace_val <- sum(diag(P_y %*% C %*% t(P_y)))
        rank_val <- nrow(P_y)
        rho_y <- trace_val / rank_val
        eff_y <- rho_y / r_vec[1]
        results <- rbind(results, data.frame(label = labels[i], rho = round(rho_y,6), efficiency = round(eff_y,6), stringsAsFactors = FALSE))
        cat(sprintf("Efficiency(%s) = %g\n", labels[i], eff_y))
      }
      # if (verbose) {
      #   cat("\nSummary table:\n"); print(results, row.names = FALSE)
      # }
      return(invisible(list(mode = "OFS", results = results, N = N, C = C)))
    }
    
    # If we reach here, OFS failed. Apply general method checks.
    if (!is.null(v_ofs)) {
      cat("Design does NOT satisfy OFS (mean off-diagonal =",
          ifelse(is.null(v_ofs$offdiag_all), "NA", format(v_ofs$offdiag_all, digits = 6)),
          "). Attempting general method if design is proper & equireplicate...\n\n")
    } else {
      cat("OFS check returned NULL - Attempting general method if design is proper & equireplicate...\n\n")
    }
    
    pr <- check_proper(N)
    er <- check_equireplicate(N)
    if (!pr$proper || !er$equireplicate) {
      cat("Design is not proper and/or not equireplicate:\n")
      cat("  proper:", pr$proper, "  equireplicate:", er$equireplicate, "\n")
      cat("Efficiency cannot be computed via general method\n")
      return(invisible(list(mode = "NO_OFS", reason = "not_proper_or_not_equirep", N = N, C = C)))
    }
    
    # Design is proper and equireplicate ==> proceed with general method algorithm
    cat("Design is proper and equireplicate. Proceeding with general method of computation.\n\n")
    r_scalar <- er$replicates[1]
    v_total <- prod(levels)
    
    # Step 1: divisors
    D <- compute_divisors(levels)   # vector length m
    names(D) <- paste0("F", seq_len(m))
    # if (verbose) {
    #   cat("Divisors D(Fi):\n"); print(D); cat("\n")
    # }
    
    # Step 2: f sequences
    f_seqs <- compute_f_sequences(levels)
    # if (verbose) {
    #   cat("Sample f sequences (first factor shown):\n"); print(f_seqs[[1]]); cat("\n")
    # }
    
    # Step 3: Build L for mains
    L_mains <- build_L_mains(levels, grids, f_seqs)  # list length m, each length v
    names(L_mains) <- paste0("F", seq_len(m))
    
    # Step 4: Build interactions Ls (all non-empty subsets)
    tuples <- generate_y_tuples(m)
    labels <- generate_labels_from_tuples(tuples)
    results <- data.frame(label = character(0), rho = numeric(0), efficiency = numeric(0), stringsAsFactors = FALSE)
    
    for (ti in seq_len(nrow(tuples))) {
      y <- tuples[ti, ]
      lbl <- labels[ti]
      idxs <- which(y == 1)  # factor indices involved
      # build L vector
      if (length(idxs) == 1) {
        L_vec <- L_mains[[ idxs ]]
      } else {
        L_vec <- build_L_interaction(L_mains, idxs)
      }
      # Step 5: efficiency formula
      eff_val <- compute_efficiency_from_L(L_vec, C, r_scalar)
      # compute rho (scalar) as L C L' (not divided)
      rho_val <- as.numeric(L_vec %*% C %*% L_vec)
      results <- rbind(results, data.frame(label = lbl, rho = round(rho_val,6), efficiency = round(eff_val,6), stringsAsFactors = FALSE))
      cat(sprintf("Efficiency(%s) = %g\n", lbl, eff_val))
    }
    
    
    invisible(list( results = results, N = N, C = C, L_mains = L_mains))
  }
  res <- analyze_design_efficiency(levels, blocks, verbose = TRUE)
}

run_app <- function() {
  htmlEscape <- htmltools::htmlEscape

  # ---------------------------
  # Shiny UI
  # ---------------------------
  app_ui <- function() {
# ui <- fluidPage(
shiny::fluidPage(
  shiny::titlePanel("Design Properties Analyzer with Live validation of inputs"),
  shiny::sidebarLayout(
    shiny::sidebarPanel(
      shiny::numericInput("n_factors", "Number of factors (integer >1):", value = 2, min = 2, step = 1),
      shiny::textInput("factor_levels", "Factor levels (comma-separated):", value = "3,4"),
      shiny::numericInput("block_size", "Block size (k):", value = 3, min = 1, step = 1),
      shiny::numericInput("replications", "Replications (r):", value = 1, min = 1, step = 1),
      shiny::actionButton("generate_blocks", "Auto-generate matrix blocks", class = "btn-info"),
      shiny::tags$hr(),
      shiny::tags$div(HTML("<b>Blocks (matrix-style)</b><br>First row = block names (B1 B2 ...). Edit if needed.")),
      shiny::textAreaInput("blocks", NULL, value = "", rows = 8, placeholder = "B1 B2 B3 B4\n00 01 02 03\n10 11 12 13\n20 21 22 23"),
      shiny::tags$br(),
      shiny::actionButton("analyze", "Analyze design", class = "btn-success"),
      shiny::tags$br(), shiny::tags$small("Invalid entries highlighted in red; blanks in yellow.")
    ),
    shiny::mainPanel(
      shiny::htmlOutput("preview_table"),
      shiny::tags$hr(),
      shiny::htmlOutput("analysis_output")
    )
  )
)
}

  # ---------------------------
  # Shiny Server
  # ---------------------------
  app_server <- function(input, output, session) {
  
  # Auto-generate matrix blocks
  observeEvent(input$generate_blocks, {
    n <- input$n_factors
    levels <- suppressWarnings(as.numeric(unlist(strsplit(input$factor_levels,","))))
    if (length(levels) != n || any(is.na(levels))) {
      showNotification("Factor levels must be numeric and match n_factors", type = "error"); return()
    }
    k <- input$block_size; r <- input$replications
    total_treatments <- prod(levels)
    if (total_treatments %% k != 0) {
      showNotification("Block size k must divide total number of treatments", type = "error"); return()
    }
    base_blocks <- total_treatments / k
    grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s-1)))
    grids <- grids[, rev(seq_len(ncol(grids)))]
    treatments <- apply(grids, 1, paste0, collapse = "")
    ncol_mat <- base_blocks * r
    mat <- matrix("", nrow = k, ncol = ncol_mat)
    for (rep_i in 0:(r-1)) {
      for (b in 1:base_blocks) {
        col_idx <- rep_i*base_blocks + b
        start <- (b-1)*k + 1
        mat[, col_idx] <- treatments[start:(start + k - 1)]
      }
    }
    header <- paste0("B", 1:ncol_mat)
    table_text <- paste(c(paste(header, collapse = "\t"), apply(mat, 1, paste, collapse = "\t")), collapse = "\n")
    updateTextAreaInput(session, "blocks", value = table_text)
  })
  
  # Live preview + validation
  observe({
    txt <- input$blocks
    if (is.null(txt) || txt == "") {
      output$preview_table <- renderUI(HTML("<div style='color:gray;'>No blocks to preview.</div>"))
      return()
    }
    lines <- unlist(strsplit(txt, "\n"))
    lines <- lines[nzchar(lines)]
    if (length(lines) < 2) {
      output$preview_table <- renderUI(HTML("<div style='color:red;'>Enter header and at least one treatment row.</div>"))
      return()
    }
    header_tokens <- unlist(strsplit(lines[1], "\\s+"))
    n_blocks <- length(header_tokens)
    data_rows <- lines[-1]
    mat <- tryCatch({
      do.call(rbind, lapply(data_rows, function(x) unlist(strsplit(x, "\\s+"))))
    }, error = function(e) NULL)
    if (is.null(mat) || ncol(mat) != n_blocks) {
      output$preview_table <- renderUI(HTML("<div style='color:red;'>Matrix parse error: check consistent number of columns per row.</div>"))
      return()
    }
    factor_levels <- suppressWarnings(as.numeric(unlist(strsplit(input$factor_levels,","))))
    if (any(is.na(factor_levels))) factor_levels <- rep(0, input$n_factors)
    valid_ranges <- lapply(factor_levels, function(s) 0:(s-1))
    n_rows <- nrow(mat); n_cols <- ncol(mat)
    invalid <- matrix(FALSE, n_rows, n_cols)
    blank <- matrix(FALSE, n_rows, n_cols)
    for (i in 1:n_rows) {
      for (j in 1:n_cols) {
        cell <- mat[i, j]
        if (cell == "" || is.na(cell)) { blank[i, j] <- TRUE; next }
        digits <- unlist(strsplit(cell, ""))
        if (length(digits) != length(factor_levels)) { invalid[i, j] <- TRUE; next }
        nums <- suppressWarnings(as.numeric(digits))
        if (any(is.na(nums))) { invalid[i, j] <- TRUE; next }
        ok <- all(mapply(function(num, vr) num %in% vr, nums, valid_ranges))
        if (!ok) invalid[i, j] <- TRUE
      }
    }
    html <- "<table style='border-collapse:collapse;font-family:Courier New;'>"
    html <- paste0(html, "<tr>")
    for (h in header_tokens) html <- paste0(html, "<th style='border:1px solid #999;padding:6px;background:#f0f0f0;'>", h, "</th>")
    html <- paste0(html, "</tr>")
    for (i in 1:n_rows) {
      html <- paste0(html, "<tr>")
      for (j in 1:n_cols) {
        bg <- if (invalid[i, j]) "#ffcccc" else if (blank[i, j]) "#ffff99" else "#ffffff"
        html <- paste0(html, "<td style='border:1px solid #999;padding:6px;background:", 
                       bg, "'>", htmlEscape(mat[i,j]), "</td>")
      }
      html <- paste0(html, "</tr>")
    }
    html <- paste0(html, "</table>")
    problems <- c()
    if (any(invalid)) problems <- c(problems, paste0(sum(invalid), " invalid cells"))
    if (any(blank)) problems <- c(problems, paste0(sum(blank), " blank cells"))
    problems_text <- if (length(problems) == 0) "<span style='color:green;'>All cells valid</span>" else paste0("<span style='color:red;'><b>", paste(problems, collapse = "; "), "</b></span>")
    output$preview_table <- renderUI({
      HTML(paste0("<div>", problems_text, "</div><br>", html))
    })
  })
  
  # Analyze button (replacement)
  observeEvent(input$analyze, {
    txt <- input$blocks
    lines <- unlist(strsplit(txt, "\n")); lines <- lines[nzchar(lines)]
    if (length(lines) < 2) {
      output$analysis_output <- renderUI(HTML("<div style='color:red;'>Enter header and treatments before analysis.</div>")); return()
    }
    
    header_tokens <- unlist(strsplit(lines[1], "\\s+"))
    n_blocks <- length(header_tokens)
    
    mat <- tryCatch(
      do.call(rbind, lapply(lines[-1], function(x) unlist(strsplit(x, "\\s+")))),
      error = function(e) NULL
    )
    if (is.null(mat) || ncol(mat) != n_blocks) {
      output$analysis_output <- renderUI(HTML("<div style='color:red;'>Matrix parse error: inconsistent columns.</div>")); return()
    }
    
    # canonical treatment list built in the same way as generation
    factor_levels <- suppressWarnings(as.numeric(unlist(strsplit(input$factor_levels,","))))
    if (any(is.na(factor_levels)) || length(factor_levels) < 1) {
      output$analysis_output <- renderUI(HTML("<div style='color:red;'>Invalid factor_levels. Must be comma-separated integers.</div>")); return()
    }
    n_factors <- length(factor_levels)
    
    make_treatments <- function(levels) {
      # build lexicographic treatment labels like "00","01",... for multi-factor case
      grids <- do.call(expand.grid, lapply(rev(levels), function(s) 0:(s-1)))
      # reorder columns to original factor order
      grids <- grids[, rev(seq_len(ncol(grids))), drop = FALSE]
      apply(grids, 1, paste0, collapse = "")
    }
    treatments <- make_treatments(factor_levels)
    
    # find offending cells (not in canonical treatments)
    n_rows <- nrow(mat); n_cols <- ncol(mat)
    bad_cells <- list()
    for (i in seq_len(n_rows)) {
      for (j in seq_len(n_cols)) {
        cell <- mat[i, j]
        if (is.na(cell) || cell == "") next
        if (!(cell %in% treatments)) {
          bad_cells[[length(bad_cells) + 1]] <- list(row = i, col = j, header = header_tokens[j], value = cell)
        }
      }
    }
    
    if (length(bad_cells) > 0) {
      # present clear error with locations
      details <- paste0(sapply(bad_cells, function(b) {
        sprintf("Row %d, Column %d (block %s): '%s'", b$row, b$col, b$header, b$value)
      }), collapse = "<br>")
      msg <- paste0(
        "<div style='color:red;'><b>Error:</b> The following block entries are not in the canonical treatment list:<br>",
        details,
        "<br><br>Expected treatments look like (first 20 shown):<br><pre style='font-family:Courier New;'>",
        paste(head(treatments, 20), collapse = " "), if (length(treatments) > 20) " ..." , "</pre></div>"
      )
      output$analysis_output <- renderUI(HTML(msg))
      return()
    }
    
    # All entries present - call analyze_design safely
    blocks <- lapply(1:n_blocks, function(j) mat[, j])
    res_html <- tryCatch({
      cap <- capture.output(analyze_design(factor_levels, blocks))
      paste0("<pre style='white-space:pre-wrap;font-family:Courier New;'>", paste(cap, collapse = "\n"), "</pre>")
    }, error = function(e) {
      paste0("<div style='color:red;'><b>Error in analyze_design():</b><br>", htmlEscape(e$message), "</div>")
    })
    output$analysis_output <- renderUI(HTML(res_html))
  })
  
}

  shiny::shinyApp(ui = app_ui(), server = app_server)  
}



# examples
# blocks = autogenerate_design(2,c(3,4),3,2)
# blocks = as.matrix(blocks)
# blocks = lapply(seq_len(ncol(blocks)), function(i) blocks[, i])
# analyze_design(c(3,4), blocks)
