% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimal.R
\name{csmtc}
\alias{csmtc}
\title{Cross-sectional optimal multi-task forecast combination}
\usage{
csmtc(base, comb = "ols", res = NULL, approach = "proj",
      nn = NULL, settings = NULL, bounds = NULL, agg_mat = NULL, ...)
}
\arguments{
\item{base}{A list of \eqn{p} numeric (\eqn{h \times n}) matrix or multivariate time series
(\code{mts} class) containing the base forecasts to be reconciled; \eqn{h} is the forecast
horizon, \eqn{n} is the total number of time series (\eqn{n = n_u + n_b}) and
\eqn{p} is the total number of experts.}

\item{comb}{A string specifying the reconciliation method. For details, see \link{cscov}.}

\item{res}{A list of \eqn{p} numeric (\eqn{N \times n}) matrix containing the
in-sample residuals. This input is used to compute some covariance matrices.}

\item{approach}{A string specifying the approach used to compute the reconciled
forecasts. Options include:
\itemize{
\item "\code{proj}" (\emph{default}): zero-constrained projection approach.
\item "\code{osqp}": OSQP solver (Stellato et al., 2020).
}}

\item{nn}{A string specifying the algorithm to compute non-negative forecasts:
\itemize{
\item "\code{osqp}": OSQP solver (Stellato et al., 2020).
\item "\code{sntz}": heuristic "set-negative-to-zero".
}}

\item{settings}{An object of class \code{osqpSettings} specifying settings
for the \href{https://osqp.org/}{\pkg{osqp}} solver. For details, refer to the
\href{https://osqp.org/}{\pkg{osqp} documentation} (Stellato et al., 2020).}

\item{bounds}{A matrix (see \code{set_bounds} in
\href{https://danigiro.github.io/FoReco/reference/set_bounds.html}{\pkg{FoReco}})
with 3 columns (\eqn{i,lower,upper}), such that
\itemize{
\item Column 1 represents the cross-sectional series (\eqn{i = 1, \dots, n}).
\item Column 2 indicates the \emph{lower} bound.
\item Column 3 indicates the \emph{upper} bound.
}}

\item{agg_mat}{A (\eqn{n_u \times n_b}) numeric matrix representing the cross-sectional
aggregation matrix, mapping the \eqn{n_b} bottom-level (free)
variables into the \eqn{n_u} upper (constrained) variables.}

\item{...}{Arguments passed on to \link{cscov}.}
}
\value{
A (\eqn{h \times n}) numeric matrix of cross-sectional multi-task combined forecasts.
}
\description{
This function computes the optimal multi-task linear forecast combination, as described
in Girolimetto and Di Fonzo (2024)
}
\examples{
set.seed(123)
# (2 x 3) base forecasts matrix (simulated), expert 1
base1 <- matrix(rnorm(6, mean = c(20, 10, 10)), 2, byrow = TRUE)
# (10 x 3) in-sample residuals matrix (simulated), expert 1
res1 <- t(matrix(rnorm(n = 30), nrow = 3))

# (2 x 3) base forecasts matrix (simulated), expert 2
base2 <- matrix(rnorm(6, mean = c(20, 10, 10)), 2, byrow = TRUE)
# (10 x 3) in-sample residuals matrix (simulated), expert 2
res2 <- t(matrix(rnorm(n = 30), nrow = 3))

## BALANCED PANEL OF FORECASTS
# Base forecasts' and residuals' lists
brc <- list(base1, base2)
erc <- list(res1, res2)

# Aggregation matrix for Z = X + Y
A <- t(c(1,1))
rrc <- csocc(base = brc, agg_mat = A, comb = "shr", res = erc)

yc <- csmtc(base = brc, agg_mat = A, comb = "shr", res = erc)
M <- occmat(base = brc, agg_mat = A, comb = "shr", p = 2, res = erc)$M
M\%*\%t(yc)-t(rrc)

## UNBALANCED PANEL OF FORECASTS
base2[, 2] <- res2[, 2] <-  NA

# Base forecasts' and residuals' lists
bgc <- list(base1, base2)
egc <- list(res1, res2)
matNA <- matrix(1, 3, 2)
matNA[2,2] <- 0

# Aggregation matrix for Z = X + Y
A <- t(c(1,1))
rgc <- csocc(base = bgc, agg_mat = A, comb = "shr", res = egc)

yc <- csmtc(base = bgc, agg_mat = A, comb = "shr", res = egc)
M <- occmat(base = bgc, agg_mat = A, comb = "shr", p = 2, res = egc, matNA = matNA)$M
M\%*\%t(yc)-t(rgc)

}
\references{
Girolimetto, D. and Di Fonzo, T. (2024), Coherent forecast combination for linearly
constrained multiple time series, \doi{10.48550/arXiv.2412.03429}.
}
\seealso{
Other Optimal combination: 
\code{\link{cscov}()},
\code{\link{csocc}()},
\code{\link{occmat}()}
}
\concept{Optimal combination}
