\name{localdecodeHMM}
\alias{localdecodeHMM}
\title{Local Decoding for Hidden Markov Models}
\description{
Computes the most likely hidden state sequence for a univariate time series
using a fitted Hidden Markov Model (HMM) object via the forward-backward algorithm.
}
\usage{
localdecodeHMM(x, HMM, obsdist)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HMM}{List. A fitted HMM object, typically returned by \code{\link{findmleHMM}}. Must contain estimated parameters \code{delta} and \code{Pi}, as well as state-dependent observation parameters.}

\item{obsdist}{Character string. The observation distribution used in the fitted HMM. Supported distributions: "norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}
}
\value{
Numeric vector of length \code{length(x)}, containing the most likely hidden state at each observation, computed via local decoding using the forward-backward algorithm.
}
\details{
This function uses the forward-backward algorithm to compute local state probabilities
from a fitted Hidden Markov Model (HMM). The most probable state at each observation
is returned. Initial state probabilities are derived from the stationary distribution
of the transition matrix \code{Pi} if not explicitly provided.

Supported observation distributions include normal, Poisson, Weibull, zero-inflated Poisson (ZIP),
negative binomial, zero-inflated negative binomial (ZINB), exponential, gamma, log-normal, generalized extreme value (GEV), zero-inflated normal, and zero-inflated gamma.
}
\examples{
# Example with 3 states, normal observations
J <- 3

# HMM transition matrix
Pi <- matrix(c(0.8, 0.15, 0.05,
               0.1, 0.7, 0.2,
               0.2, 0.3, 0.5), nrow = 3, byrow = TRUE)

# Observation parameters (normal distribution)
obspar <- list(
  mean = c(-2, 0, 3),
  sd = c(0.5, 1, 1.5)
)

# Simulate HMM data
sim_data <- generateHMM(n = 200, J = J, obsdist = "norm", obspar = obspar, Pi = Pi)

# Fit HMM to simulated data
HMM_fit <- findmleHMM(J = J, x = sim_data$x, obsdist = "norm",
                       obspar = obspar, Pi = Pi, EM = FALSE)

# Compute local decoding
decoded_states <- localdecodeHMM(x = sim_data$x, HMM = HMM_fit, obsdist = "norm")

decoded_states
}
\seealso{
\code{\link{findmleHMM}}, for estimating HMM parameters from data.
\code{\link{generateHMM}}, for simulating HMM data.
\code{\link{localdecodeHSMM}}, for local decoding of Hidden Semi-Markov Models.
\code{\link{globaldecodeHMM}}, for global decoding in HMMs.
}
\author{Aimee Cody}

