obsprobs <- function(x, J, obsdist, obspar) {
  tau <- length(x)
  # Initialize observation probability matrix
  obsprobs <- matrix(NA, nrow = tau, ncol = J)

  if (obsdist == "norm") {
    # Normal distribution
    if (is.null(obspar$mean) || is.null(obspar$sd)) {
      stop("please supply 'mean' and 'sd' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dnorm(x, obspar$mean[i], obspar$sd[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1  # Handle numerical issues
    }
  }

  if (obsdist == "pois") {
    # Poisson distribution
    if (is.null(obspar$lambda)) {
      stop("please supply 'lambda' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dpois(x, obspar$lambda[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "weibull") {
    # Weibull distribution
    if (is.null(obspar$shape) || is.null(obspar$scale)) {
      stop("please supply 'shape' and 'scale' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dweibull(x, shape = obspar$shape[i], scale = obspar$scale[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "zip") {
    # Zero-inflated Poisson distribution
    if (is.null(obspar$pi) || is.null(obspar$lambda)) {
      stop("please supply 'pi' and 'lambda' for the observation parameters")
    }
    for (i in 1:tau) {
      for (j in 1:J) {
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(Poisson=0)
          obsprobs[i, j] <- obspar$pi[j] + (1 - obspar$pi[j]) * dpois(0, obspar$lambda[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        } else {
          # P(X>0) = (1-pi) * P(Poisson=x)
          obsprobs[i, j] <- (1 - obspar$pi[j]) * dpois(x[i], obspar$lambda[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        }
      }
    }
  }

  if (obsdist == "nbinom") {
    # Negative binomial distribution
    if (is.null(obspar$size) || is.null(obspar$mu)) {
      stop("please supply 'size' and 'mu' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dnbinom(x, size = obspar$size[i], mu = obspar$mu[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "zinb") {
    # Zero-inflated negative binomial distribution
    for (i in 1:tau) {
      for (j in 1:J) {
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(NegBinom=0)
          obsprobs[i, j] <- obspar$pi[j] + (1 - obspar$pi[j]) * dnbinom(0, size = obspar$size[j], mu = obspar$mu[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        } else {
          # P(X>0) = (1-pi) * P(NegBinom=x)
          obsprobs[i, j] <- (1 - obspar$pi[j]) * dnbinom(x[i], size = obspar$size[j], mu = obspar$mu[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        }
      }
    }
  }

  if (obsdist == "exp") {
    # Exponential distribution
    if (is.null(obspar$rate)) {
      stop("please supply 'rate' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dexp(x, rate = obspar$rate[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "gamma") {
    # Gamma distribution
    if (is.null(obspar$shape) || is.null(obspar$rate)) {
      stop("please supply 'shape' and 'rate' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dgamma(x, shape = obspar$shape[i], rate = obspar$rate[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "lnorm") {
    # Log-normal distribution
    if (is.null(obspar$meanlog) || is.null(obspar$sdlog)) {
      stop("please supply 'meanlog' and 'sdlog' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- dlnorm(x, meanlog = obspar$meanlog[i], sdlog = obspar$sdlog[i])
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "gev") {
    # Generalized extreme value distribution
    if (is.null(obspar$loc) || is.null(obspar$scale) || is.null(obspar$shape)) {
      stop("please supply 'loc' and 'scale' and 'shape' for the observation parameters")
    }
    for (i in 1:J) {
      obsprobs[, i] <- devd(x, loc = obspar$loc[i], scale = obspar$scale[i], shape = obspar$shape[i], log = FALSE, type = "GEV")
      obsprobs[is.na(obsprobs[, i]), i] <- 1
    }
  }

  if (obsdist == "ZInormal") {
    # Zero-inflated normal distribution
    for (i in 1:tau) {
      for (j in 1:J) {
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(Norm=0)
          obsprobs[i, j] <- obspar$pi[j] + (1 - obspar$pi[j]) * dnorm(0, mean = obspar$mean[j], sd = obspar$sd[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        } else {
          # P(X>0) = (1-pi) * P(Norm=x)
          obsprobs[i, j] <- (1 - obspar$pi[j]) * dnorm(x[i], mean = obspar$mean[j], sd = obspar$sd[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        }
      }
    }
  }

  if (obsdist == "ZIgamma") {
    # Zero-inflated gamma distribution
    for (i in 1:tau) {
      for (j in 1:J) {
        if (x[i] == 0) {
          # P(X=0) = pi + (1-pi) * P(Gamma=0)
          obsprobs[i, j] <- obspar$pi[j] + (1 - obspar$pi[j]) * dgamma(0, shape = obspar$shape[j], rate = obspar$rate[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        } else {
          # P(X>0) = (1-pi) * P(Gamma=x)
          obsprobs[i, j] <- (1 - obspar$pi[j]) * dgamma(x[i], shape = obspar$shape[j], rate = obspar$rate[j])
          obsprobs[is.na(obsprobs[, j]), j] <- 1
        }
      }
    }
  }
  return(obsprobs)
}



