plotHMMParameters <- function(x, HMM, obsdist, confint_result = NULL,
                              level = 0.95, B = 100,
                              time_structure = NULL,
                              plot_title = "HMM Parameters Over Time",
                              overlay_data = NULL, overlay_label = "Data",
                              colors = c("black", "red", "blue", "green"),
                              save_plot = FALSE, filename = NULL,
                              width = 12, height = 8, dpi = 300,
                              verbose = TRUE, seed = NULL) {

  # Validate observation distribution
  if (!obsdist %in% c("pois", "norm", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma")) {
    stop("Observation distribution not supported")
  }

  # Extract basic dimensions
  n <- length(x)
  J <- length(HMM$estimate$delta)

  # Decode most likely state sequence using Viterbi algorithm
  decoded_states <- globaldecodeHMM(
    x = x,
    HMM = HMM,
    obsdist = obsdist
  )

  # Extract observation parameters and get distribution-specific info
  obspar <- HMM$estimate[1:(length(HMM$estimate) - 2)]
  param_info <- getParameterInfo(obsdist, J)

  # Compute confidence intervals if not provided
  if (is.null(confint_result)) {
    if (verbose) message("Computing confidence intervals...")
    confint_result <- confintHMM(x, HMM, obsdist, level = level, B = B,
                                 verbose = verbose, seed = seed)
  }

  # Set up time axis information
  time_info <- createTimeInfo(n, time_structure)

  # Create parameter time series based on decoded states
  param_series <- createParameterTimeSeries(obspar, decoded_states, param_info, n, J)

  # Create confidence interval time series
  ci_series <- createCITimeSeries(confint_result, decoded_states, param_info, n, J)

  # Initialize graphics device for saving if requested
  if (save_plot) {
    if (is.null(filename)) {
      stop("filename must be specified when save_plot = TRUE")
    }
    png(filename, width = width, height = height, units = "in", res = dpi)
    on.exit(dev.off(), add = TRUE)
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  # Generate the parameter plots
  createParameterPlots(
    param_series = param_series,
    ci_series = ci_series,
    param_info = param_info,
    time_info = time_info,
    plot_title = plot_title,
    overlay_data = overlay_data,
    overlay_label = overlay_label,
    colors = colors
  )

  if (save_plot && verbose) {
    message("Plot saved to: ", filename)
  }

  invisible(list(
    param_series = param_series,
    ci_series = ci_series,
    time_info = time_info,
    decoded_states = decoded_states
  ))
}


createTimeInfo <- function(n, time_structure = NULL) {

  # Default to simple observation numbering if no structure provided
  if (is.null(time_structure)) {
    return(list(
      labels = 1:n,
      unit = "observation",
      conversion_factor = 1,
      unit_name = "observation",
      x_label = "Time"
    ))
  }

  # Validate required fields based on time structure type
  required_fields <- c("unit", "observations_per_unit")
  if (time_structure$unit == "custom") {
    required_fields <- c("conversion_factor", "unit_name")
  }

  missing_fields <- setdiff(required_fields, names(time_structure))
  if (length(missing_fields) > 0) {
    stop("Missing required fields in time_structure: ", paste(missing_fields, collapse = ", "))
  }

  # Set conversion factor and unit names
  if (time_structure$unit == "custom") {
    conversion_factor <- time_structure$conversion_factor
    unit_name <- time_structure$unit_name
    x_label <- paste("Time (", unit_name, "s)", sep = "")
  } else {
    conversion_factor <- time_structure$observations_per_unit
    unit_name <- time_structure$unit

    # Create appropriate axis labels for common units
    if (unit_name == "year") {
      x_label <- "Time (years)"
    } else if (unit_name == "day") {
      x_label <- "Time (days)"
    } else if (unit_name == "hour") {
      x_label <- "Time (hours)"
    } else {
      x_label <- paste("Time (", unit_name, "s)", sep = "")
    }
  }

  # Calculate time duration and handle start/end points
  duration <- n / conversion_factor
  has_start <- !is.null(time_structure$start_point)
  has_end <- !is.null(time_structure$end_point)

  if (has_start && has_end) {
    start_point <- time_structure$start_point
    end_point <- time_structure$end_point
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (has_start && !has_end) {
    start_point <- time_structure$start_point
    end_point <- start_point + duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else if (!has_start && has_end) {
    end_point <- time_structure$end_point
    start_point <- end_point - duration
    time_labels <- seq(start_point, end_point, length.out = n)
  } else {
    start_point <- 0
    end_point <- duration
    time_labels <- seq(start_point, end_point, length.out = n)
  }

  return(list(
    labels = time_labels,
    unit = ifelse(time_structure$unit == "custom", "custom", time_structure$unit),
    conversion_factor = conversion_factor,
    unit_name = unit_name,
    x_label = x_label,
    duration = duration,
    start_point = start_point,
    end_point = end_point
  ))
}


`%||%` <- function(x, y) {
  if (is.null(x)) y else x
}


getParameterInfo <- function(obsdist, J) {
  param_info <- switch(obsdist,
                       "pois" = list(
                         names = "lambda",
                         labels = "Poisson rate",
                         count = 1,
                         transform = exp
                       ),
                       "norm" = list(
                         names = c("mean", "sd"),
                         labels = c("Normal mean", "Normal SD"),
                         count = 2,
                         transform = list(identity, exp)
                       ),
                       "weibull" = list(
                         names = c("shape", "scale"),
                         labels = c("Weibull shape", "Weibull scale"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "zip" = list(
                         names = c("pi", "lambda"),
                         labels = c("ZIP zero probability", "ZIP rate"),
                         count = 2,
                         transform = list(plogis, exp)
                       ),
                       "nbinom" = list(
                         names = c("size", "mu"),
                         labels = c("NB size", "NB mean"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "zinb" = list(
                         names = c("pi", "size", "mu"),
                         labels = c("ZINB zero probability", "ZINB size", "ZINB mean"),
                         count = 3,
                         transform = list(plogis, exp, exp)
                       ),
                       "exp" = list(
                         names = "rate",
                         labels = "Exponential rate",
                         count = 1,
                         transform = exp
                       ),
                       "gamma" = list(
                         names = c("shape", "rate"),
                         labels = c("Gamma shape", "Gamma rate"),
                         count = 2,
                         transform = list(exp, exp)
                       ),
                       "lnorm" = list(
                         names = c("meanlog", "sdlog"),
                         labels = c("Log-normal mean", "Log-normal SD"),
                         count = 2,
                         transform = list(identity, exp)
                       ),
                       "gev" = list(
                         names = c("loc", "scale", "shape"),
                         labels = c("GEV location", "GEV scale", "GEV shape"),
                         count = 3,
                         transform = list(identity, exp, identity)
                       ),
                       "ZInormal" = list(
                         names = c("pi", "mean", "sd"),
                         labels = c("ZI-Normal zero probability", "ZI-Normal mean", "ZI-Normal SD"),
                         count = 3,
                         transform = list(plogis, identity, exp)
                       ),
                       "ZIgamma" = list(
                         names = c("pi", "shape", "rate"),
                         labels = c("ZI-Gamma zero probability", "ZI-Gamma shape", "ZI-Gamma rate"),
                         count = 3,
                         transform = list(plogis, exp, exp)
                       )
  )

  return(param_info)
}


createParameterTimeSeries <- function(obspar, decoded_states, param_info, n, J) {
  param_series <- list()

  # Create time series for each parameter
  for (p in 1:param_info$count) {
    param_name <- param_info$names[p]
    param_values <- obspar[[param_name]]

    # Map state-specific parameters to time points
    series <- numeric(n)
    for (t in 1:n) {
      series[t] <- param_values[decoded_states[t]]
    }

    param_series[[param_name]] <- series
  }

  return(param_series)
}


createCITimeSeries <- function(confint_result, decoded_states, param_info, n, J) {
  ci_series <- list()

  # Create CI time series for each parameter
  for (p in 1:param_info$count) {
    param_name <- param_info$names[p]

    # Find parameter rows in confidence interval results
    param_rows <- grep(param_name, confint_result$obspar_CI$Parameter)

    if (length(param_rows) == J) {
      lower_values <- confint_result$obspar_CI$Lower[param_rows]
      upper_values <- confint_result$obspar_CI$Upper[param_rows]

      # Map state-specific CIs to time points
      lower_series <- numeric(n)
      upper_series <- numeric(n)

      for (t in 1:n) {
        lower_series[t] <- lower_values[decoded_states[t]]
        upper_series[t] <- upper_values[decoded_states[t]]
      }

      ci_series[[param_name]] <- list(
        lower = lower_series,
        upper = upper_series
      )
    }
  }

  return(ci_series)
}


createParameterPlots <- function(param_series, ci_series, param_info, time_info,
                                 plot_title, overlay_data, overlay_label, colors) {

  n_params <- param_info$count
  time_labels <- time_info$labels
  n <- length(time_labels)

  # Set up panel layout based on number of parameters
  if (n_params <= 3) {
    par(mfrow = c(1, n_params), mar = c(4.5, 4.5, 4.5, 3.5), oma = c(0, 0, 2, 0))
  } else {
    par(mfrow = c(2, ceiling(n_params/2)), mar = c(4.5, 4.5, 4.5, 3.5), oma = c(0, 0, 2, 0))
  }

  # Create subplot for each parameter
  for (p in 1:n_params) {
    param_name <- param_info$names[p]
    param_label <- param_info$labels[p]

    series <- param_series[[param_name]]

    # Set y-axis limits to accommodate parameter values and CIs
    if (param_name %in% names(ci_series)) {
      ylim <- range(c(series, ci_series[[param_name]]$lower, ci_series[[param_name]]$upper), na.rm = TRUE)
    } else {
      ylim <- range(series, na.rm = TRUE)
    }

    # Adjust y-axis limits for overlay data if present
    if (!is.null(overlay_data) && length(overlay_data) == length(time_labels)) {
      ylim_range <- diff(ylim)
      ylim[1] <- ylim[1] - 0.5 * ylim_range
      ylim[2] <- ylim[2] + 0.05 * ylim_range
    } else {
      ylim_range <- diff(ylim)
      ylim[1] <- ylim[1] - 0.05 * ylim_range
      ylim[2] <- ylim[2] + 0.05 * ylim_range
    }

    # Plot main parameter time series
    plot(time_labels, series, type = "l", lwd = 2, col = colors[1],
         xlab = time_info$x_label, ylab = param_label,
         main = paste("(", letters[p], ") ", param_label, sep = ""),
         ylim = ylim, cex.lab = 1.3, cex.axis = 1.2)

    # Add confidence intervals if available
    if (param_name %in% names(ci_series)) {
      lines(time_labels, ci_series[[param_name]]$lower, lty = "dashed", col = colors[2])
      lines(time_labels, ci_series[[param_name]]$upper, lty = "dashed", col = colors[2])
    }

    # Add overlay data if provided
    if (!is.null(overlay_data) && length(overlay_data) == length(time_labels)) {
      addOverlayData(time_labels, overlay_data, overlay_label, param_name,
                     param_label, ylim, p, n_params)
    }
  }

  # Add main title
  mtext(plot_title, outer = TRUE, cex = 1, font = 1)
}


addOverlayData <- function(time_labels, overlay_data, overlay_label, param_name,
                           param_label, ylim, p, n_params) {

  # Choose scaling factor based on parameter type
  if (param_name == "loc" || grepl("location", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  } else if (param_name == "scale" || grepl("scale", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  } else if (param_name == "shape" || grepl("shape", param_label, ignore.case = TRUE)) {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.32)
  } else {
    overlay_scaled <- scale_overlay_to_bottom(overlay_data, ylim, scale_factor = 0.28)
  }

  # Add overlay line
  lines(time_labels, overlay_scaled, col = "grey30", lwd = 1)

  # Create right-hand axis with original overlay scale
  overlay_range <- range(overlay_data, na.rm = TRUE)
  overlay_ticks <- pretty(overlay_range, n = 3)

  # Scale tick positions to match overlay data scaling
  if (param_name == "loc" || grepl("location", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  } else if (param_name == "scale" || grepl("scale", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  } else if (param_name == "shape" || grepl("shape", param_label, ignore.case = TRUE)) {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.32)
  } else {
    overlay_scaled_ticks <- scale_overlay_to_bottom(overlay_ticks, ylim, scale_factor = 0.28)
  }

  # Add right-hand axis
  axis(4, at = overlay_scaled_ticks, labels = overlay_ticks,
       cex.axis = 1.2, col = "darkgrey", col.axis = "darkgrey")

  # Add overlay label to rightmost subplot
  if (p == n_params) {
    mtext(overlay_label, side = 4, line = 2.2, cex = 1.3,
          las = 0, adj = 0, col = "darkgrey")
  }
}


scale_overlay_to_bottom <- function(x, target_range, scale_factor = 0.28) {
  x_range <- range(x, na.rm = TRUE)
  x_normalized <- (x - x_range[1]) / diff(x_range)
  bottom_start <- target_range[1]
  bottom_height <- diff(target_range) * scale_factor
  x_scaled <- x_normalized * bottom_height + bottom_start
  return(x_scaled)
}


scale_to_range <- function(x, target_range) {
  x_range <- range(x, na.rm = TRUE)
  x_scaled <- (x - x_range[1]) / diff(x_range) * diff(target_range) + target_range[1]
  return(x_scaled)
}

