\name{confintHSMM}
\alias{confintHSMM}
\title{Bootstrap Confidence Intervals for Hidden Semi-Markov Models}
\description{
Computes bootstrap confidence intervals for the parameters of a fitted Hidden Semi-Markov Model (HSMM).
The function resamples data by simulation from the fitted model, refits the HSMM repeatedly,
and constructs confidence intervals for the state-dependent observation parameters,
dwell-time distribution parameters, and the transition probability matrix.
}
\usage{
confintHSMM(x, HSMM, obsdist, dwelldist, level = 0.95, B = 100, M = NA,
            shift = FALSE, maxiter = 100, tol = 1e-5, verbose = TRUE, seed = NULL)
}
\arguments{
\item{x}{Numeric vector. The observed data sequence.}

\item{HSMM}{A list containing estimated HSMM parameters, typically returned by \code{\link{findmleHSMM}}.
Must include the transition matrix \code{Pi}, observation parameters, and dwell-time parameters.}

\item{obsdist}{Character string. Observation distribution. Supported distributions:
"norm", "pois", "weibull", "zip", "nbinom", "zinb", "exp", "gamma", "lnorm", "gev", "ZInormal", "ZIgamma".}

\item{dwelldist}{Character string. Dwell-time distribution. Supported distributions:
"pois", "nbinom", "betabinom".}

\item{level}{Numeric. Confidence level for the intervals. Default is \code{0.95}.}

\item{B}{Integer. Number of bootstrap replicates to perform. Default is \code{100}.}

\item{M}{Integer. Maximum dwell time to consider for semi-Markov states.
If \code{NA}, defaults to \code{min(length(x), 1000)}.}

\item{shift}{Logical. Indicates whether the dwell-time distribution includes a user-specified shift parameter.
If \code{FALSE}, a shift of 1 is applied automatically. Defaults to \code{FALSE}.}

\item{maxiter}{Integer. Maximum number of EM iterations for each bootstrap refit. Default is \code{100}.}

\item{tol}{Numeric. Convergence tolerance for the EM algorithm during bootstrap refits. Default is \code{1e-5}.}

\item{verbose}{Logical. If \code{TRUE}, progress messages are printed to the console. Default is \code{TRUE}.}

\item{seed}{Integer or \code{NULL}. Random seed for reproducibility. Default is \code{NULL}.}
}
\value{
A list with four elements:
\item{obspar_CI}{A data frame with columns:
  \code{Parameter}, \code{Estimate}, \code{Lower}, and \code{Upper}, giving bootstrap confidence intervals for the observation distribution parameters.}
\item{dwellpar_CI}{A data frame with columns:
  \code{Parameter}, \code{Estimate}, \code{Lower}, and \code{Upper}, giving bootstrap confidence intervals for the dwell-time distribution parameters.}
\item{Pi_CI}{A list containing:
  \code{Estimate}, the estimated transition matrix from the fitted model;
  \code{Lower}, the lower bounds of the bootstrap confidence intervals for each transition probability;
  \code{Upper}, the corresponding upper bounds.}
\item{bootstrap_samples}{A list containing:
  \code{obspar}, matrix of bootstrap samples for observation parameters;
  \code{dwellpar}, matrix of bootstrap samples for dwell-time parameters;
  \code{Pi}, array of bootstrap samples for transition matrix;
  \code{n_successful}, number of successful bootstrap iterations;
  \code{n_attempts}, total number of attempts made.}
}
\details{
The function uses a parametric bootstrap procedure. Data are simulated from the fitted HSMM using \code{\link{generateHSMM}},
and parameters are re-estimated with \code{\link{findmleHSMM}}. Confidence intervals are then obtained from
the empirical quantiles of the bootstrap replicates.
When \code{verbose = TRUE}, progress is reported. Replicates that fail to converge are retried until \code{B} successful fits are obtained.
}
\examples{
J <- 2
Pi <- matrix(c(0, 1,
               1, 0), nrow = 2, byrow = TRUE)

obspar <- list(mean = c(4, 3), sd = c(1, 1))

dwellpar <- list(lambda = c(5, 7))

sim_data <- generateHSMM(n = 1000, J = J, obsdist = "norm",
                         dwelldist = "pois", obspar = obspar,
                         dwellpar = dwellpar, Pi = Pi)

HSMM_fit <- findmleHSMM(x = sim_data$x, J = J, M = 100,
                        obsdist = "norm", dwelldist = "pois",
                        obspar = obspar, dwellpar = dwellpar, Pi = Pi)

\donttest{
ci <- confintHSMM(x = sim_data$x, HSMM = HSMM_fit,
                  obsdist = "norm", dwelldist = "pois",
                  B = 50, M = 50, level = 0.9)

ci$obspar_CI
ci$dwellpar_CI
ci$Pi_CI

ci_silent <- confintHSMM(x = sim_data$x, HSMM = HSMM_fit,
                         obsdist = "norm", dwelldist = "pois",
                         B = 50, M = 50, level = 0.9, verbose = FALSE)
}
}
\seealso{
\code{\link{findmleHSMM}}, for estimating HSMM parameters from data.
\code{\link{generateHSMM}}, for simulating HSMM data.
\code{\link{confintHMM}}, computing bootstrapped confidence intervals for a hidden Markov model.
}
\author{Aimee Cody}

