% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{ssmf}
\alias{ssmf}
\title{Simplex-structured matrix factorisation algorithm (SSMF).}
\usage{
ssmf(
  data,
  k,
  H = NULL,
  W = NULL,
  meth = c("kmeans", "uniform", "dirichlet", "nmf"),
  lr = 0.01,
  nruns = 50
)
}
\arguments{
\item{data}{Data matrix or data frame.}

\item{k}{The number of prototypes/clusters.}

\item{H}{Matrix, user input \eqn{H} matrix to start the algorithm. If input is empty, the function will initialise \eqn{H} matrix automatically.}

\item{W}{Matrix, user input \eqn{W} matrix to start the algorithm. If input is empty, the function will initialise \eqn{W} matrix automatically.}

\item{meth}{Specification of method to initialise the \eqn{W} and \eqn{H} matrix, see 'method' in \code{init()}.}

\item{lr}{Optimisation learning rate.}

\item{nruns}{The maximum times of running the algorithm.}
}
\value{
\code{W} The optimised \eqn{W} matrix, containing the values of prototypes.

\code{H} The optimised \eqn{H} matrix, containing the values of soft memberships.

\code{SSE} The residuals sum of square.
}
\description{
This function implements on SSMF on a data matrix or data frame.
}
\details{
Let \eqn{X \in R^{n \times p}} be the data set  with \eqn{n} observations and \eqn{p} variables.
Given an integer \eqn{k \ll \text{min}(n,p)},
the data set is clustered by simplex-structured matrix factorisation (SSMF), which aims to process soft clustering
and partition the observations into \eqn{k} fuzzy clusters such that the sum of squares from observations to the
assigned cluster prototypes is minimised.
SSMF finds \eqn{H_{n \times k}} and \eqn{W_{k \times p}},
such that \deqn{X \approx HW,}
A cluster prototype refers to a vector that represent the characteristics of a particular cluster,
denoted by \eqn{w_r \in \mathbb{R}^{p}} , where \eqn{r} is the \eqn{r^{th}} cluster.
A cluster membership vector \eqn{h_i \in \mathbb{R}^{k}} describes the proportion of the cluster prototypes
of the \eqn{i^{th}} observation. \eqn{W} is the prototype matrix where each row is the cluster prototype and
\eqn{H} is the soft membership matrix where each row gives the soft cluster membership of each observation.
The problem of finding the approximate matrix factorisation is solved by minising residual sum of squares (RSS), that is
\deqn{\mathrm{RSS} = \| X-HW \|^2 = \sum_{i=1}^{n}\sum_{j=1}^{p} \left\{ X_{ij}-(HW)_{ij}\right\}^2,}
such that \eqn{\sum_{r=1}^k h_{ir}=1}  and  \eqn{h_{ir}\geq 0}.
}
\examples{

\donttest{
library(MetabolSSMF)

# Initialisation by user
data <- SimulatedDataset
k <- 4

## Initialised by kmeans
fit.km <- kmeans(data, centers = k)

H <- mclust::unmap(fit.km$cluster)
W <- fit.km$centers

fit1 <- ssmf(data, k = k, H = H) #start the algorithm from H
fit2 <- ssmf(data, k = k, W = W) #start the algorithm from W

# Initialisation inside the function
fit3 <- ssmf(data, k = 4, meth = 'dirichlet')
fit4 <- ssmf(data, k = 4)
}
}
\references{
Abdolali, Maryam & Gillis, Nicolas. (2020). Simplex-Structured Matrix Factorization: Sparsity-based Identifiability and Provably Correct Algorithms. <doi:10.1137/20M1354982>
}
\author{
Wenxuan Liu
}
