#' Generate Mixed-Level or Regular Fractional Factorial Designs
#'
#' @description
#' `generate_ff()` constructs mixed-level fractional factorial designs or,
#' when all factors share the same number of levels \eqn{s}, automatically
#' searches for a regular \eqn{s^{k-p}} fractional factorial using generator
#' relations.
#'
#' #' The function performs:
#'
#' * **Part 1:** Design construction (regular or mixed-level)
#' * **Part 2:** Design diagnostics: balance (Hhat), J2, GBM,
#'   alias chains, and strong confounding summary
#' * **Part 3 (optional):** Deterministic trend-free run order based on
#'   Coster (1993)
#'
#' Output printing can be controlled via the `parts` argument.
#'
#'
#' @param levels_spec A numeric vector such as `c(2,3,4)` (levels per factor),
#'   or a named list of level labels (e.g. `list(A = 1:2, B = 1:3, C = 1:4)`).
#'
#' @param n_runs Number of experimental runs required.
#'
#' @param max_iter Maximum number of iterations for coordinate exchange or
#'   unique-subset improvement (default: `100`).
#'
#' @param a Weight for the J2 near-orthogonality criterion.
#' @param b Weight for the Hhat balance criterion.
#'
#' @param max_int_order Highest-order interaction used when building model
#'   matrices for alias calculations. Default is `3`.
#'
#' @param alias_min_abs_corr Minimum absolute correlation required for a pair of
#'   model terms to appear in the strong confounding summary table.
#'
#' @param tf Logical. When `TRUE`, computes a trend-free run order (Part 3)
#'   based on deterministic pairwise swapping.
#'
#' @param parts Integer vector selecting which sections to print:
#'   * `1` -> Part 1: Design
#'   * `2` -> Part 2: Properties & alias
#'   * `3` -> Part 3: Trend-free order
#'
#'   To print all: `parts = c(1,2,3)` (default).
#'
#' @param verbose Logical. When `TRUE`, prints results. When `FALSE`, performs
#'   all computations silently.
#'
#'
#' @details
#' ## Automatic Regular Fractional Factorial Detection
#' When all factors have the same number of levels \eqn{s}, and when
#' \eqn{n\_runs = s^{k-p}}, the function attempts to find a regular
#' \eqn{s^{k-p}} fraction via a heuristic generator search inspired by Guo et al.
#' (2007). The objective minimized is:
#'
#' The objective function minimized is
#' \code{Z = a * J2 + b * Hhat}.
#'
#' If no acceptable generator set is found, the algorithm reverts to a mixed-level
#' design strategy.
#'
#'
#' ## Mixed-Level Fractional Factorials
#' Mixed-level designs are constructed using the Pantoja-Pacheco et al. (2021)
#' NONBPA skeleton for nonmultiple levels, followed by a Guo-style coordinate
#' exchange improvement on the \eqn{Z}-criterion.
#'
#'
#' ## Alias Structure (Rios-Lira et al., 2021)
#' Alias relationships are computed from the correlation matrix of the model
#' matrix (main effects + interactions). Chains are built by selecting the pair
#' of terms with the highest absolute correlation, choosing the lower-order term
#' as the chain leader.
#'
#'
#' ## Trend-Free Run Order (Coster, 1993)
#' When `tf = TRUE`, a deterministic pairwise-swap algorithm produces a run order
#' orthogonal to polynomial time trends (linear by default).
#'
#'
#' @return
#' A list containing:
#'
#' * `design_int` -> integer-coded design
#' * `design_labels` -> decoded factor levels
#' * `level_maps` -> internal mapping of coded levels
#' * `Hhat` -> balance measure
#' * `J2hat` -> near-orthogonality measure
#' * `GBM` -> general balance metric
#' * `gbm_resolution` -> smallest order with imbalance
#' * `model_matrix` -> model matrix with interactions
#' * `corr_matrix` -> correlation matrix
#' * `alias_chains` -> alias chain structure
#' * `alias_summary` -> strong confounding pairs
#' * `trend_free_result` -> (if `tf = TRUE`) run order, reordered design,
#'   and final trend objective
#'
#' Returned invisibly unless assigned.
#'
#'
#' @references
#' Guo, Y., Simpson, J. R., & Pignatiello, J. J. (2007).
#' *Construction of Efficient Mixed-Level Fractional Factorial Designs.*
#' Journal of Quality Technology, 39(3), 241-257.
#' \doi{10.1080/00224065.2007.11917691}
#'
#' Pantoja-Pacheco, Y. V. et al. (2021).
#' *One Note for Fractionation and Increase for Mixed-Level Designs When the Levels Are Not Multiple.*
#' Mathematics, 9(13), 1455.
#' \doi{10.3390/math9131455}
#'
#' Rios-Lira, A. J. et al. (2021).
#' *Alias Structures and Sequential Experimentation for Mixed-Level Designs.*
#' Mathematics, 9(23), 3053.
#' \doi{10.3390/math9233053}
#'
#' Coster, D. C. (1993).
#' *Trend-Free Run Orders of Mixed-Level Fractional Factorial Designs.*
#' Annals of Statistics, 21(4), 2072-2086.
#' \doi{10.1214/aos/1176349410}
#' @examples
#' \donttest{
#'
#' # Mixed-level design (2 x 3 x 4), 12 runs, all parts printed
#' generate_ff(c(2,3,4), 12, tf = TRUE)
#'
#' # Only Part 1 (design)
#' generate_ff(c(2,3,4), 12, parts = 1, tf = FALSE)
#'
#' # Parts 1 and 2 only
#' generate_ff(c(2,3,4), 12, parts = c(1,2), tf = FALSE)
#'
#' # Only Part 3 (trend-free run order)
#' generate_ff(c(2,3,4), 12, parts = 3, tf = TRUE)
#'
#' }
#' @importFrom stats contr.poly cor
#' @importFrom utils combn head
#' @export
generate_ff <- function(levels_spec, n_runs,
                        max_iter = 100,
                        a = 1, b = 1,
                        max_int_order = 3,
                        alias_min_abs_corr = 0.9,
                        tf = FALSE,
                        parts = c(1, 2, 3),
                        verbose = TRUE) {
   encode_design <- function(design) {
    design <- as.data.frame(design, stringsAsFactors = FALSE)
    m <- ncol(design)
    names_f <- colnames(design)
    int_mat <- matrix(NA_integer_, nrow = nrow(design), ncol = m)
    colnames(int_mat) <- names_f
    level_maps <- vector("list", m)
    names(level_maps) <- names_f

    for (j in seq_len(m)) {
      col <- design[[j]]
      f <- as.factor(col)
      int_mat[, j] <- as.integer(f)
      level_maps[[j]] <- levels(f)
    }
    list(
      D_int = int_mat,
      level_maps = level_maps
    )
  }

  decode_design <- function(D_int, level_maps) {
    D_lab <- as.data.frame(D_int, stringsAsFactors = FALSE)
    for (j in seq_len(ncol(D_int))) {
      D_lab[[j]] <- level_maps[[j]][D_int[, j]]
    }
    colnames(D_lab) <- names(level_maps)
    D_lab
  }

  balance_Hhat <- function(D) {
    n <- nrow(D)
    m <- ncol(D)
    Hhat <- 0
    for (j in seq_len(m)) {
      counts <- table(D[, j])
      lj <- length(counts)
      pj <- counts / n
      Hhat <- Hhat + sum((pj - 1 / lj)^2)
    }
    Hhat
  }

  J2_hat <- function(D) {
    n <- nrow(D)
    m <- ncol(D)
    if (n < 2L) return(0)
    sum_delta2 <- 0
    pair_count <- 0L
    for (i in 1:(n - 1L)) {
      for (k in (i + 1L):n) {
        coincidences <- sum(D[i, ] == D[k, ])
        delta_hat <- coincidences / m
        sum_delta2 <- sum_delta2 + delta_hat^2
        pair_count <- pair_count + 1L
      }
    }
    if (pair_count == 0L) return(0)
    (1 / n) * (sum_delta2 / pair_count)
  }

  objective_Z <- function(D, a = 1, b = 1) {
    a * J2_hat(D) + b * balance_Hhat(D)
  }

  generate_NONBPA <- function(levels_per_factor, n_runs) {
    m <- length(levels_per_factor)
    int_levels <- vector("list", m)
    level_maps <- vector("list", m)
    names(int_levels) <- names(levels_per_factor)
    names(level_maps) <- names(levels_per_factor)

    for (j in seq_len(m)) {
      lev <- levels_per_factor[[j]]
      f <- as.factor(lev)
      level_maps[[j]] <- levels(f)
      int_levels[[j]] <- seq_len(length(levels(f)))
    }

    D <- matrix(NA_integer_, nrow = n_runs, ncol = m)
    colnames(D) <- names(int_levels)
    for (j in seq_len(m)) {
      D[, j] <- rep(int_levels[[j]], length.out = n_runs)
    }

    attr(D, "level_maps") <- level_maps
    D
  }

  improve_design_exchange <- function(D_init,
                                      level_sets,
                                      max_iter = 200,
                                      a = 1, b = 1,
                                      verbose = TRUE) {
    D <- D_init
    n <- nrow(D)
    m <- ncol(D)
    current_Z <- objective_Z(D, a, b)
    if (verbose) cat("Initial Z =", current_Z, "\n")
    for (it in seq_len(max_iter)) {
      improved <- FALSE
      for (j in seq_len(m)) {
        possible_vals <- level_sets[[j]]
        for (i in seq_len(n)) {
          cur_val <- D[i, j]
          best_val <- cur_val
          best_Z <- current_Z
          for (cand in possible_vals) {
            if (cand == cur_val) next
            D[i, j] <- cand
            z_new <- objective_Z(D, a, b)
            if (z_new + 1e-12 < best_Z) {
              best_Z <- z_new
              best_val <- cand
            }
          }
          if (best_val != cur_val) {
            D[i, j] <- best_val
            current_Z <- best_Z
            improved <- TRUE
          } else {
            D[i, j] <- cur_val
          }
        }
      }
      if (verbose) cat("Iter", it, "Z =", current_Z, "\n")
      if (!improved) break
    }
    if (verbose) cat("Final Z =", current_Z, "\n")
    D
  }

  generate_unique_fraction_design <- function(levels_per_factor,
                                              n_runs,
                                              max_iter = 200,
                                              a = 1, b = 1,
                                              verbose = TRUE) {
    full_lab <- expand.grid(levels_per_factor,
                            KEEP.OUT.ATTRS = FALSE,
                            stringsAsFactors = FALSE)
    N <- nrow(full_lab)
    if (n_runs > N) {
      stop("n_runs > total number of unique treatment combinations; cannot avoid repeats.")
    }

    enc <- encode_design(full_lab)
    full_int <- enc$D_int
    level_maps <- enc$level_maps

    sel <- seq_len(n_runs)
    current_D <- full_int[sel, , drop = FALSE]
    current_Z <- objective_Z(current_D, a, b)
    if (verbose) cat("Initial Z (unique subset) =", current_Z, "\n")

    for (it in seq_len(max_iter)) {
      improved <- FALSE
      for (i in seq_len(n_runs)) {
        available <- setdiff(seq_len(N), sel)
        best_local_Z <- current_Z
        best_cand <- NA_integer_
        for (cand in available) {
          new_sel <- sel
          new_sel[i] <- cand
          D_try <- full_int[new_sel, , drop = FALSE]
          z_try <- objective_Z(D_try, a, b)
          if (z_try + 1e-12 < best_local_Z) {
            best_local_Z <- z_try
            best_cand <- cand
          }
        }
        if (!is.na(best_cand)) {
          sel[i] <- best_cand
          current_D <- full_int[sel, , drop = FALSE]
          current_Z <- best_local_Z
          improved <- TRUE
        }
      }
      if (verbose) cat("Iter", it, "Z (unique subset) =", current_Z, "\n")
      if (!improved) break
    }
    if (verbose) cat("Final Z (unique subset) =", current_Z, "\n")

    D_opt <- full_int[sel, , drop = FALSE]
    attr(D_opt, "level_maps") <- level_maps
    D_opt
  }

  build_regular_design_from_generators <- function(levels_per_factor,
                                                   generators,
                                                   verbose = TRUE) {
    factor_names <- names(levels_per_factor)
    if (is.null(factor_names)) {
      stop("levels_per_factor must be a *named* list for generator-based designs.")
    }

    lev_counts <- sapply(levels_per_factor, length)
    uniq_s <- unique(lev_counts)
    if (length(uniq_s) != 1L) {
      if (verbose) {
        message("Generator engine requires all factors to have same number of levels s.")
      }
      return(NULL)
    }
    s <- uniq_s[1]
    if (s < 2L) stop("Number of levels s must be >= 2.")

    gens_list <- as.list(generators)
    gen_names <- names(gens_list)
    if (is.null(gen_names) || any(gen_names == "")) {
      stop("generators must be a *named* vector/list.")
    }
    if (!all(gen_names %in% factor_names)) {
      stop("All generator names must be among factor names.")
    }

    basic_factors <- setdiff(factor_names, gen_names)
    k <- length(factor_names)
    p <- length(gen_names)
    k_basic <- length(basic_factors)
    if (k_basic <= 0) stop("At least one basic factor is required.")

    n_reg <- s^k_basic

    if (verbose) {
      cat("Building regular ", s, "^(", k, "-", p, ") design with n_runs = ",
          n_reg, " from generators:\n", sep = "")
      print(generators)
      cat("Basic factors:", paste(basic_factors, collapse = ", "), "\n")
    }

    basic_grid <- expand.grid(
      rep(list(0:(s - 1L)), k_basic),
      KEEP.OUT.ATTRS = FALSE,
      stringsAsFactors = FALSE
    )
    colnames(basic_grid) <- basic_factors

    design_bin <- matrix(0L, nrow = n_reg, ncol = k)
    colnames(design_bin) <- factor_names
    for (bf in basic_factors) design_bin[, bf] <- basic_grid[[bf]]

    build_generated_column <- function(def, s, design_bin, basic_factors) {
      if (is.character(def)) {
        if (s != 2L) {
          stop("Character generator like 'AB' only for s=2.")
        }
        word <- def[1]
        letters <- strsplit(word, "", fixed = TRUE)[[1]]
        if (!all(letters %in% basic_factors)) {
          stop("Generator words must involve only basic factors.")
        }
        col_sum <- rep(0L, nrow(design_bin))
        for (lf in letters) col_sum <- (col_sum + design_bin[, lf]) %% s
        return(col_sum)
      } else if (is.numeric(def)) {
        if (is.null(names(def)) || any(names(def) == "")) {
          stop("Numeric generator definition must be named vector.")
        }
        vars <- names(def)
        if (!all(vars %in% basic_factors)) {
          stop("Numeric generator uses non-basic factor(s).")
        }
        col_sum <- rep(0L, nrow(design_bin))
        for (lf in vars) {
          coef <- def[[lf]] %% s
          col_sum <- (col_sum + coef * design_bin[, lf]) %% s
        }
        return(col_sum)
      } else {
        stop("Generator definition must be character (s=2) or numeric named vector.")
      }
    }

    for (g in gen_names) {
      def <- gens_list[[g]]
      design_bin[, g] <- build_generated_column(def, s, design_bin, basic_factors)
    }

    design_bin <- design_bin[, factor_names, drop = FALSE]
    D_int <- design_bin + 1L
    level_maps <- lapply(levels_per_factor, function(levs) as.character(levs))
    attr(D_int, "level_maps") <- level_maps
    if (verbose) {
      cat("Regular ", s, "^(", k, "-", p, ") design successfully constructed.\n", sep = "")
    }
    D_int
  }

  auto_regular_design_if_possible <- function(levels_per_factor,
                                              n_runs,
                                              a = 1, b = 1,
                                              max_sets = 200,
                                              verbose = TRUE) {
    factor_names <- names(levels_per_factor)
    if (is.null(factor_names)) {
      factor_names <- paste0("F", seq_along(levels_per_factor))
      names(levels_per_factor) <- factor_names
    }

    lev_counts <- sapply(levels_per_factor, length)
    uniq_s <- unique(lev_counts)

    if (length(uniq_s) != 1L) return(NULL)
    s <- uniq_s[1]
    if (s < 2L) return(NULL)

    k <- length(levels_per_factor)
    log_val <- log(n_runs, base = s)
    k_basic <- round(log_val)
    if (abs(log_val - k_basic) > 1e-8 || k_basic < 1L || k_basic >= k) return(NULL)

    if (verbose && FALSE) {
      cat("Auto-regular engine: equal-s design (s =", s,
          ", k =", k, ", n_runs =", n_runs, ")\n")
    }

    basic_factors <- factor_names[seq_len(k_basic)]
    gen_factors   <- factor_names[(k_basic + 1L):k]
    p <- length(gen_factors)
    if (p <= 0L) return(NULL)

    make_candidate_defs <- function(basic_factors, s, max_defs = 50) {
      defs <- list()
      for (bf in basic_factors) {
        for (coef in 1:(s - 1L)) {
          def <- c(coef); names(def) <- bf
          defs[[length(defs) + 1L]] <- def
          if (length(defs) >= max_defs) return(defs)
        }
      }
      if (length(basic_factors) >= 2L && length(defs) < max_defs) {
        pairs <- combn(basic_factors, 2L, simplify = FALSE)
        for (pr in pairs) {
          for (c1 in 1:(s - 1L)) {
            for (c2 in 1:(s - 1L)) {
              def <- c(c1, c2); names(def) <- pr
              defs[[length(defs) + 1L]] <- def
              if (length(defs) >= max_defs) return(defs)
            }
          }
        }
      }
      defs
    }

    cand_defs <- make_candidate_defs(basic_factors, s, max_defs = 50)
    if (length(cand_defs) == 0L) return(NULL)

    best_Z <- Inf
    best_design <- NULL
    set_count <- 0L
    idx <- rep(1L, p)
    more <- TRUE

    get_current_generators <- function() {
      gens <- vector("list", p)
      names(gens) <- gen_factors
      for (j in seq_len(p)) {
        gens[[j]] <- cand_defs[[idx[j]]]
      }
      gens
    }

    while (more && set_count < max_sets) {
      gens <- get_current_generators()
      design_try <- try(
        build_regular_design_from_generators(
          levels_per_factor = levels_per_factor,
          generators = gens,
          verbose = FALSE
        ),
        silent = TRUE
      )
      if (!inherits(design_try, "try-error") && !is.null(design_try)) {
        Z_val <- objective_Z(design_try, a = a, b = b)
        if (Z_val < best_Z - 1e-12) {
          best_Z <- Z_val
          best_design <- design_try
        }
        set_count <- set_count + 1L
      }
      for (pos in seq_len(p)) {
        if (idx[pos] < length(cand_defs)) {
          idx[pos] <- idx[pos] + 1L
          if (pos > 1L) idx[1:(pos - 1L)] <- 1L
          break
        } else if (pos == p) {
          more <- FALSE
        }
      }
    }

    if (verbose && FALSE) {
      cat("Auto-regular engine tested", set_count, "generator sets.\n")
    }
    best_design
  }

  generate_mixed_design_part1 <- function(levels_per_factor,
                                          n_runs,
                                          max_iter = 200,
                                          a = 1, b = 1,
                                          verbose = TRUE) {
    lev_counts <- sapply(levels_per_factor, length)
    full_size <- prod(lev_counts)

    auto_reg <- auto_regular_design_if_possible(
      levels_per_factor = levels_per_factor,
      n_runs = n_runs,
      a = a, b = b,
      max_sets = 200,
      verbose = FALSE
    )
    if (!is.null(auto_reg)) {
      return(auto_reg)
    }

    if (n_runs <= full_size) {
      D_opt <- generate_unique_fraction_design(
        levels_per_factor = levels_per_factor,
        n_runs = n_runs,
        max_iter = max_iter,
        a = a, b = b,
        verbose = FALSE
      )
    } else {
      D0 <- generate_NONBPA(levels_per_factor, n_runs)
      level_maps <- attr(D0, "level_maps")
      level_sets <- lapply(level_maps, function(v) seq_along(v))
      D_opt <- improve_design_exchange(D0, level_sets,
                                       max_iter = max_iter,
                                       a = a, b = b,
                                       verbose = FALSE)
      attr(D_opt, "level_maps") <- level_maps
    }
    D_opt
  }

  interaction_code <- function(D, cols) {
    sub <- D[, cols, drop = FALSE]
    lj <- sapply(cols, function(j) length(unique(D[, j])))
    bases <- c(1, cumprod(lj[-length(lj)]))
    codes <- integer(nrow(sub))
    for (r in seq_len(nrow(sub))) {
      x <- sub[r, ]
      codes[r] <- sum((x - 1L) * bases) + 1L
    }
    codes
  }

  compute_GBM <- function(D, max_t = 3) {
    k <- ncol(D)
    max_t <- min(max_t, k)
    H <- numeric(max_t)
    names(H) <- paste0("H", 1:max_t)
    for (t in 1:max_t) {
      combs <- combn(k, t, simplify = FALSE)
      Ht <- 0
      for (cols in combs) {
        code <- interaction_code(D, cols)
        n <- length(code)
        counts <- table(code)
        lt <- length(counts)
        Ttj <- n / lt
        Htj <- sum((counts - Ttj)^2)
        Ht <- Ht + Htj
      }
      H[t] <- Ht
    }
    H
  }

  gbm_resolution <- function(GBM_vec, tol = 1e-8) {
    idx <- which(GBM_vec > tol)
    if (length(idx) == 0) return(Inf)
    idx[1L]
  }

  build_model_matrix <- function(D, max_order = 3) {
    k <- ncol(D)
    max_order <- min(max_order, k)
    term_list <- list()
    term_names <- character(0)
    for (j in seq_len(k)) {
      term_list[[length(term_list) + 1L]] <- as.numeric(D[, j])
      term_names <- c(term_names, colnames(D)[j])
    }
    if (max_order >= 2) {
      for (order in 2:max_order) {
        if (order > k) break
        combs <- combn(k, order, simplify = FALSE)
        for (cols in combs) {
          term_list[[length(term_list) + 1L]] <- interaction_code(D, cols)
          term_names <- c(term_names,
                          paste(colnames(D)[cols], collapse = ":"))
        }
      }
    }
    M <- do.call(cbind, term_list)
    colnames(M) <- term_names
    M
  }

  term_order <- function(term_name) {
    length(strsplit(term_name, ":", fixed = TRUE)[[1]])
  }

  compute_correlation_matrix <- function(M) {
    M_centered <- scale(M, center = TRUE, scale = TRUE)
    suppressWarnings(cor(M_centered))
  }

  build_alias_chains <- function(M) {
    C <- compute_correlation_matrix(M)
    terms <- colnames(M)
    remaining <- terms
    chains <- list()
    while (length(remaining) > 0) {
      if (length(remaining) == 1L) {
        t <- remaining[1L]
        chains[[t]] <- list(
          lead = t,
          followers = character(0),
          corrs = numeric(0)
        )
        break
      }
      subC <- C[remaining, remaining, drop = FALSE]
      diag(subC) <- 0
      absC <- abs(subC)
      max_val <- max(absC, na.rm = TRUE)
      if (is.finite(max_val) && max_val < 1e-8) {
        for (t in remaining) {
          chains[[t]] <- list(
            lead = t,
            followers = character(0),
            corrs = numeric(0)
          )
        }
        break
      }
      idx <- which(absC == max_val, arr.ind = TRUE)[1, ]
      t1 <- rownames(absC)[idx[1]]
      t2 <- colnames(absC)[idx[2]]
      r12 <- C[t1, t2]
      o1 <- term_order(t1)
      o2 <- term_order(t2)
      if (o1 < o2) {
        lead <- t1; foll <- t2
      } else if (o2 < o1) {
        lead <- t2; foll <- t1; r12 <- C[lead, foll]
      } else {
        lead <- t1; foll <- t2
      }
      chains[[lead]] <- list(
        lead = lead,
        followers = foll,
        corrs = r12
      )
      remaining <- setdiff(remaining, c(lead, foll))
    }
    chains
  }

  print_alias_chains <- function(chains, digits = 3) {
    for (nm in names(chains)) {
      ch <- chains[[nm]]
      lead <- ch$lead
      followers <- ch$followers
      corrs <- ch$corrs
      if (length(followers) == 0L) {
        cat(sprintf("[%s] = %s\n", lead, lead))
      } else {
        rhs <- paste(sprintf(paste0("%.", digits, "f %s"), corrs, followers),
                     collapse = " + ")
        cat(sprintf("[%s] = %s + %s\n", lead, lead, rhs))
      }
    }
  }

  alias_summary_table <- function(C, min_abs_corr = 0.7) {
    terms <- colnames(C)
    res <- list()
    k <- length(terms)
    idx <- 1L
    for (i in 1:(k - 1L)) {
      for (j in (i + 1L):k) {
        r <- C[i, j]
        if (is.na(r)) next
        if (abs(r) >= min_abs_corr) {
          t1 <- terms[i]; t2 <- terms[j]
          res[[idx]] <- data.frame(
            term1 = t1,
            term2 = t2,
            order1 = term_order(t1),
            order2 = term_order(t2),
            corr = r,
            stringsAsFactors = FALSE
          )
          idx <- idx + 1L
        }
      }
    }
    if (length(res) == 0) {
      return(data.frame(
        term1 = character(0),
        term2 = character(0),
        order1 = integer(0),
        order2 = integer(0),
        corr = numeric(0),
        stringsAsFactors = FALSE
      ))
    }
    out <- do.call(rbind, res)
    out[order(-abs(out$corr)), ]
  }

  make_levels_list <- function(levels_spec) {
    if (is.list(levels_spec) && !is.null(names(levels_spec))) {
      return(levels_spec)
    }
    if (is.numeric(levels_spec) && is.null(names(levels_spec))) {
      m <- length(levels_spec)
      fac_names <- LETTERS[seq_len(m)]
      out <- vector("list", m)
      names(out) <- fac_names
      for (i in seq_len(m)) out[[i]] <- seq_len(levels_spec[i])
      return(out)
    }
    stop("levels_spec must be named list of levels or numeric vector of levels.")
  }

  build_linear_contrasts <- function(design) {
    design <- as.data.frame(design, stringsAsFactors = FALSE)
    n <- nrow(design)
    k <- ncol(design)
    X <- matrix(NA_real_, nrow = n, ncol = k)
    colnames(X) <- colnames(design)
    for (j in seq_len(k)) {
      f <- as.factor(design[[j]])
      levs <- levels(f)
      nlev <- length(levs)
      if (nlev < 2L) {
        X[, j] <- 0
      } else {
        Cmat <- contr.poly(nlev)
        lin <- Cmat[, 1]
        names(lin) <- levs
        X[, j] <- lin[as.character(f)]
      }
    }
    X
  }

  trend_objective <- function(X_lin, ord, degree = 1L) {
    n <- nrow(X_lin)
    Xo <- X_lin[ord, , drop = FALSE]
    t <- seq_len(n)
    T_basis <- sapply(1:degree, function(d) t^d)
    if (degree == 1L) T_basis <- matrix(T_basis, ncol = 1)
    T_basis <- scale(T_basis, center = TRUE, scale = TRUE)
    total <- 0
    for (j in seq_len(ncol(Xo))) {
      xj <- scale(Xo[, j], center = TRUE, scale = TRUE)
      for (d in seq_len(ncol(T_basis))) {
        r <- suppressWarnings(cor(xj, T_basis[, d]))
        if (is.na(r)) next
        total <- total + r^2
      }
    }
    total
  }

  trend_free_order <- function(design,
                               degree = 1,
                               max_passes = 50,
                               tol_improve = 1e-12,
                               target_obj = 1e-12,
                               verbose = TRUE) {
    n <- nrow(design)
    if (n < 2L) stop("Design must have at least 2 runs.")
    X_lin <- build_linear_contrasts(design)
    t <- seq_len(n)
    T_basis <- sapply(1:degree, function(d) t^d)
    if (degree == 1L) T_basis <- matrix(T_basis, ncol = 1)
    T_basis <- scale(T_basis, center = TRUE, scale = TRUE)
    ord <- seq_len(n)
    best_obj <- trend_objective(X_lin, ord, degree)
    if (verbose) cat("\nInitial trend objective =", best_obj, "\n")
    for (pass in seq_len(max_passes)) {
      improved <- FALSE
      best_swap_obj <- best_obj
      best_i <- NA_integer_; best_j <- NA_integer_
      for (i in 1:(n - 1L)) {
        for (j in (i + 1L):n) {
          cand_ord <- ord
          tmp <- cand_ord[i]; cand_ord[i] <- cand_ord[j]; cand_ord[j] <- tmp
          cand_obj <- trend_objective(X_lin, cand_ord, degree)
          if (cand_obj + tol_improve < best_swap_obj) {
            best_swap_obj <- cand_obj; best_i <- i; best_j <- j; improved <- TRUE
          }
        }
      }
      if (!improved) {
        if (verbose) cat("No improving swap found on pass", pass, "\n")
        break
      }
      tmp <- ord[best_i]
      ord[best_i] <- ord[best_j]
      ord[best_j] <- tmp
      best_obj <- best_swap_obj
      if (verbose) {
        cat(sprintf("Pass %d: improved trend objective = %.12g\n",
                    pass, best_obj))
      }
      if (best_obj < target_obj) {
        if (verbose) cat("Target objective reached; stopping.\n")
        break
      }
    }
    if (verbose) cat("Final deterministic trend objective =", best_obj, "\n")
    list(
      order = ord,
      design_reordered = design[ord, , drop = FALSE],
      objective = best_obj
    )
  }

  ff_core_compute <- function(levels_spec,
                              n_runs,
                              max_iter = 100,
                              a = 1, b = 1,
                              max_int_order = 3,
                              alias_min_abs_corr = 0.9) {
    levels_per_factor <- make_levels_list(levels_spec)
    D_int <- generate_mixed_design_part1(
      levels_per_factor = levels_per_factor,
      n_runs = n_runs,
      max_iter = max_iter,
      a = a, b = b,
      verbose = FALSE
    )
    if (is.null(D_int)) stop("ERROR: generate_mixed_design_part1() did not return a design.")
    level_maps <- attr(D_int, "level_maps")
    if (is.null(level_maps)) stop("ERROR: level_maps attribute missing from design.")
    D_labels <- decode_design(D_int, level_maps)
    Hhat <- balance_Hhat(D_int)
    J2   <- J2_hat(D_int)
    GBM  <- compute_GBM(D_int, max_t = max_int_order)
    reso <- gbm_resolution(GBM)
    M         <- build_model_matrix(D_int, max_order = max_int_order)
    C         <- compute_correlation_matrix(M)
    chains    <- build_alias_chains(M)
    alias_tbl <- alias_summary_table(C, min_abs_corr = alias_min_abs_corr)
    list(
      levels_spec        = levels_spec,
      n_runs             = n_runs,
      max_iter           = max_iter,
      a                  = a,
      b                  = b,
      max_int_order      = max_int_order,
      alias_min_abs_corr = alias_min_abs_corr,
      design_int         = D_int,
      design_labels      = D_labels,
      level_maps         = level_maps,
      Hhat               = Hhat,
      J2hat              = J2,
      GBM                = GBM,
      gbm_resolution     = reso,
      model_matrix       = M,
      corr_matrix        = C,
      alias_chains       = chains,
      alias_summary      = alias_tbl
    )
  }

  ff_print_part1_design <- function(core, verbose = TRUE) {
    if (!verbose) return(invisible(NULL))
    D_int <- core$design_int
    cat("\n=== PART 1: MIXED / REGULAR FRACTIONAL DESIGN ===\n")
    cat("\n=== DESIGN ===\n")
    D_print <- D_int
    attr(D_print, "level_maps") <- NULL
    print(D_print)
    invisible(NULL)
  }

  ff_print_part2_properties <- function(core, verbose = TRUE) {
    if (!verbose) return(invisible(NULL))
    D_int              <- core$design_int
    Hhat               <- core$Hhat
    J2                 <- core$J2hat
    GBM                <- core$GBM
    reso               <- core$gbm_resolution
    chains             <- core$alias_chains
    alias_tbl          <- core$alias_summary
    max_int_order      <- core$max_int_order
    alias_min_abs_corr <- core$alias_min_abs_corr

    cat("\n=== PART 2:  PROPERTIES OF MIXED / REGULAR FRACTIONAL DESIGN ===\n")
    cat("\n=== LEVEL BALANCE PER FACTOR ===\n")
    for (j in seq_len(ncol(D_int))) {
      cat("Factor", colnames(D_int)[j], ":\n")
      print(table(D_int[, j]))
    }

    cat("\n=== METRICS ===\n")
    cat("H^ (main effects) =", Hhat, "\n")
    cat("J2^               =", J2,   "\n")
    cat("GBM (H1..H", max_int_order, ") = ",
        paste(GBM, collapse = ", "), "\n", sep = "")
    cat("GBM resolution (smallest t with Ht>0) =", reso, "\n")

    cat("\n=== ALIAS CHAINS ===\n")
    print_alias_chains(chains)

    cat("\n=== STRONG CONFOUNDING SUMMARY (|r| >=", alias_min_abs_corr, ") ===\n")
    if (nrow(alias_tbl) == 0) {
      cat("No pairs with |correlation| >=", alias_min_abs_corr, "\n")
    } else {
      print(alias_tbl, row.names = FALSE)
    }
    invisible(NULL)
  }

  ff_print_part3_trend <- function(core,
                                   degree = 1,
                                   verbose = TRUE) {
    if (!verbose) {
      return(trend_free_order(
        design  = core$design_labels,
        degree  = degree,
        verbose = FALSE
      ))
    }
    cat("\n=== PART 3: TREND-FREE RUN ORDER ===\n")
    tf_res <- trend_free_order(
      design  = core$design_labels,
      degree  = degree,
      verbose = TRUE
    )
    cat("\nTrend-free run order (indices in original design):\n")
    print(tf_res$order)
    cat("\nTrend-free design (first 100 runs or fewer):\n")
    print(head(tf_res$design_reordered,
               min(100, nrow(tf_res$design_reordered))))
    invisible(tf_res)
  }

  core <- ff_core_compute(
    levels_spec        = levels_spec,
    n_runs             = n_runs,
    max_iter           = max_iter,
    a                  = a,
    b                  = b,
    max_int_order      = max_int_order,
    alias_min_abs_corr = alias_min_abs_corr
  )

  # PART 1
  if (1 %in% parts) {
    ff_print_part1_design(core, verbose = verbose)
  }

  # PART 2
  if (2 %in% parts) {
    ff_print_part2_properties(core, verbose = verbose)
  }

  # PART 3
  tf_res <- NULL
  if (3 %in% parts) {
    if (!tf) stop("Part 3 requested but tf = FALSE. Use tf = TRUE to enable Part 3.")
    tf_res <- ff_print_part3_trend(core,
                                   degree  = 1,
                                   verbose = verbose)
  }

  core$trend_free_result <- tf_res
  invisible(core)
}
