# Visualization functions for stable distribution analysis

# =============================================================================
# DISTRIBUTION COMPARISON PLOTS
# =============================================================================

#' Plot histogram with normal and stable PDF overlays
#'
#' Displays a histogram of the data with overlaid normal and alpha-stable probability density functions.
#'
#' @param x Numeric vector of data.
#' @param params_stable List with alpha, beta, gamma, delta.
#' @importFrom graphics hist lines legend grid
#' @importFrom stats dnorm
#' @return NULL (plot is displayed as a side effect)
#' @export
plot_distributions <- function(x, params_stable) {
  if (!is.numeric(x) || length(x) < 2) {
    stop("Input x must be a numeric vector of length >= 2.")
  }

  xx <- seq(min(x), max(x), length.out = 500)
  norm_pdf <- stats::dnorm(xx, mean(x, na.rm = TRUE), sd(x, na.rm = TRUE))

  stable_pdf <- tryCatch({
    dstable(xx,
            alpha = params_stable$alpha,
            beta  = params_stable$beta,
            gamma = params_stable$gamma,
            delta = params_stable$delta,
            pm = 1)
  }, error = function(e) {
    message("[Warning] Stable PDF computation failed; using normal PDF instead.")
    norm_pdf
  })

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  hist(x, breaks = 50, freq = FALSE, col = rgb(0.7, 0.7, 0.7, 0.5),
       main = "Density Comparison",
       xlab = "x", ylab = "Density")
  lines(xx, norm_pdf, col = "red", lty = 2, lwd = 2)
  lines(xx, stable_pdf, col = "blue", lwd = 2)

  legend("topright",
         legend = c("Data", "Normal PDF", "Stable PDF"),
         col = c("gray", "red", "blue"),
         lty = c(0, 2, 1),   # 0 = no line for histogram
         lwd = c(NA, 2, 2),
         pch = c(15, NA, NA),
         pt.cex = 1.5)

  grid()
  invisible(NULL)
}

# =============================================================================
# MIXTURE VISUALIZATION FUNCTIONS
# =============================================================================


#' Plot mixture of two alpha-stable distributions
#'
#' Plots the histogram of data and overlays the estimated mixture density and its components.
#'
#' @param data Numeric vector of observations.
#' @param params1 Vector of parameters for component 1 (alpha, beta, gamma, delta).
#' @param params2 Vector of parameters for component 2.
#' @param w Mixing weight for component 1.
#' @param label Optional label for output file.
#' @importFrom graphics hist lines legend grid
#' @importFrom grDevices png
#' @return NULL (plot is saved as PNG)
#' @export
plot_mixture <- function(data, params1, params2, w, label = "EM") {
  x <- seq(min(data), max(data), length.out = 1000)
  pdf1 <- r_stable_pdf(x, params1[1], params1[2], params1[3], params1[4])
  pdf2 <- r_stable_pdf(x, params2[1], params2[2], params2[3], params2[4])
  mix <- w * pdf1 + (1 - w) * pdf2

  png(file.path(tempdir(), paste0("mixture_alpha_stable_", label, ".png")),
      width = 10, height = 5, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)

  hist(data, breaks = 40, freq = FALSE, col = rgb(0.7, 0.7, 0.7, 0.4),
       main = paste("Mixture of Alpha-Stable Distributions (", label, ")", sep = ""),
       xlab = "x", ylab = "Density")
  lines(x, mix, col = "blue", lwd = 2)
  lines(x, w * pdf1, col = "green", lty = 2)
  lines(x, (1 - w) * pdf2, col = "orange", lty = 2)
  legend("topright",
         legend = c("Data", paste(label, "Fit"), paste(label, "Comp 1"), paste(label, "Comp 2")),
         col = c("gray", "blue", "green", "orange"),
         lty = c(0, 1, 2, 2), lwd = c(NA, 2, 1, 1), pch = c(15, NA, NA, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}


#' Plot final fitted mixture of alpha-stable distributions
#'
#' Plots the final fitted mixture PDF over the data histogram.
#'
#' @param data Numeric vector of observations.
#' @param p1 Vector of parameters for component 1.
#' @param p2 Vector of parameters for component 2.
#' @param w Mixing weight for component 1.
#' @importFrom graphics hist lines legend grid
#' @importFrom grDevices png
#' @return NULL (plot is saved as PNG)
#' @export
plot_final_mixture_fit <- function(data, p1, p2, w) {
  x_vals <- seq(min(data), max(data), length.out = 1000)
  y_vals <- tryCatch({
    mixture_stable_pdf(x_vals, p1, p2, w)
  }, error = function(e) rep(NA, length(x_vals)))

  png(file.path(tempdir(), "mixture_alpha_stable_fit_final.png"), width = 9, height = 5, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)

  hist(data, breaks = 40, freq = FALSE, col = rgb(0.7, 0.7, 0.7, 0.6),
       main = "Fitted Mixture of Alpha-Stable Distributions", xlab = "x", ylab = "Density")
  lines(x_vals, y_vals, col = "red", lwd = 2)
  legend("topright", legend = c("Data", "Fitted Mixture"),
         col = c("gray", "red"), lty = c(0, 1), lwd = c(NA, 2), pch = c(15, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}


# =============================================================================
# METHOD COMPARISON AND MCMC DIAGNOSTICS
# =============================================================================


#' Plot RMSE and Log-Likelihood comparison across methods
#'
#' Generates bar plots comparing RMSE and log-likelihood values across different estimation methods.
#'
#' @param scores Named list of score objects with RMSE and LogLikelihood.
#' @importFrom graphics barplot grid par
#' @importFrom grDevices rainbow png
#' @return NULL (plot is saved as PNG)
#' @export
plot_method_comparison <- function(scores) {
  methods <- names(scores)
  rmse_vals <- sapply(scores, function(x) x$RMSE)
  ll_vals <- sapply(scores, function(x) x$LogLikelihood)

  png(file.path(tempdir(), "method_comparison.png"), width = 12, height = 4, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)
  par(mfrow = c(1, 2))

  barplot(rmse_vals, names.arg = methods, main = "RMSE per Method",
          ylab = "RMSE", col = rainbow(length(methods)), las = 2)
  grid()
  barplot(ll_vals, names.arg = methods, main = "LogLikelihood per Method",
          ylab = "LogLikelihood", col = rainbow(length(methods)), las = 2)
  grid()
  invisible(NULL)
}



#' Plot trace of a parameter across MCMC iterations
#'
#' Displays the evolution of a parameter across MCMC iterations to assess convergence.
#'
#' @param samples List of sample objects from MCMC.
#' @param param_name Name of the parameter to trace.
#' @importFrom graphics plot grid
#' @return NULL (plot is displayed as a side effect)
#' @export
plot_trace <- function(samples, param_name) {
  if (length(samples) < 1 || !(param_name %in% names(samples[[1]]))) {
    message("Parameter not found in samples.")
    return(invisible(NULL))
  }
  values <- sapply(samples, function(x) x[[param_name]])
  plot(values, type = "l", main = paste("Trace plot for", param_name),
       xlab = "Iteration", ylab = param_name)
  grid()
  invisible(NULL)
}


#' Plot posterior mixture density from MCMC samples
#'
#' Visualizes the estimated two-component alpha-stable mixture density
#' using parameters obtained from MCMC sampling. Optionally overlays
#' the true density for comparison.
#'
#' @param M2_w1 Numeric scalar. Mixture weight of first component.
#' @param M2_alpha1 Numeric vector. Stability parameter samples of first component.
#' @param M2_beta1 Numeric vector. Skewness parameter samples of first component.
#' @param M2_delta1 Numeric vector. Location parameter samples of first component.
#' @param M2_omega1 Numeric vector. Scale parameter samples of first component.
#' @param M2_w2 Numeric scalar. Mixture weight of second component.
#' @param M2_alpha2 Numeric vector. Stability parameter samples of second component.
#' @param M2_beta2 Numeric vector. Skewness parameter samples of second component.
#' @param M2_delta2 Numeric vector. Location parameter samples of second component.
#' @param M2_omega2 Numeric vector. Scale parameter samples of second component.
#' @param xx Numeric vector of grid values for density evaluation.
#' @param xx_true Optional numeric vector for true density x-values.
#' @param yy_true Optional numeric vector for true density y-values.
#' @return Invisibly returns the file path to the saved PNG image of the posterior mixture density plot.
#' @importFrom graphics plot lines legend grid
#' @importFrom grDevices png
#' @export
plot_results <- function(M2_w1, M2_alpha1, M2_beta1, M2_delta1, M2_omega1,
                         M2_w2, M2_alpha2, M2_beta2, M2_delta2, M2_omega2,
                         xx, xx_true = NULL, yy_true = NULL) {
  burn_in <- 100
  idx <- if (length(M2_alpha1) > burn_in + 200) {
    (burn_in + 1):(burn_in + 201)
  } else {
    (burn_in + 1):length(M2_alpha1)
  }

  params1 <- c(mean(M2_alpha1[idx]), mean(M2_beta1[idx]),
               mean(M2_delta1[idx]), mean(M2_omega1[idx]))
  params2 <- c(mean(M2_alpha2[idx]), mean(M2_beta2[idx]),
               mean(M2_delta2[idx]), mean(M2_omega2[idx]))

  pdf1 <- tryCatch(r_stable_pdf(xx, params1[1], params1[2], params1[3], params1[4]),
                   error = function(e) rep(0, length(xx)))
  pdf2 <- tryCatch(r_stable_pdf(xx, params2[1], params2[2], params2[3], params2[4]),
                   error = function(e) rep(0, length(xx)))
  yy <- mean(M2_w1[idx]) * pdf1 + (1 - mean(M2_w1[idx])) * pdf2

  png(file.path(tempdir(), "bayesian_stable_mixture_final.png"), width = 10, height = 6, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)

  plot(xx, yy, type = "l", col = "red", lty = 2, lwd = 2,
       main = "Mixture of Alpha-Stable Distributions - Metropolis-Hastings",
       xlab = "x", ylab = "Density")
  if (!is.null(xx_true) && !is.null(yy_true)) {
    lines(xx_true, yy_true, col = "black", lwd = 2)
    legend("topright", legend = c("Estimate", "Truth"), col = c("red", "black"),
           lty = c(2, 1), lwd = c(2, 2))
  } else {
    legend("topright", legend = "Estimate", col = "red", lty = 2, lwd = 2)
  }
  grid()
  invisible(NULL)
}



# =============================================================================
# MIXTURE COMPARISON AND COMPONENT VISUALIZATION
# =============================================================================


#' Compare EM-estimated mixture with a non-optimized reference model
#'
#' Plots the fitted EM mixture density alongside a manually defined reference mixture
#' to visually compare estimation quality.
#'
#' @param data Numeric vector of observations.
#' @param p1 Vector of parameters for component 1 (alpha, beta, gamma, delta).
#' @param p2 Vector of parameters for component 2.
#' @param w Mixing weight for component 1.
#' @return Invisibly returns the file path to the saved PNG image of the comparison plot.
#' @importFrom graphics hist lines legend grid
#' @importFrom grDevices png
#' @export
plot_comparison <- function(data, p1, p2, w) {
  x <- seq(min(data), max(data), length.out = 1000)
  pdf1 <- pmax(r_stable_pdf(x, p1[1], p1[2], p1[3], p1[4]), 1e-300)
  pdf2 <- pmax(r_stable_pdf(x, p2[1], p2[2], p2[3], p2[4]), 1e-300)
  y_em <- w * pdf1 + (1 - w) * pdf2

  # Reference model
  params_ref1 <- c(1.5, 0, 1, -1)
  params_ref2 <- c(1.7, 0.5, 2, 6)
  y_bad <- 0.5 * pmax(r_stable_pdf(x, params_ref1[1], params_ref1[2], params_ref1[3], params_ref1[4]), 1e-300) +
    0.5 * pmax(r_stable_pdf(x, params_ref2[1], params_ref2[2], params_ref2[3], params_ref2[4]), 1e-300)

  png(file.path(tempdir(), "mixture_alpha_stable_comparison.png"), width = 10, height = 6, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)

  hist(data, breaks = 40, freq = FALSE, col = rgb(0.5, 0.5, 0.5, 0.5),
       main = "Alpha-Stable Mixture: EM Fit vs Non-Optimal", xlab = "x", ylab = "Density")
  lines(x, y_em, col = "green", lwd = 2)
  lines(x, y_bad, col = "red", lty = 2, lwd = 2)
  legend("topright", legend = c("Data", "EM Estimated", "Non-Optimal"),
         col = c("gray", "green", "red"), lty = c(0, 1, 2), lwd = c(NA, 2, 2),
         pch = c(15, NA, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}


#' Plot mixture fit with individual components
#'
#' Displays the estimated mixture density and optionally its individual components
#' over a histogram of the data.
#'
#' @param data Numeric vector of observations.
#' @param estimated_params List with weights, alphas, betas, gammas, deltas.
#' @param bins Number of histogram bins.
#' @param plot_components Logical, whether to show individual components.
#' @param save_path Optional PNG filename.
#' @param show_plot Logical, whether to display the plot.
#' @param title Plot title.
#' @return Invisibly returns the file path to the saved plot (if \code{save_path} is provided), or \code{NULL} otherwise.
#' @importFrom graphics hist lines legend grid par
#' @importFrom grDevices rainbow png
#' @export
plot_mixture_fit <- function(data, estimated_params, bins = 200,
                             plot_components = TRUE, save_path = NULL,
                             show_plot = TRUE,
                             title = "Mixture of Alpha-Stable Distributions") {
  x <- seq(min(data), max(data), length.out = 1000)
  pdf_mix <- rep(0, length(x))
  num_components <- length(estimated_params$weights)
  colors <- rainbow(num_components)

  if (!is.null(save_path)) {
    save_path <- file.path(tempdir(), save_path)
    png(save_path, width = 10, height = 5, units = "in", res = 150)
    oldpar <- par(no.readonly = TRUE)
    on.exit({ par(oldpar); dev.off() }, add = TRUE)
  } else {
    oldpar <- par(no.readonly = TRUE)
    on.exit(par(oldpar), add = TRUE)
  }

  hist(data, breaks = bins, freq = FALSE, col = rgb(0.4, 0.6, 1, 0.5),
       main = title, xlab = "x", ylab = "Density")

  for (i in 1:num_components) {
    w <- estimated_params$weights[i]
    a <- estimated_params$alphas[i]
    b <- estimated_params$betas[i]
    g <- estimated_params$gammas[i]
    d <- estimated_params$deltas[i]

    if (!any(is.na(c(a,b,g,d)))) {
      pdf_i <- r_stable_pdf(x, a, b, g, d)
      pdf_mix <- pdf_mix + w * pdf_i
      if (plot_components) lines(x, w * pdf_i, lty = 2, lwd = 1.5, col = colors[i])
    }
  }

  lines(x, pdf_mix, col = "red", lwd = 2)

  legend_items <- c("Observed Data")
  legend_colors <- c("cornflowerblue")
  legend_lty <- c(0)
  legend_lwd <- c(NA)
  legend_pch <- c(15)

  if (plot_components) {
    for (i in 1:num_components) {
      legend_items <- c(legend_items, paste("Component", i))
      legend_colors <- c(legend_colors, colors[i])
      legend_lty <- c(legend_lty, 2)
      legend_lwd <- c(legend_lwd, 1.5)
      legend_pch <- c(legend_pch, NA)
    }
  }

  legend_items <- c(legend_items, "Estimated Mixture")
  legend_colors <- c(legend_colors, "red")
  legend_lty <- c(legend_lty, 1)
  legend_lwd <- c(legend_lwd, 2)
  legend_pch <- c(legend_pch, NA)

  legend("topright", legend = legend_items, col = legend_colors,
         lty = legend_lty, lwd = legend_lwd, pch = legend_pch, pt.cex = 1.5)
  grid()
  invisible(NULL)
}


# =============================================================================
# REAL DATA VISUALIZATION AND EPIDEMIOLOGICAL ANALYSIS
# =============================================================================


#' Plot fitted mixture on real dataset
#'
#' Plots the fitted mixture model over a histogram of real-world data,
#' showing both components and the overall mixture.
#'
#' @param X Numeric vector of observations.
#' @param result List containing params1, params2, and lambda1.
#' @importFrom graphics hist lines legend grid par
#' @importFrom grDevices png
#' @return NULL (plot is saved as PNG)
#' @export
plot_real_mixture_fit <- function(X, result) {
  X_range <- seq(min(X), max(X), length.out = 500)

  pdf1 <- tryCatch({
    dstable(X_range, result$params1$alpha, result$params1$beta,
            gamma = result$params1$gamma, delta = result$params1$delta)
  }, error = function(e) dnorm(X_range, result$params1$delta, result$params1$gamma))

  pdf2 <- tryCatch({
    dstable(X_range, result$params2$alpha, result$params2$beta,
            gamma = result$params2$gamma, delta = result$params2$delta)
  }, error = function(e) dnorm(X_range, result$params2$delta, result$params2$gamma))

  mixture <- result$lambda1 * pdf1 + (1 - result$lambda1) * pdf2

  png(file.path(tempdir(), "real_mixture_fit.png"), width = 10, height = 5, units = "in", res = 150)
  oldpar <- par(no.readonly = TRUE)
  on.exit({ par(oldpar); dev.off() }, add = TRUE)

  hist(X, breaks = 40, freq = FALSE, col = "lightgray",
       main = "Real Dataset: Mixture of Two Alpha-Stable Distributions",
       xlab = "x", ylab = "Density")
  lines(X_range, pdf1, col = "blue", lty = 2, lwd = 2)
  lines(X_range, pdf2, col = "green", lty = 2, lwd = 2)
  lines(X_range, mixture, col = "black", lwd = 2)
  legend("topright", legend = c("Data", "Component 1", "Component 2", "Mixture Fit"),
         col = c("lightgray", "blue", "green", "black"),
         lty = c(0, 2, 2, 1), lwd = c(NA, 2, 2, 2),
         pch = c(15, NA, NA, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}


#' Plot effective reproduction number (Re) over time
#'
#' Computes and visualizes the effective reproduction number (Rt) over time
#' using incidence data and generation time distribution. Optionally overlays
#' a smoothed spline curve.
#'
#' @param GT Numeric vector. Generation time distribution.
#' @param S Numeric vector. Secondary cases.
#' @param inc Numeric vector. Incidence time series.
#' @param dates Date vector corresponding to observations.
#' @param est_r0_ml Function to estimate R0.
#' @param RT Function to compute Rt.
#' @param output_file Output PDF filename.
#' @return NULL (plot is saved as PDF)
#' @importFrom graphics plot polygon grid par
#' @importFrom stats splinefun
#' @importFrom grDevices pdf
#' @export
plot_effective_reproduction_number <- function(GT, S, inc, dates,
                                               est_r0_ml, RT,
                                               output_file = "RN_avec_dates_EM-ML.pdf") {
  min_len <- min(length(GT), length(S))
  GT <- GT[1:min_len]
  S <- S[1:min_len]

  est_r0_ml_empirical <- est_r0_ml(GT, S)
  message(sprintf("Estimated R0 (Empirical): %.4f", est_r0_ml_empirical))

  Rt <- RT(inc, GT)
  rt_df <- data.frame(Date = dates, Rt = Rt)
  rt_df <- rt_df[is.finite(rt_df$Rt), ]
  date_numeric <- as.numeric(difftime(rt_df$Date, min(rt_df$Date), units = "days"))

  if (length(date_numeric) >= 4) {
    spline_fit <- splinefun(date_numeric, rt_df$Rt, method = "natural")
    date_range_numeric <- seq(min(date_numeric), max(date_numeric), length.out = 500)
    Rt_smooth <- spline_fit(date_range_numeric)
    date_range <- min(rt_df$Date) + date_range_numeric

    pdf(file.path(tempdir(), output_file), width = 10, height = 6)
    oldpar <- par(no.readonly = TRUE)
    on.exit({ par(oldpar); dev.off() }, add = TRUE)

    plot(date_range, Rt_smooth, type = "l", col = "black", lwd = 2,
         main = "Effective Reproduction Number over Time",
         xlab = "Time", ylab = "Effective Reproduction Number")
    polygon(c(date_range, rev(date_range)),
            c(Rt_smooth, rep(min(Rt_smooth), length(Rt_smooth))),
            col = rgb(1, 0, 0, 0.2), border = NA)
    grid()
    message(sprintf("Plot saved as %s", output_file))
  } else {
    message("Not enough data points for cubic spline interpolation.")
  }
  invisible(NULL)
}


# =============================================================================
# TRUE VS ESTIMATED DENSITY COMPARISON
# =============================================================================


#' Plot true vs estimated mixture density
#'
#' Compares the true mixture density with the estimated one by overlaying both
#' on the histogram of the observed data.
#'
#' @param true_params List of true mixture components with alpha, beta, gamma, delta, pi.
#' @param est_params List of estimated mixture components.
#' @param data Numeric vector of observations.
#' @param bins Number of histogram bins.
#' @return NULL (plot is displayed)
#' @importFrom graphics hist lines legend grid par
#' @export
plot_fit_vs_true <- function(true_params, est_params, data, bins = 100) {
  x <- seq(min(data), max(data), length.out = 1000)

  mixture_pdf <- function(x, params) {
    y <- rep(0, length(x))
    for (p in params) {
      y <- y + p$pi * dstable(x, alpha = p$alpha, beta = p$beta,
                              gamma = p$gamma, delta = p$delta, pm = 1)
    }
    y
  }

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  hist(data, breaks = bins, freq = FALSE, col = rgb(0, 0, 0, 0.4),
       xlab = "", ylab = "", main = "True vs Estimated Mixture Density")
  lines(x, mixture_pdf(x, true_params), col = "green", lwd = 2, lty = 2)
  lines(x, mixture_pdf(x, est_params), col = "red", lwd = 2)
  legend("topright", legend = c("Data", "True PDF", "Estimated PDF"),
         col = c("black", "green", "red"), lwd = c(1, 2, 2),
         lty = c(NA, 2, 1), pch = c(15, NA, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}

#' Compare estimated mixture densities from two methods against the true density
#'
#' Plots the true mixture density and compares it with two estimated densities
#' obtained from different methods (e.g., MLE vs ECF).
#'
#' @param data Numeric vector of observations.
#' @param true_params List of true mixture components.
#' @param method1 First estimation method ("MLE", "ECF", etc.).
#' @param method2 Second estimation method.
#' @param bins Number of histogram bins.
#' @return NULL (plot is displayed)
#' @importFrom graphics hist lines legend grid par
#' @export
plot_fit_vs_true_methods <- function(data, true_params,
                                     method1 = "MLE", method2 = "ECF",
                                     bins = 100) {

  ensure_list_of_dicts <- function(params) {
    if (is.list(params) && !is.null(params$alpha)) return(list(params))
    if (is.list(params) && all(sapply(params, is.list))) return(params)
    stop("Params must be a named list or list of named lists.")
  }

  true_params <- ensure_list_of_dicts(true_params)
  x <- seq(min(data), max(data), length.out = 1000)

  mixture_pdf <- function(x, params) {
    params <- ensure_list_of_dicts(params)
    y <- rep(0, length(x))
    for (p in params) {
      if (is.null(p$pi)) p$pi <- 1.0
      alpha <- as.numeric(p$alpha); beta <- as.numeric(p$beta)
      gamma <- as.numeric(p$gamma); delta <- as.numeric(p$delta); pi <- as.numeric(p$pi)
      if (any(is.na(c(alpha, beta, gamma, delta, pi)))) {
        message("Skipping component due to NA parameter.")
        next
      }
      y <- y + pi * dstable(x, alpha = alpha, beta = beta, gamma = gamma, delta = delta, pm = 1)
    }
    y
  }

  est1 <- switch(method1,
                 "MLE" = mle_estimate(data),
                 "ECF" = estimate_stable_kernel_ecf(data, seq(0.1, 1, length.out = 20)),
                 stop(paste("Unknown method:", method1)))
  est2 <- switch(method2,
                 "MLE" = mle_estimate(data),
                 "ECF" = estimate_stable_kernel_ecf(data, seq(0.1, 1, length.out = 20)),
                 stop(paste("Unknown method:", method2)))

  est_params1 <- ensure_list_of_dicts(est1)
  est_params2 <- ensure_list_of_dicts(est2)

  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar), add = TRUE)

  hist(data, breaks = bins, freq = FALSE, col = rgb(0, 0, 0, 0.4),
       xlab = "", ylab = "", main = sprintf("True vs Estimated Mixture Density (%s vs %s)", method1, method2))
  lines(x, mixture_pdf(x, true_params), col = "green", lwd = 2, lty = 2)
  lines(x, mixture_pdf(x, est_params1), col = "blue", lwd = 2)
  lines(x, mixture_pdf(x, est_params2), col = "red", lwd = 2)
  legend("topright", legend = c("Data", "True PDF",
                                sprintf("Estimated PDF (%s)", method1),
                                sprintf("Estimated PDF (%s)", method2)),
         col = c("black", "green", "blue", "red"), lwd = c(1, 2, 2, 2),
         lty = c(NA, 2, 1, 1), pch = c(15, NA, NA, NA), pt.cex = 1.5)
  grid()
  invisible(NULL)
}
