% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mot_test_func.R
\name{MOT_test}
\alias{MOT_test}
\title{Multivariate Overall Trend Test}
\usage{
MOT_test(data, covar = NULL, width, alpha = 0.05, Nbs = 1000)
}
\arguments{
\item{data}{Numeric matrix, a two dimensional data set where each column is a
variable of the series.}

\item{covar}{Numeric vector, time order of the data series. If none given \eqn{covar = 1, 2, ..., n}.}

\item{width}{Integer, the width to be used for the \emph{moving window} algorithm.
Should be bigger than 1 and smaller than the number of observations, see \strong{details}.}

\item{alpha}{Numeric value, significance level for the test. Must be between 0 and 1, default is 0.05}

\item{Nbs}{Positive Integer, number of samples to be used for bootstrapping. Default is 1000.}
}
\value{
A named list:
\itemize{
\item \code{statistic}, computed MOT test statistic.
\item \code{p.value}, computed \emph{p-value} computed using bootstrap.
\item \code{result}, the result of the test. If \code{TRUE} the trend is considered significant.
}
}
\description{
Computes the test statistic for the multivariate overall trend
(MOT) test and the p-value.
}
\details{
The multivariate overall trend (MOT) test allows to check for trend in
the marginals and dependence structure of a multivariate distribution.
The moving window technique has been employed in order to take into account the dependence evolution,
represented by Kendall's \eqn{\tau}, according to time.

The test evaluates a null hypothesis of no trend in the data series,
against an alternative hypothesis of a monotonic trend.
Goutali and Chebana (2024) propose the following test statistic:

Let \eqn{X = (x_1, x_2, ..., x_n)} and \eqn{Y = (y_1, y_2, ..., y_n)} be variables in a data series,
and \eqn{T} the time order, the test statistic is given by:

\deqn{T_{MOT}=\frac{1}{3}\left(\tau_n(X,T)^2 + \tau_n(Y,T)^2 + \tau_n(\tau_{nw}(X,Y),T')^2  \right)}

where \eqn{\tau_n} is the empirical version of bivariate Kendall's \eqn{\tau}. The
series of the Kendall's \eqn{\tau} obtained through moving window is \eqn{\tau_{nw}(X, Y)},
and \eqn{T'} is the time order of the moving window series.

The choice of \code{width} is a trade-off. A small \eqn{w} increases the number of
rolling windows for reliable analysis, while a large \eqn{w} is necessary to have
sufficient values to identify the dependence structure. The p-value is computed
using a bootstrap procedure.
}
\examples{
\donttest{
# CASE 1: Only trend in the dependence structure
# Sample data:
DependenceStructure <- generate_data("dependenceStructure", n = 50)

width <- 10

# Perform the mot test:
mot <- MOT_test(DependenceStructure, covar = NULL, width, alpha = 0.05, Nbs = 1000)
print(mot)


# CASE 2: Only trend in the marginal distributions
# Sample data:
MarginalTrend <- generate_data("marginalTrend", n = 50)

# Perform the mot test:
mot <- MOT_test(MarginalTrend, width = width)
print(mot)


# CASE 3: No trend
# Sample data:
NoTrend <- generate_data("noTrend", n = 50)

# Perform the mot test:
mot <- MOT_test(NoTrend, width = width)
print(mot)
}


}
\references{
Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
}
\seealso{
\itemize{
\item \code{kendall.tau}: Function from the package \code{VGAM} used for computing the bivariate Kendall's \eqn{\tau}.
\item \code{rollapply}: Function from the package \code{zoo} used to apply the moving window technique.
\item \code{samp.bootstrap}: Function from the \code{resample} package, used to generate the samples
necessary to perform bootstrapping.
}
}
