# InvExpPower.R
#' Inverse Exponential Power Distribution 
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Inverse Exponential Power distribution.
#'
#' The Inverse Exponential Power distribution is parameterized by the parameters
#' \eqn{\alpha > 0} and \eqn{\lambda > 0}.
#'
#' @details
#' The Inverse Exponential Power distribution has CDF:
#'
#' \deqn{
#' F(x; \alpha, \lambda) =
#' \quad \exp \left\{1-\exp \left(\frac{\lambda}{x}\right)^\alpha\right\} 
#' \, ; \quad x > 0.
#' }
#'
#' where \eqn{\alpha} and \eqn{\lambda} are the parameters.
#'
#' The implementation includes the following functions:
#' \itemize{
#'   \item \code{dinv.exp.power()} — Density function
#'   \item \code{pinv.exp.power()} — Distribution function
#'   \item \code{qinv.exp.power()} — Quantile function
#'   \item \code{rinv.exp.power()} — Random generation
#'   \item \code{hinv.exp.power()} — Hazard function
#' }
#'
#' @usage
#' dinv.exp.power(x, alpha, lambda, log = FALSE)
#' pinv.exp.power(q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' qinv.exp.power(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
#' rinv.exp.power(n, alpha, lambda)
#' hinv.exp.power(x, alpha, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dinv.exp.power}: numeric vector of (log-)densities
#'   \item \code{pinv.exp.power}: numeric vector of probabilities
#'   \item \code{qinv.exp.power}: numeric vector of quantiles
#'   \item \code{rinv.exp.power}: numeric vector of random variates
#'   \item \code{hinv.exp.power}: numeric vector of hazard values
#' }
#'
#' @references   
#' Chaudhary, A.K., Sapkota,L.P. & Kumar, V.(2023). 
#' Inverse Exponential Power distribution: Theory and Applications. 
#' \emph{International Journal of Mathematics, Statistics and Operations 
#' Research}, \bold{3(1)}, 175--185. 
#'  
#' @examples
#' x <- seq(1.0, 5.0, 0.2)
#' dinv.exp.power(x, 2.5, 0.5)
#' pinv.exp.power(x, 2.5, 0.5)
#' qinv.exp.power(0.5, 2.5, 0.5)
#' rinv.exp.power(10, 2.5, 0.5)
#' hinv.exp.power(x, 2.5, 0.5)
#' 
#' # Data
#' x <- relief
#' # ML estimates
#' params = list(alpha=2.8286, lambda=1.3346)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pinv.exp.power, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qinv.exp.power, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' out <- gofic(x, params = params,
#'              dfun = dinv.exp.power, pfun=pinv.exp.power, plot=FALSE)
#' print.gofic(out)
#' 
#' @name InvExpPower
#' @aliases dinv.exp.power pinv.exp.power qinv.exp.power rinv.exp.power hinv.exp.power
#' @export
NULL

#' @rdname InvExpPower
#' @usage NULL
#' @export   
dinv.exp.power <- function (x, alpha, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {
		u <- exp((lambda / x[valid]) ^ alpha)
		pdf[valid] <- alpha * (lambda ^ alpha) * (x[valid] ^ -(alpha + 1.0))* u * exp(1.0 - u)
    }
	if(log)
        pdf <- log(pdf)
    return(pdf)
}

#' @rdname InvExpPower
#' @usage NULL
#' @export
pinv.exp.power <- function(q, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, lambda) <= 0)) {
		stop("Parameters 'alpha' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {
		u <- exp((lambda / q[valid]) ^ alpha)
		cdf[valid] <- exp(1.0 - u)   
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)   
}

#' @rdname InvExpPower
#' @usage NULL
#' @export
qinv.exp.power <- function(p, alpha, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {
	qtl[valid] <- lambda *((log(1.0 - log(p[valid]))) ^ -(1.0/alpha))   
	}
    return(qtl)  
}

#' @rdname InvExpPower
#' @usage NULL
#' @export
rinv.exp.power <- function(n, alpha, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, lambda) <= 0))
		stop("Invalid parameters: alpha, and lambda must be positive.")

    u <- runif(n)
    q <- qinv.exp.power(u, alpha, lambda)
    return(q)
}

#' @rdname InvExpPower
#' @usage NULL
#' @export
hinv.exp.power <- function (x, alpha, lambda)
{
    if (!all(sapply(list(x, alpha, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")
      
    nume <- dinv.exp.power(x, alpha, lambda) 
    surv <- 1.0 - pinv.exp.power(x, alpha, lambda)
    hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)  
} 
