#' @title Internal: direction-aware min–max membership
#' @description
#' Converts a numeric vector to 0–1 membership scores using min–max scaling.
#' Optionally winsorizes extremes for robustness and flips the scale if
#' \code{larger_better = FALSE}.
#' @param v Numeric vector.
#' @param larger_better Logical; if FALSE, higher values are worse.
#' @param robust Logical; if TRUE, winsorize using \code{probs}.
#' @param probs Numeric length-2 vector of quantiles for winsorization (in \code{[0,1]}).
#' @keywords internal
#' @noRd
.membership_minmax <- function(v,
                               larger_better = TRUE,
                               robust = FALSE,
                               probs = c(0.01, 0.99)) {
  # coerce and sanitize
  x <- as.numeric(v)
  x[!is.finite(x)] <- NA_real_

  # all NA -> all NA
  if (all(is.na(x))) {
    return(rep(NA_real_, length(x)))
  }

  # validate probs when robust
  if (isTRUE(robust)) {
    if (length(probs) != 2L || anyNA(probs) ||
        any(probs < 0 | probs > 1) || probs[1] >= probs[2]) {
      stop("`probs` must be length-2, 0 <= probs[1] < probs[2] <= 1.", call. = FALSE)
    }
    q <- stats::quantile(x, probs = probs, na.rm = TRUE, names = FALSE)
    # winsorize
    x <- pmin(pmax(x, q[1]), q[2])
  }

  rng <- range(x, na.rm = TRUE)
  d   <- diff(rng)

  if (!is.finite(d) || d == 0) {
    m <- rep(0.5, length(x))
  } else {
    m <- (x - rng[1]) / d
  }

  if (!isTRUE(larger_better)) m <- 1 - m

  # restore NA positions from original input
  m[is.na(v)] <- NA_real_
  m
}
