#' @title Membership Function Value for Stress Tolerance (MFVST)
#' @description
#' This function computes membership function scores (0..1) for each available index column using
#' min–max scaling with direction awareness, then aggregates them into a simple average MFVST.
#' For more details see Vinu et al. (2025) <doi:10.1007/s12355-025-01595-1>.
#'
#' @param df A data frame containing the stress indices (e.g., from \code{all_indices()$all}).
#' @param gen_col Name of genotype column; if present, it is included in the output.
#' @param lower_better Character vector listing indices where a lower value is preferred.
#' @param higher_better Character vector listing indices where a higher value is preferred.
#' @param weights Optional named numeric vector of weights for indices.
#' @param robust Logical; if \code{TRUE}, winsorizes by \code{probs}.
#' @param probs Two-element numeric vector of quantiles for robust winsorization.
#'
#' @return A list with \code{$MFVST_indexwise}: a data frame that contains per-index
#'   membership columns (suffix \code{"_M"}) and the average MFVST, that is, \code{Mean_MFVST}.
#' @references Vinu, V., Lakshmi Pathy, T., Mahadeva Swamy, H.K. et al. (2025). <doi:10.1007/s12355-025-01595-1>.
#' @importFrom stats setNames
#' @examples
#' df <- all_indices(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' df1 <- as.data.frame(df$all)
#' mfvst <- mfvst_from_indices(df1)
#' print(mfvst)
#' @export
mfvst_from_indices <- function(df,
                              gen_col = "Gen",
                              lower_better  = c("TOL","SSPI","RSI","PYR"),
                              higher_better = c("STI","YI","YSI","MP","GMP","HM","MRP"),
                              weights = NULL,
                              robust = FALSE,
                              probs = c(0.01, 0.99)) {
  stopifnot(is.data.frame(df))

  idx_names <- intersect(c(lower_better, higher_better), names(df))
  if (length(idx_names) == 0L) stop("No index columns found in df.")

  larger_better <- setNames(rep(NA, length(idx_names)), idx_names)
  larger_better[idx_names %in% higher_better] <- TRUE
  larger_better[idx_names %in% lower_better]  <- FALSE
  if (any(is.na(larger_better))) {
    missing <- paste(idx_names[is.na(larger_better)], collapse = ", ")
    stop("Some index columns have no direction assigned: ", missing)
  }

  Mlist <- lapply(idx_names, function(nm)
    .membership_minmax(df[[nm]], larger_better = larger_better[[nm]],
                       robust = robust, probs = probs))
  names(Mlist) <- paste0(idx_names, "_M")
  Mdf <- as.data.frame(Mlist, check.names = FALSE)

  if (is.null(weights)) {
    w <- rep(1/length(idx_names), length(idx_names))
    names(w) <- idx_names
  } else {
    w <- rep(0, length(idx_names)); names(w) <- idx_names
    w[names(weights)] <- as.numeric(weights)
    if (sum(w, na.rm = TRUE) <= 0) stop("All weights are zero or missing.")
    w <- w / sum(w, na.rm = TRUE)
  }

  W <- matrix(rep(w, each = nrow(Mdf)), nrow = nrow(Mdf))
  M <- as.matrix(Mdf)
  num <- rowSums(M * W, na.rm = TRUE)
  den <- rowSums((!is.na(M)) * W, na.rm = TRUE)
  MFVST <- ifelse(den > 0, num / den, NA_real_)

  out_df <- if (gen_col %in% names(df)) {
    data.frame(Gen = df[[gen_col]], Mdf, Mean_MFVST = MFVST,
               check.names = FALSE, row.names = NULL)
  } else {
    data.frame(Mdf, Mean_MFVST = MFVST, check.names = FALSE, row.names = NULL)
  }

  list(MFVST_indexwise = out_df)
}
